#pragma once

#include "websocket_rpc/stream.h"
#include <memory>
#include <ymod_webserver/websocket.h>
#include "web/settings.h"
#include <processor/interface.h>
#include "base_web_subscriber.h"

namespace yxiva { namespace web {

typedef ymod_webserver::websocket::message ws_message_t;
typedef ymod_webserver::websocket::output_stream_ptr websocket_stream_ptr;

template <typename Message>
inline json_value message_to_json(const Message& message)
{
    return message.to_json();
}

template <>
inline json_value message_to_json(const string& message)
{
    json_value data;
    if (auto error = data.parse(message))
    {
        YLOG_G(error) << "failed to parse message content: " << *error;
        return json_value{ message.c_str() };
    }
    return data;
}

class websocket_rpc_subscriber : public base_web_subscriber
{
    typedef std::shared_ptr<websocket_rpc::stream> stream_ptr;
    typedef std::weak_ptr<websocket_rpc::stream> stream_weak_ptr;

public:
    websocket_rpc_subscriber(
        task_context_ptr context,
        stream_ptr stream,
        formatters::formatter_ptr formatter,
        std::time_t sign_expiration,
        const string& client_id);

    ~websocket_rpc_subscriber();

    void init();

    void ping(const ping_message& message) override;

    void on_message(const json_value& value);

    void notify_error(const error_message&) override;

    void close() override;

    void notify_position(const position_message&) override;

    void notify_disconnected(const disconnected_message&) override;

    boost::weak_ptr<websocket_rpc_subscriber> weak_from_this()
    {
        return boost::dynamic_pointer_cast<websocket_rpc_subscriber>(shared_from_this());
    }

protected:
    void impl_notify_text(const string& message) override;
    void impl_notify_binary(const string& header, const string& message) override;
    const string& name() const override;

    template <typename Message>
    void impl_notify(const Message& message)
    {
        if (auto stream_copy = stream())
        {
            json_value params;
            params["data"] = message_to_json(message);
            params["subscription_token"] = id();
            stream_copy->push_notification(params);
        }
    }

private:
    stream_ptr stream();
    void reset_stream();

    mutex mux_;
    stream_ptr stream_;
};

}}
