#include "websocket_subscriber.h"
#include "mod_log/find.h"
#include "parse_push_ack.h"
#include "api2/watch_subscribers.h" // TODO refactor hierarchy
#include "impl.h"
#include "processor/interface.h"

#include <yxiva/core/split.h>
#include <yplatform/context_repository.h>
#include <yplatform/find.h>
#include <boost/bind/protect.hpp>
#include <boost/pointer_cast.hpp>

namespace yxiva { namespace web {

static const string NAME = string("ss_event");

namespace {
template <typename F, typename Arg1>
void on_message_weak(F const& f, boost::weak_ptr<websocket_subscriber> wptr, Arg1 arg1)
{
    if (auto ptr = wptr.lock())
    {
        f(ptr, arg1);
    }
}
}

websocket_subscriber::websocket_subscriber(
    task_context_ptr context,
    websocket_stream_ptr output_stream,
    formatters::formatter_ptr formatter,
    std::time_t sign_expiration,
    const time_duration& inactive_timeout,
    const string& client_id)
    : base_web_subscriber(context, formatter, sign_expiration, client_id)
    , stream_(output_stream)
    , inactive_timeout_(inactive_timeout)
{
}

websocket_subscriber::~websocket_subscriber()
{
    reset_stream();
}

void websocket_subscriber::init()
{
    auto member_f = boost::protect(boost::bind(&websocket_subscriber::on_ws_message, _1, _2));
    auto callback = boost::bind(
        on_message_weak<decltype(member_f), ws_message_t>, member_f, weak_from_this(), _1);
    stream_->add_message_callback(callback);
    stream_->begin_receive(inactive_timeout_);
    context_repo.rem_context(ctx());
}

void websocket_subscriber::impl_notify_text(const string& message)
{
    stream_ptr stream_copy = stream();
    if (!stream_copy) return;
    stream_copy->send_text(message);
}

void websocket_subscriber::impl_notify_binary(const string& header, const string& message)
{
    stream_ptr stream_copy = stream();
    if (!stream_copy) return;
    auto bin_stream = stream_copy->bin_stream(header.size() + message.size());
    bin_stream << header << message;
}

void websocket_subscriber::ping(const ping_message& message)
{
    stream_ptr stream_copy = stream();
    if (!stream_copy) return;
    stream_copy->send_text(message.to_string());
}

void websocket_subscriber::notify_error(const error_message& message)
{
    stream_ptr stream_copy = stream();
    if (!stream_copy) return;
    stream_copy->send_text(message.to_string());
}

void websocket_subscriber::close()
{
    reset_stream();
}

void websocket_subscriber::notify_position(const position_message& message)
{
    if (auto stream_copy = stream())
    {
        stream_copy->send_text(message.to_string());
    }
}

const string& websocket_subscriber::name() const
{
    return NAME;
}

void websocket_subscriber::on_ws_message(ws_message_t const& msg)
{
    switch (msg.opcode)
    {
    case ws_message_t::opcode_text:
    {
        string data(msg.data.begin(), msg.data.end());
        if (msg.is_finished())
        {
            perform_message(data);
        }
        else
        {
            input_buffer = data;
        }
    }
    break;
    case ws_message_t::opcode_continuation:
        input_buffer.append(msg.data.begin(), msg.data.end());
        if (msg.is_finished())
        {
            perform_message(input_buffer);
            input_buffer.clear();
        }
        break;
    default:
        break;
    }
}

void websocket_subscriber::perform_message(string const& text)
{
    json_value value;
    if (auto error = value.parse(text))
    {
        YLOG_CTX_GLOBAL(ctx(), info) << "message from client parse error: " << *error
                                     << ", raw_message=\"" << text.substr(0, 50) << "\"";
        return;
    }

    push_ack ack;
    if (auto parsed = parse_push_ack(value, ack))
    {
    }
    else
    {
        YLOG_CTX_GLOBAL(ctx(), info) << parsed.error_reason;
        return;
    }
    find_xivaws_log()->log_notification_ack(
        ctx(), ack.send_context, ack.transit_id, ack.client, ack.service);
}

websocket_subscriber::stream_ptr websocket_subscriber::stream()
{
    scoped_lock lock(mux_);
    return stream_;
}

void websocket_subscriber::reset_stream()
{
    scoped_lock lock(mux_);
    if (stream_)
    {
        stream_->close_connection();
        stream_.reset();
    }
}

}}
