#include "app_config_transformer.h"

#include <yxiva/core/platforms.h>

#include <yplatform/encoding/base64.h>
#include <yplatform/util/sstream.h>

#include <algorithm>
#include <stdexcept>
#include <utility>

namespace yxiva { namespace web { namespace webui {

namespace {

inline void empty_or_has_two_non_empty_lines(const string& str, const char* msg)
{
    auto newline_pos = str.find('\n');
    if (str.size() &&
        (newline_pos == 0 ||                               // 1st line empty
         newline_pos == str.size() - 1 ||                  // 2nd line empty
         newline_pos == string::npos ||                    // \n missed
         str.find('\n', newline_pos + 1) != string::npos)) // more than 1 \n
    {
        throw std::runtime_error(msg);
    }
}

inline void assign_if_not_empty(string& target, const string& src)
{
    if (src.size())
    {
        target = src;
    }
}

}

app_config_transformer::app_config_transformer(ymod_webserver::request_ptr req)
    : request_(parse_request(req))
{
    static const fields_type required_fields = { { "app_name", json_type::tstring },
                                                 { "platform", json_type::tstring },
                                                 { "service", json_type::tstring },
                                                 { "environment", json_type::tstring } };
    check_required_fields_throws(request_, required_fields);

    platform_specific_impl_ = platform_specific::make(platform(), request_);
}

json_value app_config_transformer::parse_request(ymod_webserver::request_ptr req)
{
    auto str = string(req->raw_body.begin(), req->raw_body.end());
    json_value request;
    if (auto error = request.parse(str))
    {
        throw std::runtime_error(*error);
    }
    return request;
}

string app_config_transformer::app_name() const
{
    return request_["app_name"].to_string();
}

string app_config_transformer::platform() const
{
    return request_["platform"].to_string();
}

string app_config_transformer::service() const
{
    return request_["service"].to_string();
}

string app_config_transformer::environment() const
{
    return request_["environment"].to_string();
}

application_config app_config_transformer::revoke(const application_config& app_config) const
{
    auto res_config = app_config;
    res_config.platform = platform::hacks::fcm_to_gcm(app_config.platform);

    platform_specific_impl_->backup_secret(res_config);
    if (res_config.secret_key.size())
    {
        res_config.secret_key.clear();
        res_config.expiration_time = 0;
    }

    validate(res_config);

    return res_config;
}

application_config app_config_transformer::apply(const application_config& app_config) const
{
    auto res_config = app_config;
    res_config.platform = platform::hacks::fcm_to_gcm(app_config.platform);
    if (res_config.app_name.empty())
    {
        res_config.app_name = app_name();
    }
    if (res_config.platform.empty())
    {
        res_config.platform = platform::hacks::fcm_to_gcm(platform());
    }
    res_config.xiva_service = service();
    auto [res, env] = resolve_app_environment(environment());
    if (!res)
    {
        throw std::runtime_error(res.error_reason);
    }
    res_config.environment = env;
    res_config.ttl = 0;

    if (platform_specific_impl_->secret_need_update())
    {
        platform_specific_impl_->backup_secret(res_config);
        platform_specific_impl_->apply_secret(res_config);
    }

    validate(res_config);

    return res_config;
}

application_config app_config_transformer::revert(const application_config& app_config) const
{
    auto res_config = app_config;
    res_config.platform = platform::hacks::fcm_to_gcm(app_config.platform);

    if (res_config.key_backup.empty())
    {
        throw std::runtime_error("no backup record");
    }

    platform_specific_impl_->revert(res_config);

    validate(res_config);

    return res_config;
}

void app_config_transformer::validate(const application_config& conf) const
{
    if (conf.xiva_service.empty())
    {
        throw std::runtime_error("lost xiva_service");
    }
    if (conf.platform.empty())
    {
        throw std::runtime_error("lost platform");
    }
    if (conf.app_name.empty())
    {
        throw std::runtime_error("lost app_name");
    }

    platform_specific_impl_->validate(conf);
}

void app_config_transformer::platform_specific::backup_secret(application_config& conf) const
{
    if (conf.secret_key.size())
    {
        conf.key_backup = conf.secret_key;
    }
}

std::unique_ptr<app_config_transformer::platform_specific> app_config_transformer::
    platform_specific::make(const string& platform, const json_value& request)
{
    if (platform == platform::FCM)
    {
        return std::make_unique<fcm>(request);
    }
    else if (platform == platform::APNS)
    {
        return std::make_unique<apns>(request);
    }
    else if (platform == platform::WNS)
    {
        return std::make_unique<wns>(request);
    }
    else if (platform == platform::HMS)
    {
        return std::make_unique<hms>(request);
    }
    else
    {
        throw std::runtime_error("unsupported platform");
    }
}

app_config_transformer::fcm::fcm(const json_value& req)
{
    static const fields_type platform_fields = { { "apikey", json_type::tstring } };
    check_optional_fields_throws(req, platform_fields);

    apikey_ = json_get<string>(req, "apikey", "");
}

void app_config_transformer::fcm::apply_secret(application_config& conf) const
{
    auto new_credential = yxiva::fcm::parse_secret_key(conf.secret_key);
    assign_if_not_empty(new_credential.api_key, apikey());
    conf.secret_key = new_credential.to_secret_key();
}

void app_config_transformer::fcm::revert(application_config& conf) const
{
    std::swap(conf.key_backup, conf.secret_key);
}

void app_config_transformer::fcm::validate(const application_config& /*conf*/) const
{
}

bool app_config_transformer::fcm::secret_need_update() const
{
    return apikey().size();
}

bool validate_p8_secret(const string& secret)
{
    try
    {
        p8_token token{ secret };
        resolve_certificate_type(token.type);
        return token.key.size() && token.key_id.size() && token.issuer_key.size() &&
            token.topic.size();
    }
    catch (...)
    {
        return false;
    }
}

app_config_transformer::apns::apns(const json_value& req)
{
    static const fields_type pem_p12_fields = { { "cert", json_type::tstring },
                                                { "cert-pass", json_type::tstring } };
    check_optional_fields_throws(req, pem_p12_fields);

    auto base64_cert = json_get<string>(req, "cert", "");
    cert_ += yplatform::base64_decode(base64_cert);
    cert_pass_ = json_get<string>(req, "cert-pass", "");

    p8_fields_ = p8_token(req);

    if (has_pem_p12_fields() && has_p8_fields())
    {
        throw std::runtime_error("invalid combination of apns fields");
    }
}

void app_config_transformer::apns::apply_secret(application_config& conf) const
{
    if (has_pem_p12_fields())
    {
        fill_pem_p12_config(conf, cert(), cert_pass());
    }
    else if (has_p8_fields())
    {
        p8_token token(conf.secret_key);

        assign_if_not_empty(token.key, p8_fields().key);
        assign_if_not_empty(token.key_id, p8_fields().key_id);
        assign_if_not_empty(token.issuer_key, p8_fields().issuer_key);
        assign_if_not_empty(token.topic, p8_fields().topic);
        assign_if_not_empty(token.type, p8_fields().type);

        conf.secret_key = token.to_string();
        conf.expiration_time = 0;
    }
    else
    {
        throw std::logic_error("impossible state");
    }
}

void app_config_transformer::apns::revert(application_config& conf) const
{
    std::swap(conf.key_backup, conf.secret_key);
    if (validate_p8_secret(conf.secret_key))
    {
        conf.expiration_time = 0;
    }
    else
    {
        fill_pem_p12_config(conf, conf.secret_key, {});
    }
}

bool app_config_transformer::apns::has_pem_p12_fields() const
{
    return cert().size() || cert_pass().size();
}

bool app_config_transformer::apns::has_p8_fields() const
{
    return p8_fields_.key.size() || p8_fields_.key_id.size() || p8_fields_.issuer_key.size() ||
        p8_fields_.topic.size() || p8_fields_.type.size();
}

void app_config_transformer::apns::fill_pem_p12_config(
    application_config& conf,
    string cert,
    string passw) const
{
    if (pem_certificate::contains_pem(cert))
    {
        conf.secret_key = std::move(cert);
    }
    else
    {
        conf.secret_key = pem_certificate::from_p12(cert, passw);
    }
    conf.expiration_time = pem_certificate::get_expiration_time(conf.secret_key);
}

void app_config_transformer::apns::validate(const application_config& conf) const
{
    const auto& key = conf.secret_key;
    const auto& exp = conf.expiration_time;

    if (key.empty())
    {
        if (exp != 0)
        {
            throw std::runtime_error("nonzero apns expiration time");
        }
        return;
    }

    if (validate_p8_secret(key))
    {
        return;
    }

    if (exp == 0)
    {
        throw std::runtime_error("zero apns expiration time");
    }

    pem_certificate::validate(key);

    auto ts = pem_certificate::get_expiration_time(key);
    if (exp != ts)
    {
        throw std::runtime_error("wrong apns expiration time");
    }

    pem_certificate::get_topic(key);
    pem_certificate::get_type(key);
}

bool app_config_transformer::apns::secret_need_update() const
{
    if (has_pem_p12_fields())
    {
        return cert().size();
    }
    else if (has_p8_fields())
    {
        return true;
    }
    return false;
}

void app_config_transformer::apns::backup_secret(application_config& conf) const
{
    if (conf.secret_key.size())
    {
        // Temporary additional condition for migration period
        if (pem_certificate::contains_pem(conf.secret_key) || validate_p8_secret(conf.key_backup))
        {
            conf.key_backup = conf.secret_key;
        }
    }
}

app_config_transformer::wns::wns(const json_value& req)
{
    static const fields_type platform_fields = { { "secret", json_type::tstring },
                                                 { "sid", json_type::tstring } };
    check_optional_fields_throws(req, platform_fields);

    secret_ = json_get<string>(req, "secret", "");
    sid_ = json_get<string>(req, "sid", "");
}

void app_config_transformer::wns::apply_secret(application_config& conf) const
{
    auto new_credential = yxiva::wns::parse_secret_key(conf.secret_key);
    assign_if_not_empty(new_credential.sid, sid());
    assign_if_not_empty(new_credential.secret, secret());
    conf.secret_key = new_credential.to_secret_key();
}

void app_config_transformer::wns::revert(application_config& conf) const
{
    std::swap(conf.key_backup, conf.secret_key);
}

void app_config_transformer::wns::validate(const application_config& conf) const
{
    empty_or_has_two_non_empty_lines(conf.secret_key, "wrong wns secret key");
}

bool app_config_transformer::wns::secret_need_update() const
{
    return secret().size() || sid().size();
}

app_config_transformer::hms::hms(const json_value& req)
{
    static const fields_type platform_fields = { { "secret", json_type::tstring },
                                                 { "client_id", json_type::tstring } };
    check_optional_fields_throws(req, platform_fields);

    secret_ = json_get<string>(req, "secret", "");
    client_id_ = json_get<string>(req, "client_id", "");
}

void app_config_transformer::hms::apply_secret(application_config& conf) const
{
    auto new_credential = yxiva::hms::parse_secret_key(conf.secret_key);
    assign_if_not_empty(new_credential.id, client_id());
    assign_if_not_empty(new_credential.secret, secret());
    conf.secret_key = new_credential.to_secret_key();
}

void app_config_transformer::hms::revert(application_config& conf) const
{
    std::swap(conf.key_backup, conf.secret_key);
}

void app_config_transformer::hms::validate(const application_config& conf) const
{
    empty_or_has_two_non_empty_lines(conf.secret_key, "wrong hms secret key");
}

bool app_config_transformer::hms::secret_need_update() const
{
    return secret().size() || client_id().size();
}

}}}
