#pragma once

#include "json_helpers.h"
#include "common.h"
#include "web/types.h"

#include <yxiva/core/conf.h>
#include <yxiva/core/json.h>
#include <yxiva/core/x509.h>

#include <memory>

namespace yxiva { namespace web { namespace webui {

class app_config_transformer
{
public:
    explicit app_config_transformer(ymod_webserver::request_ptr req);

    string app_name() const;
    string platform() const;

    application_config apply(const application_config& app_config) const;
    application_config revoke(const application_config& app_config) const;
    application_config revert(const application_config& app_config) const;

private:
    class platform_specific
    {
    public:
        virtual ~platform_specific()
        {
        }

        virtual void apply_secret(application_config& conf) const = 0;
        virtual void revert(application_config& conf) const = 0;
        virtual void validate(const application_config& conf) const = 0;
        virtual bool secret_need_update() const = 0;
        virtual void backup_secret(application_config& conf) const;

        static std::unique_ptr<platform_specific> make(
            const string& platform,
            const json_value& request);
    };

    class fcm : public platform_specific
    {
    public:
        explicit fcm(const json_value& req);

        void apply_secret(application_config& conf) const override;
        void revert(application_config& conf) const override;
        void validate(const application_config& conf) const override;
        bool secret_need_update() const override;

        const string& apikey() const
        {
            return apikey_;
        }

    private:
        string apikey_;
    };

    class apns : public platform_specific
    {
    public:
        explicit apns(const json_value& req);

        void apply_secret(application_config& conf) const override;
        void revert(application_config& conf) const override;
        void validate(const application_config& conf) const override;
        bool secret_need_update() const override;
        void backup_secret(application_config& conf) const override;

        const string& cert() const
        {
            return cert_;
        }
        const string& cert_pass() const
        {
            return cert_pass_;
        }

        const p8_token& p8_fields() const
        {
            return p8_fields_;
        }

    private:
        string cert_;
        string cert_pass_;
        p8_token p8_fields_;

        bool has_pem_p12_fields() const;
        bool has_p8_fields() const;
        void fill_pem_p12_config(application_config& conf, string cert, string passw) const;
    };

    class wns : public platform_specific
    {
    public:
        explicit wns(const json_value& req);

        void apply_secret(application_config& conf) const override;
        void revert(application_config& conf) const override;
        void validate(const application_config& conf) const override;
        bool secret_need_update() const override;

        const string& secret() const
        {
            return secret_;
        }
        const string& sid() const
        {
            return sid_;
        }

    private:
        string secret_;
        string sid_;
    };

    class hms : public platform_specific
    {
    public:
        explicit hms(const json_value& req);

        void apply_secret(application_config& conf) const override;
        void revert(application_config& conf) const override;
        void validate(const application_config& conf) const override;
        bool secret_need_update() const override;

        const string& secret() const
        {
            return secret_;
        }
        const string& client_id() const
        {
            return client_id_;
        }

    private:
        string secret_;
        string client_id_;
    };

    json_value request_;
    std::unique_ptr<platform_specific> platform_specific_impl_;

    static json_value parse_request(ymod_webserver::request_ptr req);
    void validate(const application_config& conf) const;

    string environment() const;
    string service() const;
};

}}}
