#pragma once

#include "owner_type.h"
#include "common.h"
#include "web/auth/user.h"
#include "web/find_deps.h"

namespace yxiva { namespace web { namespace webui {

inline bool is_admin(const settings_ptr& settings, const auth_info& auth)
{
    for (auto& user : auth.users)
    {
        if (settings->api.webui.admins.count(static_cast<string>(user.uid)))
        {
            return true;
        }
    }
    return false;
}

inline bool check_user(const auth_info& auth, const string& uid)
{
    return std::any_of(auth.users.begin(), auth.users.end(), [&uid](const user_info& ui) {
        return ui.uid == static_cast<user_id>(uid);
    });
}

template <typename Handler, typename... Args>
auto webui_auth_impl(
    http_stream_ptr stream,
    settings_ptr settings,
    const auth_info& auth,
    const std::vector<string>& abc_services,
    bool admin_only,
    Handler&& handler,
    const string& owner_type_name,
    const string& optional_owner_id,
    Args&&... args)
{
    auto owner_type = owner_type_from_name(owner_type_name);
    if (owner_type == owner_type::unknown)
    {
        send_bad_request(stream, "unknown owner type");
        return;
    }
    // Default owner_id only makes sense for user auth.
    const string& owner_id = optional_owner_id.empty() && owner_type == owner_type::user ?
        auth.users.front().uid :
        optional_owner_id;
    if (owner_id.empty())
    {
        send_json_error(stream, http_codes::unauthorized, "empty owner");
        return;
    }
    bool admin = is_admin(settings, auth);
    if (admin_only && !admin)
    {
        send_json_error(stream, http_codes::unauthorized, "must be admin");
        return;
    }
    if (!admin)
    {
        if (owner_type == owner_type::user && !check_user(auth, owner_id))
        {
            send_json_error(stream, http_codes::unauthorized, "requested uid is not authorized");
            return;
        }
        else if (
            owner_type == owner_type::abc &&
            !std::count(abc_services.begin(), abc_services.end(), owner_id))
        {
            send_json_error(stream, http_codes::unauthorized, "not a member of provided service");
            return;
        }
    }
    auto service_manager = find_service_manager(settings->api.webui.service_manager);
    if (!service_manager)
    {
        send_json_error(stream, http_codes::internal_server_error, "not configured");
        return;
    }

    handler(
        stream,
        settings,
        admin,
        prefix_by_owner_type(settings, owner_type),
        owner_id,
        service_manager,
        std::forward<Args>(args)...);
}

template <typename Handler>
auto webui_auth_admin_only(Handler&& handler)
{
    return [handler = std::forward<Handler>(handler)](
               http_stream_ptr stream,
               settings_ptr settings,
               const auth_info& auth,
               const std::vector<string>& abc_services,
               auto&&... args) {
        webui_auth_impl(
            stream,
            settings,
            auth,
            abc_services,
            true,
            handler,
            std::forward<decltype(args)>(args)...);
    };
}

template <typename Handler>
auto webui_auth(Handler&& handler)
{
    return [handler = std::forward<Handler>(handler)](
               http_stream_ptr stream,
               settings_ptr settings,
               const auth_info& auth,
               const std::vector<string>& abc_services,
               auto&&... args) {
        webui_auth_impl(
            stream,
            settings,
            auth,
            abc_services,
            false,
            handler,
            std::forward<decltype(args)>(args)...);
    };
}

}}}
