#include "json_app_info.h"
#include <yxiva/core/platforms.h>

namespace yxiva::web::webui {

auto make_app_info_common(const application_config& app)
{
    json_value app_info;
    app_info["updated_at"] = app.updated_at;
    app_info["environment"] = get_environment_name(app.environment);
    return app_info;
}

void write_secret_apns_pem(json_value_ref&& secret_json, const string& secret_key)
{
    using namespace apns::pem_certificate;
    secret_json["Not After"] = get_expiration_time(secret_key);
    secret_json["UID (topic)"] = get_topic(secret_key);
    secret_json["Type"] = get_certificate_type_name(get_type(secret_key));
}

void write_secret_apns_p8(json_value_ref&& secret_json, const string& secret_key)
{
    apns::p8_token token{ secret_key };
    secret_json["Key"] = token.key;
    secret_json["KeyID"] = token.key_id;
    secret_json["Issuer"] = token.issuer_key;
    secret_json["Topic"] = token.topic;
    secret_json["Type"] = token.type;
}

void write_secret_apns(json_value_ref&& secret_json, const string& secret_key)
{
    if (secret_key.empty()) return;
    if (apns::pem_certificate::contains_pem(secret_key))
    {
        write_secret_apns_pem(std::move(secret_json), secret_key);
    }
    else
    {
        write_secret_apns_p8(std::move(secret_json), secret_key);
    }
}

void write_secret_fcm(json_value_ref&& secret_json, const string& secret_key)
{
    if (secret_key.empty()) return;
    auto credential = yxiva::fcm::parse_secret_key(secret_key);
    secret_json["API key"] = credential.api_key;
}

void write_secret_hms(json_value_ref&& secret_json, const string& secret_key)
{
    if (secret_key.empty()) return;
    auto credential = yxiva::hms::parse_secret_key(secret_key);
    secret_json["ID"] = credential.id;
    secret_json["Secret"] = credential.secret;
}

void write_secret_wns(json_value_ref&& secret_json, const string& secret_key)
{
    if (secret_key.empty()) return;
    auto credential = yxiva::wns::parse_secret_key(secret_key);
    secret_json["ID"] = credential.sid;
    secret_json["Secret"] = credential.secret;
}

template <typename Writer>
auto make_app_info(const application_config& app, Writer writer)
{
    auto app_info = make_app_info_common(app);
    writer(app_info["current"], app.secret_key);
    writer(app_info["backup"], app.key_backup);
    return app_info;
}

json_value make_app_info(const application_config& app)
{
    static const std::map<string, std::function<void(json_value_ref&&, const string&)>> writers{
        { platform::APNS, write_secret_apns },
        { platform::FCM, write_secret_fcm },
        { platform::HMS, write_secret_hms },
        { platform::WNS, write_secret_wns }
    };

    if (!writers.count(app.platform))
    {
        throw std::runtime_error("no formatter for platform " + app.platform);
    }

    return make_app_info(app, writers.at(app.platform));
}

}
