#include "json_helpers.h"

#include <yxiva/core/platforms.h>
#include <yplatform/encoding/base64.h>
#include <yplatform/hash/sha1.h>

namespace yxiva { namespace web { namespace webui {

operation::result check_fields(
    const json_value& data_json,
    const fields_type& fields,
    bool required)
{
    for (auto& field : fields)
    {
        bool exists = data_json.has_member(field.first);
        if (required && !exists || exists && data_json[field.first].type() != field.second)
        {
            return string("missing or invalid value of ") + field.first;
        }
        if (required && field.second == json_type::tstring &&
            data_json[field.first].to_string_view().empty())
        {
            return string("empty field ") + field.first;
        }
        if (field.second == json_type::tnumber && json_get<int>(data_json, field.first, 0) <= 0)
        {
            return string("invalid field ") + field.first;
        }
    }
    return operation::success;
}

inline operation::result check_required_fields(
    const json_value& data_json,
    const fields_type& fields)
{
    return check_fields(data_json, fields, true);
}

inline operation::result check_optional_fields(
    const json_value& data_json,
    const fields_type& fields)
{
    return check_fields(data_json, fields, false);
}

void check_optional_fields_throws(const json_value& data_json, const fields_type& fields)
{
    auto opt_check = check_optional_fields(data_json, fields);
    if (!opt_check)
    {
        throw std::runtime_error(opt_check.error_reason);
    }
}

void check_required_fields_throws(const json_value& data_json, const fields_type& fields)
{
    auto req_check = check_required_fields(data_json, fields);
    if (!req_check)
    {
        throw std::runtime_error(req_check.error_reason);
    }
}

void json_result(const http_stream_ptr& stream, http_codes::code cd, const json_value& resp)
{
    stream->set_code(cd);
    stream->set_content_type("application/json");
    stream->result_body(resp.stringify());
}

void write_app(json_value_ref&& app_json, const application_config& app)
{
    app_json["platform"] = platform::resolve_alias(app.platform).name; // gcm_compatibility
    app_json["app_name"] = app.app_name;
    app_json["expiration"] = app.expiration_time;
    app_json["can_revert"] = !app.key_backup.empty();
    app_json["environment"] = get_environment_name(app.environment);
    app_json["revoked"] = app.secret_key.empty();
    app_json["updated_at"] = app.updated_at;
    if (!app.secret_key.empty())
    {
        app_json["sha1"] = yplatform::sha1(app.secret_key);
    }
}

void write_tvm_app(json_value_ref&& app_json, const tvm_app_info& app)
{
    app_json["id"] = app.id;
    app_json["name"] = app.name;
    app_json["suspended"] = app.suspended;
}

void write_tvm_apps(json_value_ref&& apps_json, const std::set<tvm_app_info>& apps)
{
    apps_json.set_array();
    for (auto&& app : apps)
    {
        write_tvm_app(apps_json.push_back(), app);
    }
}

void write_tvm_role(
    json_value_ref&& role_json,
    const std::map<string, std::set<tvm_app_info>>& role_map)
{
    role_json.set_object();
    for (auto&& [env, tvm_apps] : role_map)
    {
        write_tvm_apps(role_json[env], tvm_apps);
    }
}

void write_service(json_value_ref&& service_json, const service_data& service)
{
    service_json["name"] = service.properties.name;
    service_json["description"] = service.properties.description;
    service_json["is_passport"] = service.properties.is_passport;
    service_json["is_stream"] = service.properties.is_stream;
    service_json["queued_delivery_by_default"] = service.properties.queued_delivery_by_default;
    service_json["stream_count"] = service.properties.stream_count;
    service_json["revoked"] = service.properties.revoked;
    service_json["owner"] = service.properties.owner();
    service_json["auth_disabled"] = service.properties.auth_disabled;

    service_json["oauth_scopes"].set_array();
    for (auto&& scope : service.properties.oauth_scopes)
    {
        service_json["oauth_scopes"].push_back(scope);
    }

    write_tvm_role(service_json["tvm_publishers"], service.properties.tvm_publishers);
    write_tvm_role(service_json["tvm_subscribers"], service.properties.tvm_subscribers);

    // Touch "send_tokens" to create key in case there's no tokens.
    service_json["send_tokens"].set_object();
    for (auto& token : service.send_tokens)
    {
        auto&& token_json = service_json["send_tokens"][token.first.env][token.first.token];
        token_json["token"] = token.first.token;
        token_json["name"] = token.second.name;
        token_json["revoked"] = token.second.revoked;
    }

    // Touch "listen_tokens" to create key in case there's no tokens.
    service_json["listen_tokens"].set_object();
    for (auto& token : service.listen_tokens)
    {
        auto&& token_json = service_json["listen_tokens"][token.first.env][token.first.token];
        token_json["token"] = token.first.token;
        token_json["name"] = token.second.name;
        token_json["client"] = token.second.client;
        token_json["revoked"] = token.second.revoked;
        // Allowed services are not displayed.
    }

    auto&& apps_json = service_json["apps"];
    apps_json = json_value{ json_type::tarray };
    for (auto& app : service.apps)
    {
        write_app(apps_json.push_back(), app.second);
    }
}

std::pair<operation::result, tvm_app_info> parse_tvm_app(const json_value& app_json)
{
    static const fields_type required_fields = { { "id", json_type::tnumber },
                                                 { "suspended", json_type::tbool } };
    if (!app_json.is_object())
    {
        return { "tvm app json is not an object", {} };
    }
    if (auto&& res = check_required_fields(app_json, required_fields); !res)
    {
        return { res, {} };
    }

    tvm_app_info app_info;
    app_info.id = app_json["id"].to_int64();
    app_info.name = app_json["name"].to_string();
    app_info.suspended = app_json["suspended"].to_bool();

    return { operation::success, app_info };
}

operation::result parse_tvm_apps(const json_value& apps_json, std::set<tvm_app_info>& apps)
{
    if (!apps_json.is_array())
    {
        return "tvm apps json is not an array";
    }

    for (auto&& app : apps_json.array_items())
    {
        auto&& [res, app_info] = parse_tvm_app(app);
        if (!res)
        {
            return res;
        }
        apps.insert(app_info);
    }

    return operation::success;
}

operation::result parse_tvm_role(
    const json_value& data_json,
    const string& role_name,
    std::map<string, std::set<tvm_app_info>>& role_map)
{
    if (!data_json[role_name].is_object())
    {
        return "tvm role json is not an object";
    }

    for (auto&& env_apps = data_json[role_name].members_begin();
         env_apps != data_json[role_name].members_end();
         ++env_apps)
    {
        auto&& env = string(env_apps.key());
        auto&& res = parse_tvm_apps(*env_apps, role_map[env]);
        if (!res)
        {
            return res;
        }
    }

    return operation::success;
}

operation::result parse_tvm_roles(const json_value& data_json, service_properties& data)
{
    if (data_json.has_member("tvm_publishers"))
    {
        auto&& res = parse_tvm_role(data_json, "tvm_publishers", data.tvm_publishers);
        if (!res)
        {
            return res;
        }
    }

    if (data_json.has_member("tvm_subscribers"))
    {
        auto&& res = parse_tvm_role(data_json, "tvm_subscribers", data.tvm_subscribers);
        if (!res)
        {
            return res;
        }
    }

    return operation::success;
}

operation::result parse_json_properties(const json_value& data_json, service_properties& data)
{
    static const fields_type required_fields = { { "name", json_type::tstring },
                                                 { "is_passport", json_type::tbool },
                                                 { "is_stream", json_type::tbool },
                                                 { "auth_disabled", json_type::tbool } };
    static const fields_type required_fields_passport = { { "oauth_scopes", json_type::tarray } };
    static const fields_type required_fields_stream = { { "stream_count", json_type::tnumber } };

    if (auto req_check = check_required_fields(data_json, required_fields); !req_check)
    {
        return req_check;
    }

    data.name = json_get<string>(data_json, "name", "");
    data.description = json_get<string>(data_json, "description", "");
    data.is_passport = json_get<bool>(data_json, "is_passport", false);
    data.is_stream = json_get<bool>(data_json, "is_stream", false);
    data.queued_delivery_by_default = json_get<bool>(data_json, "queued_delivery_by_default", true);
    data.auth_disabled = json_get<bool>(data_json, "auth_disabled", false);

    if (data.is_passport)
    {
        if (auto req_check = check_required_fields(data_json, required_fields_passport); !req_check)
        {
            return req_check;
        }

        for (auto&& scope : data_json["oauth_scopes"].array_items())
        {
            if (!scope.is_string())
            {
                return string("invalid value of ") + "oauth_scopes";
            }
            auto scope_val = scope.to_string();
            if (scope_val.empty())
            {
                return "scope value is empty";
            }
            data.oauth_scopes.insert(scope_val);
        }
    }

    if (data.is_stream)
    {
        if (auto req_check = check_required_fields(data_json, required_fields_stream))
        {
            data.stream_count = json_get<uint16_t>(data_json, "stream_count", 0);
            // Extra check (just in case).
            if (!data.stream_count)
            {
                return "invalid stream count";
            }
        }
        else
        {
            return req_check;
        }
    }

    if (auto&& res = parse_tvm_roles(data_json, data); !res)
    {
        return res;
    }

    return operation::success;
}

operation::result parse_json_token(const json_value& data_json, token_properties& data)
{
    static const fields_type required_fields = { { "name", json_type::tstring },
                                                 { "service", json_type::tstring } };

    auto req_check = check_required_fields(data_json, required_fields);
    if (!req_check)
    {
        return req_check;
    }
    data.name = json_get<string>(data_json, "name", "");
    data.service = json_get<string>(data_json, "service", "");
    return operation::success;
}

operation::result parse_json_properties(const json_value& data_json, send_token_properties& data)
{
    return parse_json_token(data_json, data);
}

operation::result parse_json_properties(const json_value& data_json, listen_token_properties& data)
{
    static const fields_type required_fields = { { "client", json_type::tstring } };

    auto req_check = check_required_fields(data_json, required_fields);
    if (!req_check)
    {
        return req_check;
    }
    auto res = parse_json_token(data_json, data);
    if (!res)
    {
        return res;
    }
    data.client = json_get<string>(data_json, "client", "");
    if (data.client.empty())
    {
        return string("empty field ") + "client";
    }
    return operation::success;
}

}}}
