#include "webui.h"
#include "app_config_transformer.h"
#include "json_app_info.h"

#include <yxiva/core/methods/check.h>
#include <yxiva/core/conf.h>
#include <algorithm>
#include <atomic>
#include <cctype>
#include <mutex>
#include <vector>

namespace yxiva { namespace web { namespace webui {
namespace {

struct list_task
{
    list_task(const http_stream_ptr& stream) : stream(stream), ctx(stream->request()->context)
    {
    }

    json_value result;
    http_stream_ptr stream;
    task_context_ptr ctx;
};

void service_list_handler(
    std::shared_ptr<list_task> task,
    const operation::result& res,
    services_type services)
{
    if (task->ctx->is_cancelled())
    {
        return;
    }
    if (res)
    {
        for (auto& service : services)
        {
            write_service(task->result["services"][service->properties.name], *service);
        }
    }
    else
    {
        task->result["error"] = res.error_reason;
    }
    json_result(task->stream, http_codes::ok, task->result);
}

bool validate_name(const string& name)
{
    return name.size() && isalpha(name[0]) &&
        std::all_of(name.begin(), name.end(), [](char c) { return c == '-' || isalnum(c); });
}

bool validate_client(const string& client)
{
    return client.size() && isalpha(client[0]) &&
        std::all_of(client.begin(), client.end(), &isalnum_ext);
}

bool validate_scopes(const std::set<string>& scopes)
{
    return std::all_of(scopes.begin(), scopes.end(), [](const string& scope) {
        return std::all_of(scope.begin(), scope.end(), [](char c) {
            return c == '!' || (c >= '#' && c <= '[') || (c >= ']' && c <= '~');
        });
    });
}

template <typename TokenType, typename TokenStore>
bool token_exists(const TokenStore& store, const string& env, const TokenType& data)
{
    return std::any_of(
        store.begin(), store.end(), [&env, &data](const typename TokenStore::value_type& val) {
            return val.first.env == env && val.second.name == data.name;
        });
}

bool token_exists(const service_data& service, const string& env, const send_token_properties& data)
{
    return token_exists(service.send_tokens, env, data);
}

bool token_exists(
    const service_data& service,
    const string& env,
    const listen_token_properties& data)
{
    return token_exists(service.listen_tokens, env, data);
}

template <typename TokenType>
bool token_exists(service_manager& service_manager, const string& env, const TokenType& data)
{
    auto service = service_manager.find_service_by_name(data.service);
    return service && token_exists(*service, env, data);
}

auto apply_suspended_flags(
    const std::set<tvm_app_info>& origin,
    const std::set<tvm_app_info>& desired)
{
    std::set<tvm_app_info> result;
    for (auto&& origin_app : origin)
    {
        auto result_app = origin_app;
        if (auto&& it = desired.find(origin_app); it != desired.end())
        {
            result_app.suspended = it->suspended;
        }
        result.insert(result_app);
    }
    return result;
}

auto apply_suspended_flags(
    const std::map<string, std::set<tvm_app_info>>& origin,
    const std::map<string, std::set<tvm_app_info>>& desired)
{
    std::map<string, std::set<tvm_app_info>> result;
    for (auto&& [origin_env, origin_apps] : origin)
    {
        if (desired.count(origin_env))
        {
            result[origin_env] = apply_suspended_flags(origin_apps, desired.at(origin_env));
        }
        else
        {
            result[origin_env] = origin_apps;
        }
    }
    return result;
}

}

void list::operator()(
    http_stream_ptr stream,
    settings_ptr settings,
    const auth_info& auth,
    const std::vector<string>& abc_services) const
{
    namespace p = std::placeholders;

    auto service_manager = find_service_manager(settings->api.webui.service_manager);
    if (!service_manager)
    {
        json_value res;
        res["error"] = "not configured";
        json_result(stream, http_codes::ok, res);
        return;
    }

    bool admin = is_admin(settings, auth);
    auto task = std::make_shared<list_task>(stream);
    // Write special fields.
    task->result["admin"] = admin;
    task->result["prefix"] = settings->api.webui.service_create_prefix;
    // List available envs in response.
    task->result["environments"].set_array();
    auto& env = service_manager->environment();
    if (env.empty())
    {
        for (size_t i = 0; i < static_cast<size_t>(ymod_xconf::config_environment::COUNT); ++i)
        {
            task->result["environments"].push_back(
                ymod_xconf::get_environment_name(static_cast<ymod_xconf::config_environment>(i)));
        }
    }
    else
    {
        task->result["environments"].push_back(env);
    }

    if (!admin && !auth.users.size())
    {
        task->result["services"];
        json_result(stream, http_codes::ok, task->result);
        return;
    }

    if (admin)
    {
        service_manager->get_all_services(
            task->ctx, std::bind(&service_list_handler, task, p::_1, p::_2));
    }
    else
    {
        auto owners = std::make_shared<std::vector<string>>();
        owners->reserve(auth.users.size() + abc_services.size());
        std::transform(
            auth.users.begin(),
            auth.users.end(),
            std::back_inserter(*owners),
            [&settings](const user_info& ui) {
                return prefix_by_owner_type(settings, owner_type::user) +
                    static_cast<string>(ui.uid);
            });
        std::transform(
            abc_services.begin(),
            abc_services.end(),
            std::back_inserter(*owners),
            [&settings](const string& abc_service) {
                return prefix_by_owner_type(settings, owner_type::abc) + abc_service;
            });
        service_manager->find_services_by_owners(
            task->ctx, owners, std::bind(&service_list_handler, task, p::_1, p::_2));
    }
}

void service_create::operator()(
    const http_stream_ptr& stream,
    const settings_ptr& settings,
    bool /*admin*/,
    const string& owner_prefix,
    const string& owner_id,
    service_manager_ptr service_manager,
    const string& /*env*/) const
{
    service_properties data;
    auto parse_res = parse_request_properties(stream->request(), data);
    if (!parse_res)
    {
        send_bad_request(stream, parse_res.error_reason);
        return;
    }

    data.revoked = revoke;
    if (!validate_name(data.name) || data.name.size() > settings->api.webui.max_name_length)
    {
        send_bad_request(stream, "invalid service name");
        return;
    }
    if (data.description.size() > settings->api.webui.max_description_length)
    {
        send_bad_request(stream, "invalid service description");
        return;
    }
    if ((data.is_passport && !validate_scopes(data.oauth_scopes)) ||
        data.oauth_scopes.size() > settings->api.webui.max_scope_count)
    {
        send_bad_request(stream, "invalid OAuth scopes");
        return;
    }
    if (data.is_passport + data.is_stream + data.auth_disabled > 1)
    {
        send_bad_request(
            stream,
            "service can only have one property of passport,"
            " stream and auth_disabled");
        return;
    }

    auto& ctx = stream->request()->context;
    data.name = settings->api.webui.service_create_prefix + data.name;
    std::tie(data.owner_prefix, data.owner_id) = std::tie(owner_prefix, owner_id);
    service_manager->create_service(
        ctx,
        data,
        [stream, &ctx](const operation::result& res, std::shared_ptr<service_data> service) {
            if (ctx->is_cancelled())
            {
                return;
            }
            json_value resp;
            if (!res)
            {
                resp["error"] = res.error_reason;
            }
            if (service)
            {
                write_service(resp["result"], *service);
            }
            json_result(stream, http_codes::ok, resp);
        });
}

void service_update::operator()(
    const http_stream_ptr& stream,
    const settings_ptr& settings,
    bool admin,
    const string& owner_prefix,
    const string& owner_id,
    service_manager_ptr service_manager,
    const string& /*env*/) const
{
    service_properties data;
    auto parse_res = parse_request_properties(stream->request(), data);
    if (!parse_res)
    {
        send_bad_request(stream, parse_res.error_reason);
        return;
    }

    data.revoked = revoke;
    if (data.description.size() > settings->api.webui.max_description_length)
    {
        send_bad_request(stream, "invalid service description");
        return;
    }
    if ((data.is_passport && !validate_scopes(data.oauth_scopes)) ||
        data.oauth_scopes.size() > settings->api.webui.max_scope_count)
    {
        send_bad_request(stream, "invalid OAuth scopes");
        return;
    }
    if (data.is_passport + data.is_stream + data.auth_disabled > 1)
    {
        send_bad_request(
            stream,
            "service can only have one property of passport,"
            " stream and auth_disabled");
        return;
    }

    data.owner_prefix = owner_prefix;
    data.owner_id = owner_id;
    // Try to enforce admin constraints.
    if (auto service = service_manager->find_service_by_name(data.name))
    {
        data.tvm_publishers =
            apply_suspended_flags(service->properties.tvm_publishers, data.tvm_publishers);
        data.tvm_subscribers =
            apply_suspended_flags(service->properties.tvm_subscribers, data.tvm_subscribers);

        if (!admin &&
            (service->properties.is_stream != data.is_stream ||
             (data.is_stream && service->properties.stream_count != data.stream_count) ||
             service->properties.revoked != data.revoked ||
             service->properties.queued_delivery_by_default != data.queued_delivery_by_default))
        {
            send_bad_request(stream, "must be admin to edit this field");
            return;
        }
        else if (!admin && service->properties.owner() != data.owner())
        {
            send_bad_request(stream, "service owner mismatch");
            return;
        }
        else if (!admin && service->properties.tvm_publishers != data.tvm_publishers)
        {
            send_bad_request(stream, "must be admin to edit tvm_publishers field");
            return;
        }
        else if (!admin && service->properties.tvm_subscribers != data.tvm_subscribers)
        {
            send_bad_request(stream, "must be admin to edit tvm_subscribers field");
            return;
        }
        // Try to revoke with less side-effects.
        if (data.revoked)
        {
            data = service->properties;
            data.revoked = true;
        }
    }
    else
    {
        // Deprecating xconf owner requires stricter check here.
        send_bad_request(stream, "service doesn't exist");
        return;
    }

    auto& ctx = stream->request()->context;
    service_manager->update_service_properties(
        ctx, data, [stream, &ctx](const operation::result& res) {
            if (ctx->is_cancelled())
            {
                return;
            }
            json_value resp;
            if (res)
            {
                resp["result"] = "success";
            }
            else
            {
                resp["error"] = res.error_reason;
            }
            json_result(stream, http_codes::ok, resp);
        });
}

void send_token_update::operator()(
    const http_stream_ptr& stream,
    const settings_ptr& settings,
    bool /*admin*/,
    const string& owner_prefix,
    const string& owner_id,
    service_manager_ptr service_manager,
    const string& env) const
{
    send_token_properties data;
    auto parse_res = parse_request_properties(stream->request(), data);
    if (!parse_res)
    {
        send_bad_request(stream, parse_res.error_reason);
        return;
    }

    data.revoked = revoke;
    if (!token_exists(*service_manager, env, data) && !validate_name(data.name) ||
        data.name.size() > settings->api.webui.max_name_length)
    {
        send_bad_request(stream, "invalid token name");
        return;
    }

    auto& ctx = stream->request()->context;
    service_manager->update_send_token(
        ctx,
        env,
        data,
        owner_prefix + owner_id,
        [stream, &ctx](const operation::result& res, const string& token) {
            if (ctx->is_cancelled())
            {
                return;
            }
            json_value resp;
            if (res)
            {
                resp["result"]["token"] = token;
            }
            else
            {
                resp["error"] = res.error_reason;
            }
            json_result(stream, http_codes::ok, resp);
        });
}

void listen_token_update::operator()(
    const http_stream_ptr& stream,
    const settings_ptr& settings,
    bool /*admin*/,
    const string& owner_prefix,
    const string& owner_id,
    service_manager_ptr service_manager,
    const string& env) const
{
    listen_token_properties data;
    auto parse_res = parse_request_properties(stream->request(), data);
    if (!parse_res)
    {
        send_bad_request(stream, parse_res.error_reason);
        return;
    }

    data.revoked = revoke;
    if (!token_exists(*service_manager, env, data) && !validate_name(data.name) ||
        data.name.size() > settings->api.webui.max_name_length)
    {
        send_bad_request(stream, "invalid token name");
        return;
    }
    if (data.client.size() > settings->api.webui.max_name_length || !validate_client(data.client))
    {
        send_bad_request(stream, "invalid token client");
        return;
    }

    auto& ctx = stream->request()->context;
    service_manager->update_listen_token(
        ctx,
        env,
        data,
        owner_prefix + owner_id,
        [stream, &ctx](const operation::result& res, const string& token) {
            if (ctx->is_cancelled())
            {
                return;
            }
            json_value resp;
            if (res)
            {
                resp["result"]["token"] = token;
            }
            else
            {
                resp["error"] = res.error_reason;
            }
            json_result(stream, http_codes::ok, resp);
        });
}

void app_info::operator()(
    const http_stream_ptr& stream,
    const settings_ptr& /*settings*/,
    bool /*admin*/,
    const string& /*owner_prefix*/,
    const string& /*owner_id*/,
    service_manager_ptr service_manager,
    const string& app,
    const string& platform) const
{
    auto app_ptr = service_manager->find_app(platform, app);
    if (!app_ptr)
    {
        return send_bad_request(stream, "app doesn't exist");
    }
    json_result(stream, http_codes::ok, make_app_info(*app_ptr));
}

void app_update::operator()(
    const http_stream_ptr& stream,
    const settings_ptr& /*settings*/,
    bool /*admin*/,
    const string& owner_prefix,
    const string& owner_id,
    service_manager_ptr service_manager,
    const string& /*env*/) const
{
    try
    {
        auto transformer = app_config_transformer(stream->request());

        auto app = service_manager->find_app(transformer.platform(), transformer.app_name());
        if (!app && revoke)
        {
            throw std::runtime_error("can't revoke nonexistent app");
        }

        application_config conf;
        if (app)
        {
            if (revoke)
            {
                conf = transformer.revoke(*app);
            }
            else
            {
                conf = transformer.apply(*app);
            }
        }
        else
        {
            conf = transformer.apply({});
        }

        auto& ctx = stream->request()->context;
        conf.updated_at = std::time(nullptr);
        service_manager->update_app(
            ctx, conf, owner_prefix + owner_id, [stream, conf](const operation::result& res) {
                json_value resp;
                if (res)
                {
                    write_app(resp["result"], conf);
                }
                else
                {
                    resp["error"] = res.error_reason;
                }
                json_result(stream, http_codes::ok, resp);
            });
    }
    catch (const std::exception& ex)
    {
        send_bad_request(stream, ex.what());
    }
}

void app_revert::operator()(
    const http_stream_ptr& stream,
    const settings_ptr& /*settings*/,
    bool /*admin*/,
    const string& owner_prefix,
    const string& owner_id,
    service_manager_ptr service_manager,
    const string& /*env*/) const
{
    try
    {
        auto transformer = app_config_transformer(stream->request());

        auto app = service_manager->find_app(transformer.platform(), transformer.app_name());
        if (!app)
        {
            return send_bad_request(stream, "app doesn't exist");
        }

        auto conf = transformer.revert(*app);

        auto& ctx = stream->request()->context;
        conf.updated_at = std::time(nullptr);
        service_manager->update_app(
            ctx, conf, owner_prefix + owner_id, [stream, conf](const operation::result& res) {
                json_value resp;
                if (res)
                {
                    write_app(resp["result"], conf);
                }
                else
                {
                    resp["error"] = res.error_reason;
                }
                json_result(stream, http_codes::ok, resp);
            });
    }
    catch (const std::exception& ex)
    {
        send_bad_request(stream, ex.what());
    }
}

unauthorized::unauthorized(const settings_ptr& settings)
{
    json_value msg;
    msg["error"] = "unauthorized";
    msg["redirect"] = settings->api.webui.auth_redirect;
    unauthorized_message = msg.stringify();
}

void unauthorized::operator()(http_stream_ptr stream)
{
    stream->set_code(http_codes::unauthorized);
    stream->set_content_type("application/json");
    stream->result_body(unauthorized_message);
}

}}}