#pragma once

#include "xivaws_subscription_control.h"
#include "processor/subscriber.h"
#include <yxiva/core/subscription_id.h>
#include <yxiva/core/user_info.h>
#include <yxiva/core/types.h>
#include <ymod_httpclient/call.h>
#include <yplatform/net/types.h>
#include <yplatform/reactor.h>
#include <yplatform/coroutine.h>
#include <yplatform/hash/sha1.h>
#include <boost/bind.hpp>
#include <boost/optional.hpp>

#include <yplatform/yield.h>

namespace yxiva::web {

struct xivaws_subscriber_control : std::enable_shared_from_this<xivaws_subscriber_control>
{
    using yield_ctx = yplatform::yield_context<xivaws_subscriber_control>;

    subscriber_weak_ptr weak_subscriber;
    multi_channels_set channel_keys;

    time_duration ping_interval;

    yplatform::net::timer_ptr timer;
    std::vector<xivaws_subscription_control_ptr> controllers;
    std::vector<web_subscription> web_subscriptions;
    std::atomic<bool> stop{ false };
    future_close future_session_closed;

    xivaws_subscriber_control(
        subscriber_ptr subscriber,
        const multi_channels_set& channel_keys,
        yplatform::net::timer_ptr timer,
        time_duration ping_interval,
        future_close future_session_closed)
        : weak_subscriber(subscriber)
        , channel_keys(channel_keys)
        , ping_interval(ping_interval)
        , timer(timer)
        , future_session_closed(future_session_closed)
    {
        auto settings = find_processor()->settings();

        for (auto&& set : channel_keys)
        {
            for (auto&& key : set.channels)
            {
                auto sub = create_web_subscription(set.ui.uid, key, subscriber->ctx()->uniq_id());
                // create xivaws_subscription_control
                channel_options options;
                if (key.watch_subscribers)
                {
                    options =
                        channel_options{ settings->watch_subscribers_resubscribe_interval,
                                         settings->watch_subscribers_min_subscribe_retry_interval,
                                         settings->watch_subscribers_max_subscribe_retry_interval };
                }
                else
                {
                    options = channel_options{ time_traits::hours(sub.ttl_hours) -
                                                   time_traits::minutes(10),
                                               settings->min_subscribe_retry_interval,
                                               settings->max_subscribe_retry_interval };
                }
                auto controller = std::make_shared<xivaws_subscription_control<>>(
                    timer->get_io_service(), sub, options, subscriber);
                controller->start();
                controllers.push_back(controller);
                web_subscriptions.push_back(sub);
            }
        }
    }

    ~xivaws_subscriber_control()
    {
        try
        {
            for (auto&& controller : controllers)
            {
                controller->stop();
            }
        }
        catch (...)
        {
        }
    }

    web_subscription create_web_subscription(
        const string& uid,
        const channel_key& key,
        const string& extra)
    {
        auto settings = find_processor()->settings();
        auto web_settings = yplatform::find<web::impl>("web")->get_settings();
        web_subscription sub = { uid,
                                 key,
                                 "", // gen the ID later
                                 callback_uri::xiva_websocket_uri(
                                     settings->notify_url + "/" + static_cast<string>(key.service) +
                                     "?subscription_id=${subscription-id}"),
                                 settings->ttl.get(key.client),
                                 get_priority(key.service, web_settings->service_features) };
        sub.id = key.make_sub_id(
            uid, key.service, sub.callback_url, key.client, key.session, key.filter, extra);
        return sub;
    }

    void operator()(yield_ctx yield_ctx, const boost::system::error_code& ec = {})
    {
        if (ec || stop)
        {
            return;
        }

        reenter(yield_ctx)
        {
            future_session_closed.add_callback([this, self = shared_from(this)] {
                stop = true;
                timer->cancel();
            });

            while (true)
            {
                {
                    auto subscriber = weak_subscriber.lock();
                    if (!subscriber) return;

                    if (!subscriber->is_expired())
                    {
                        subscriber->ping(ping_message{ ping_interval });
                    }
                    else
                    {
                        YLOG_CTX_GLOBAL(subscriber->ctx(), info) << "signature expired, closing";
                        subscriber->notify_error(error_message{ "sign expired" });
                        subscriber->close();
                        return;
                    }
                }

                timer->expires_from_now(ping_interval);
                yield timer->async_wait(yield_ctx);
            }
        }
    }

private:
    string get_priority(const string& service, const service_features& features)
    {
        return features.high_priority_websockets.enabled_for(service) ? "high"s : "low"s;
    }
};

}

#include <yplatform/unyield.h>