#pragma once

#include <boost/bind.hpp>
#include <boost/optional.hpp>
#include <yplatform/net/types.h>
#include <yplatform/reactor.h>

#include "channel_key.h"
#include "xivaws_subscriptions_storage.h"
#include "hubrpc/impl.h"
#include "processor/subscriber.h"
#include "api2/watch_subscribers.h"
#include <yxiva/core/user_info.h>
#include <yxiva/core/types.h>

#include <ymod_httpclient/call.h>

namespace yxiva::web {

typedef std::set<subscriber_ptr> subscribers_t;

struct channel_options
{
    time_duration resubscribe_interval = seconds(50);
    time_duration min_subscribe_retry = seconds(1);
    time_duration max_subscribe_retry = seconds(5);
};

struct channel_dep_traits
{
    using xivaws_subscriptions_storage = xivaws_subscriptions_storage;
    using timer = steady_timer;
};

template <typename Traits = channel_dep_traits>
class xivaws_subscription_control
    : public std::enable_shared_from_this<xivaws_subscription_control<Traits>>
    , public boost::noncopyable
    , public yplatform::log::contains_logger
{
    typedef typename Traits::xivaws_subscriptions_storage xivaws_subscriptions_storage;
    typedef typename Traits::timer timer;

public:
    xivaws_subscription_control(
        boost::asio::io_service& io,
        const web_subscription& sub,
        channel_options const& options,
        subscriber_ptr subscriber);
    ~xivaws_subscription_control();

    string full_name() const;

    void start();
    void stop();

private:
    void subscribe();
    void handle_subscribe(const boost::system::error_code& ec, yhttp::response response);
    void unsubscribe();
    string ctx_string(task_context_ptr ctx);
    void kick_subscriber();
    void set_timer(const time_duration& duration);
    void cancel_timer();
    void handle_timer_resubscribe(const boost::system::error_code& e);
    time_duration calc_resubscribe_time();
    time_duration calc_retry_subscribe_time();

    boost::asio::io_service& io_;

    task_context_ptr ctx_;
    web_subscription sub_;
    subscriber_weak_ptr weak_subscriber_;
    channel_options options_;
    bool subscribed_ = false;
    unsigned subscribe_attempt_ = 0;
    time_point subscribe_time_ = time_point::min();
    mutable shared_mutex mux_;

    std::shared_ptr<xivaws_subscriptions_storage> xivaws_subscriptions_storage_;
    std::shared_ptr<timer> timer_;
    std::shared_ptr<web::api2::watch_subscribers_coro> watch_coro_;
};

typedef std::shared_ptr<xivaws_subscription_control<>> xivaws_subscription_control_ptr;

}

#include "xivaws_subscription_control.ipp"