#include "find_deps.h"
#include "web/impl.h"
#include "web/messages.h"
#include <yxiva/core/types.h>
#include <yxiva/core/callbacks.h>

namespace yxiva::web {

namespace p = std::placeholders;

namespace {

template <typename C, typename T>
void get_from_header(const C& c, const string& key, T& t)
{
    auto it = c.find(key);
    if (it != c.end())
    {
        t = boost::lexical_cast<T>(it->second);
    }
}

}

template <typename Traits>
xivaws_subscription_control<Traits>::xivaws_subscription_control(
    boost::asio::io_service& io,
    const web_subscription& sub,
    channel_options const& options,
    subscriber_ptr subscriber)
    : io_(io), ctx_(subscriber->ctx()), sub_(sub), weak_subscriber_(subscriber), options_(options)
{
    logger().set_log_prefix(ctx_->uniq_id());

    xivaws_subscriptions_storage_ =
        yplatform::find_module<xivaws_subscriptions_storage, std::shared_ptr>(
            io_, "xivaws_subscriptions_storage");
    timer_ = std::make_shared<timer>(io_);
}

template <typename Traits>
xivaws_subscription_control<Traits>::~xivaws_subscription_control()
{
    try
    {
        if (auto subscriber = weak_subscriber_.lock())
        {
            subscriber->notify_disconnected(
                disconnected_message{ user_id(sub_.uid), sub_.key.service });
        }
    }
    catch (...)
    {
    }
}

template <typename Traits>
void xivaws_subscription_control<Traits>::start()
{
    write_lock lock(mux_);
    auto subscriber = weak_subscriber_.lock();
    if (!subscriber) return;

    subscribe();
    if (auto catalogue = find_processor()->catalogue())
    {
        catalogue->add(sub_.uid, sub_.key.service, sub_.id, subscriber);
        find_xivaws_log()->log_subscriber_add(sub_.uid, sub_.key, ctx_string(ctx_));
    }
}

template <typename Traits>
void xivaws_subscription_control<Traits>::stop()
{
    write_lock lock(mux_);
    if (watch_coro_)
    {
        *watch_coro_->stop = true;
    }
    cancel_timer();
    kick_subscriber();
    unsubscribe();
}

template <typename Traits>
string xivaws_subscription_control<Traits>::ctx_string(task_context_ptr ctx)
{
    return (ctx ? ctx->uniq_id() : string("INVALID_UNIQ_ID"));
}

template <typename Traits>
void xivaws_subscription_control<Traits>::subscribe()
{
    xivaws_subscriptions_storage_->subscribe(
        ctx_,
        sub_,
        std::bind(
            &xivaws_subscription_control<Traits>::handle_subscribe,
            shared_from(this),
            p::_1,
            p::_2));
}

template <typename Traits>
void xivaws_subscription_control<Traits>::unsubscribe()
{
    xivaws_subscriptions_storage_->unsubscribe(ctx_, sub_, [](auto, auto) {});
}

template <typename Traits>
string xivaws_subscription_control<Traits>::full_name() const
{
    return "{" + sub_.uid + "," + static_cast<string>(sub_.key.service) + "," + sub_.key.filter +
        "," + sub_.key.client + "," + sub_.key.session + "." + sub_.id + "}";
}

template <typename Traits>
void xivaws_subscription_control<Traits>::handle_subscribe(
    const boost::system::error_code& ec,
    yhttp::response response)
{
    if (!ec && response.status == 403)
    {
        YLOG_L(error) << "can`t subscribe channel " << full_name()
                      << " reason: " << response.reason;
        write_lock lock(mux_);
        kick_subscriber();
        return;
    }

    if (ec || response.status != 200)
    {
        YLOG_L(error) << "can`t subscribe channel " << full_name()
                      << " reason: " << (ec ? ec.message() : response.reason);
        write_lock lock(mux_);
        subscribe_attempt_++;
        set_timer(calc_retry_subscribe_time());
        return;
    }

    write_lock lock(mux_);
    auto subscriber = weak_subscriber_.lock();
    if (!subscriber) return;
    subscribe_attempt_ = 0;
    subscribe_time_ = clock::now();
    set_timer(calc_resubscribe_time());

    // Notify subscriber only once.
    if (subscribed_) return;
    subscribed_ = true;

    if (hacks::connected_message_enabled(sub_.key.service))
    {
        subscriber->notify_connected(connected_message{ user_id(sub_.uid), sub_.key.service });
    }

    if (sub_.key.fetch_history)
    {
        local_id_t pos = 0;
        unsigned count = 0;
        get_from_header(response.headers, "x-xiva-position", pos);
        get_from_header(response.headers, "x-xiva-count", count);
        subscriber->notify_position(
            position_message{ user_id(sub_.uid), sub_.key.service, pos, count });
        YLOG_L(info) << "channel " << full_name() << " history position=pos" << pos
                     << " count=" << count;
    }

    if (sub_.key.watch_subscribers)
    {
        auto formatters_kit = web::find_processor()->formatters();
        // Will throw an exception if not found.
        auto formatter = formatters_kit->get("json");

        watch_coro_ =
            std::make_shared<web::api2::watch_subscribers_coro>(web::api2::watch_subscribers_coro{
                io_,
                sub_.key.service,
                std::vector<string>({ sub_.uid }),
                [weak_subscriber = weak_subscriber_] { return weak_subscriber.lock(); },
                yplatform::find<web::impl>("web")->get_settings(),
            });

        yplatform::spawn(io_.get_executor(), watch_coro_);
    }
}

template <typename Traits>
void xivaws_subscription_control<Traits>::set_timer(const time_duration& duration)
{
    YLOG_L(debug) << full_name() << " resubscribe in " << time_traits::to_string(duration);

    timer_->expires_from_now(duration);
    timer_->async_wait(boost::bind(
        &xivaws_subscription_control<Traits>::handle_timer_resubscribe, shared_from(this), _1));
}

template <typename Traits>
void xivaws_subscription_control<Traits>::cancel_timer()
{
    timer_->cancel();
}

template <typename Traits>
void xivaws_subscription_control<Traits>::kick_subscriber()
{
    auto subscriber = weak_subscriber_.lock();
    if (!subscriber) return;

    if (auto catalogue = find_processor()->catalogue())
    {
        auto deleted = catalogue->del(sub_.uid, sub_.key.service, sub_.id);
        if (deleted)
        {
            find_xivaws_log()->log_subscriber_del(sub_.uid, sub_.key, ctx_string(ctx_));
        }
    }

    YLOG_L(info) << "kick " << ctx_string(subscriber->ctx()) << " from " << full_name()
                 << " subscribers collection";
    boost::asio::post(io_, [subscriber, uid = sub_.uid, service = sub_.key.service]() {
        subscriber->notify_disconnected(disconnected_message{ user_id(uid), service });
    });
}

template <typename Traits>
void xivaws_subscription_control<Traits>::handle_timer_resubscribe(
    const boost::system::error_code& e)
{
    if (e == boost::asio::error::operation_aborted)
    {
        return;
    }

    YLOG_L(info) << "timer handler triggered for subscription " << full_name();
    write_lock lock(mux_);
    if (!weak_subscriber_.lock()) return;
    subscribe();
}

template <typename Traits>
time_duration xivaws_subscription_control<Traits>::calc_resubscribe_time()
{
    auto now = clock::now();
    auto next_expire = std::max<time_duration>(
        options_.resubscribe_interval - (now - subscribe_time_), time_traits::seconds(0));

    return next_expire;
}

template <typename Traits>
time_duration xivaws_subscription_control<Traits>::calc_retry_subscribe_time()
{
    // Simply limit attempts count to prevent overflow issues.
    static const unsigned max_attempts = 30;
    if (subscribe_attempt_ >= max_attempts)
    {
        return options_.max_subscribe_retry;
    }
    return std::max<time_traits::duration>(
        options_.min_subscribe_retry * (1 << subscribe_attempt_), options_.max_subscribe_retry);
}

}
