#include "apns_test_certificates.h"
#include <yplatform/encoding/base64.h>

namespace yxiva::web::webui::test_certificates {

const string current::pem =
    R"(-----BEGIN PRIVATE KEY-----
MIIEvgIBADANBgkqhkiG9w0BAQEFAASCBKgwggSkAgEAAoIBAQDNTmFVBSU3WN8r
pkcEGr0H2K3wSEl3eqv0tShb0pfVQoRXPslYtfpMrFcCZqeCFNdx2xJNUjeOEqMa
a783x/NQw4gd/dqo+P6BiwAeHLT6e4JD0afaDXs2xSllc4DPcSwR63LNYmxJwCDK
UwqmMpzLiGh8TZ05FeIxotuWSfZJ3oiFHgXjuaFySFMg4PUoplzgnN3sNc/JEBTx
j/Suq3/NRBW41+H80QiM6hYDGRagBmJ1jOSCAqlay5E7EgTqtJkWMHyf6qf76/5G
Qxl2ZHcVRaDv+BISzL+tC1aMBaoUCgdEH0h7tWxv6g4PHF48K1Hzd6X2f40m+kwm
N6X1FoZDAgMBAAECggEAcT/WMubms5QjbtmG/P2e3lApCo1lcTHK3u57N8OAG5RY
miO5P2UKj/bU/C8SIZBLk7OLcflzSbyAGhfrX+K9FgT19c1NK75F7cF2LCYbsrwk
jDJ/S6n6GKlDf4HqQQOlkrD1H9Q19ay0qcaK7kim8yLouAC8RHwMzOKQDGP6Jp5M
awqfHop7NzwW+D22r1sLlya3YnuLhTKt7BeONuQDYwT3wl1/jhkJXhaaiTsLDafC
HJ9neP3hQ0ibXJKK/wfkDKI2pukusBQMfUz3Hq9r4lzSaanCHIMJZpWASDy5SyAk
XiFZ9YpDeYUVk4JTITFXaGp+MF4YfLH7ED7s/MAOCQKBgQDpb4lR9tGqTeo4IpT+
xwgHVA8mPJFqkLgYPliqLvn9ubOetFyNtjSOjJ5fk933zkUFpyhnj0wKWO/lDNTx
vhprD2Kxxg76yVZUrwAmm1ARGGTpZlgDuiYXVfr6Ran/hB6aL3lYQxX7k5P+WRhq
yfHObeW4scYN6td7a6nsyJ80dwKBgQDhJsRlHYI24HKY+8G6Dedr+XamSvHuAaNu
fFpZUSjrgRNm11mhHKny/kN6yCmJBXjg/f9HqF7IEjliJKpJlW3dH/ZWqIta39cZ
BXzNAER0tFoi6NkDbpLxjOXAp9kIIUFgcgwAh8qYSM/z9jr4D0I2af0Xs4SvVVQN
2MUiOosrlQKBgQCepnRgpWv33/5oanBy+9gkCuhE53M5RuQnbfAasdSrsZM3MIOa
uJPdsgpNERURvzm/zsyyHXzjp3Bbbq4nIjmYO5urEN3LahznbwyeMH5pL/gdfLYi
ylfrEj+B6ujPxP57cfp5lxaQpcs2WV0z7s4Nqzkunnvpp4sN0a1qRSuWnwKBgBLu
1p8GHzgMIbbZnM3Pc4Fx5ddQAA4alyqoTFSsbXt4TDBetYvFxEjJttTxVgTOr/6D
fV8COo4qcibdnHA5n5ZzpJxe6fEHV6gj/P2kIx2JOPcsi2stdi/XwE42DxsO6Nd6
X6X/xTqPDOWf9A9+APxgZ0cB8otn6nyLmPAq6h8NAoGBAMmG3EkFAV8W/lwk2Sdk
PtauWMdTttXc1l3/09olkSa4F8PaasJDQ0G21qF0qlXXSXg6zlHb+iWd4LZ5OZaA
4QG2tXjmJ17ijQh84AL98yKPEyM4sGQc1FI5D58OuvgJQLZRAipGPYfLzbSxVevA
MM+idAVqKyPvKwhBr5HQ6ou4
-----END PRIVATE KEY-----
-----BEGIN CERTIFICATE-----
MIIDdDCCAlwCCQCx2G/uBeui/zANBgkqhkiG9w0BAQsFADB8MQswCQYDVQQGEwJV
UzELMAkGA1UECAwCNTIxEDAOBgNVBAcMB05vd2hlcmUxDTALBgNVBAoMBG1haWwx
GTAXBgNVBAsMEHhpdmFzZXJ2ZXJfdXRlc3QxFTATBgoJkiaJk/IsZAEBDAV0b3Bp
YzENMAsGA1UEAwwEcm9vdDAeFw0yMDAyMDQxNTE3NTRaFw0zMDAyMDExNTE3NTRa
MHwxCzAJBgNVBAYTAlVTMQswCQYDVQQIDAI1MjEQMA4GA1UEBwwHTm93aGVyZTEN
MAsGA1UECgwEbWFpbDEZMBcGA1UECwwQeGl2YXNlcnZlcl91dGVzdDEVMBMGCgmS
JomT8ixkAQEMBXRvcGljMQ0wCwYDVQQDDARyb290MIIBIjANBgkqhkiG9w0BAQEF
AAOCAQ8AMIIBCgKCAQEAzU5hVQUlN1jfK6ZHBBq9B9it8EhJd3qr9LUoW9KX1UKE
Vz7JWLX6TKxXAmanghTXcdsSTVI3jhKjGmu/N8fzUMOIHf3aqPj+gYsAHhy0+nuC
Q9Gn2g17NsUpZXOAz3EsEetyzWJsScAgylMKpjKcy4hofE2dORXiMaLblkn2Sd6I
hR4F47mhckhTIOD1KKZc4Jzd7DXPyRAU8Y/0rqt/zUQVuNfh/NEIjOoWAxkWoAZi
dYzkggKpWsuROxIE6rSZFjB8n+qn++v+RkMZdmR3FUWg7/gSEsy/rQtWjAWqFAoH
RB9Ie7Vsb+oODxxePCtR83el9n+NJvpMJjel9RaGQwIDAQABMA0GCSqGSIb3DQEB
CwUAA4IBAQAH3AHJgMf5nGfoqhJrotnbLhwJKnEyM1+cPtTMTJJrszMNN9gYcS0t
sJ07/+pdbTGUeDmyXC1Onb/gdOUGTNKGxj20N97FefqAY6F2y36avUw9sXzo/8QL
xqIkXDElXqfmfyd4mjvQqB+YsblNwsep8lPyw06D8Xmr0DOiI851eC2XRbEE/pwX
lKBI7nGHXueZ8O3YcpzQRpcu8MoNSBmO981U4/nfkwX/OPgmQ5FJecs5B5gfEBVM
hPjLrkJ/T0dJ0WZkTcgEEE3exx5RKgSm5039wluwtEK3bXAm3TPc2ATlXSQcUSv/
9f/EUB/SU1StCN49jLMSm7MTd5dIcGTa
-----END CERTIFICATE-----
)";

const string current::p12(
    "\x30\x82\x09\xb9\x02\x01\x03\x30\x82\x09\x7f\x06\x09\x2a\x86\x48"
    "\x86\xf7\x0d\x01\x07\x01\xa0\x82\x09\x70\x04\x82\x09\x6c\x30\x82"
    "\x09\x68\x30\x82\x04\x1f\x06\x09\x2a\x86\x48\x86\xf7\x0d\x01\x07"
    "\x06\xa0\x82\x04\x10\x30\x82\x04\x0c\x02\x01\x00\x30\x82\x04\x05"
    "\x06\x09\x2a\x86\x48\x86\xf7\x0d\x01\x07\x01\x30\x1c\x06\x0a\x2a"
    "\x86\x48\x86\xf7\x0d\x01\x0c\x01\x06\x30\x0e\x04\x08\xe9\x6b\x5e"
    "\xcf\xaf\x8d\x1b\x21\x02\x02\x08\x00\x80\x82\x03\xd8\x63\xd0\x4a"
    "\x71\x38\x09\x79\xd1\x45\x65\xff\x5d\xcd\x8a\xfb\x44\xe6\x7c\x75"
    "\xce\xc2\xdb\x45\xb1\x1e\x02\x0e\x6c\xaf\x88\x50\xb8\xb4\x8b\x81"
    "\x0d\xb0\x87\x14\x76\x35\x42\xe5\x3d\x77\xd1\xee\x9e\xff\xf2\xe4"
    "\xe9\x4c\x30\x2e\xfb\x0c\xc7\x17\x63\x20\x11\x5d\x0c\xa2\x1b\x50"
    "\x22\x31\xa6\x0b\x1e\x4d\x32\x13\x12\x78\x5a\xd2\xe9\xf4\x79\xb2"
    "\xed\x67\xee\xe3\x59\x5b\x5d\xb7\xf8\x77\xee\x23\x9a\x58\xc0\x2a"
    "\xf4\x24\x4c\x14\x43\x04\x46\xd0\xa6\x50\x24\x94\xeb\xc4\xd9\x34"
    "\x08\x33\xe5\x1f\x5d\x18\xb9\xc0\x15\x5b\x35\x15\x8d\x92\x7a\x80"
    "\xab\x3b\xaf\x84\x54\xfc\xa2\xdc\x2c\xa9\xc6\xab\x5f\xff\x68\x64"
    "\xb3\xba\x1d\x24\xfb\x6d\x50\x51\x0c\xd6\x1f\xcb\xf4\x56\xcb\x8f"
    "\x02\x36\xea\x70\xa6\xa3\xf7\x40\x0e\xaf\x31\xe8\x56\x08\x98\x65"
    "\x0e\x89\x25\xb6\xe7\xcf\xdf\x86\x35\x1b\x99\x89\xff\x3d\x1f\x07"
    "\x5d\xe6\x35\x1a\x86\xf5\x3d\xc3\x87\x6e\xd7\xe8\x64\xc7\x1f\x60"
    "\x9f\xdd\xd0\xe2\x29\xe8\xe2\x65\x16\x8e\x84\x8f\x50\xe4\xea\x51"
    "\xbe\x97\xfb\x9d\xc5\xde\x89\xde\x23\xe2\x16\x16\x68\x81\x65\x0c"
    "\x70\x3d\x32\xe4\xfc\x1a\x5c\xaa\x85\xc7\xf7\xd1\xdb\xab\xa4\xad"
    "\x5c\x9d\xba\x39\xcd\xa3\x82\x24\x62\xb0\xfb\x2e\xe2\x11\xd5\xf6"
    "\xab\x36\xe5\x60\x94\xd3\x93\x83\x1c\xe0\xb7\xe7\xe5\xe0\x90\xd3"
    "\x13\xee\xf6\x83\x71\xf2\x6f\x17\x63\x3a\x8e\x45\x03\x77\x4a\xf7"
    "\x5d\x6d\xf1\xf5\xe3\x35\xad\xd8\xd4\x11\x05\x37\xf8\x5b\xc6\x68"
    "\xcd\xfc\xce\x6c\xd9\x25\xe0\x38\xc5\x96\x80\x6d\x78\x25\x2d\x88"
    "\xf8\x08\xe4\x84\x45\x7b\x01\x04\xe8\x0b\x05\x30\x7b\xc8\x8a\x47"
    "\xa0\xae\xe8\x15\x2a\x27\x02\x92\x14\xe3\x6b\x8b\xb1\xaa\xf3\x5f"
    "\x91\x08\xc7\xc2\xc7\x76\xba\x07\xc4\xf6\x9f\xc3\x29\x94\x02\xe1"
    "\x91\xb3\x51\xcc\x8f\xdb\x61\xcb\x1c\x76\x65\xf5\x97\xb8\xd0\xd8"
    "\x4b\x4a\x02\x09\x8c\x62\x05\x12\x67\x53\x33\xd8\xd6\x71\x56\x11"
    "\xe7\x16\x3c\x4f\xa0\x1b\xda\x38\x7c\xa6\x67\xf0\x3d\xe4\xa6\x8e"
    "\xb9\xfe\x31\xd6\xfd\x07\x05\xd1\x33\xb4\x3e\xfc\x10\xae\xf4\x7f"
    "\xec\x92\xd6\x6f\x82\x79\x26\x4f\x52\x2c\x8f\x98\x0a\x3b\x78\x3a"
    "\xe7\xec\xb7\x49\x50\xfd\x78\xcf\x29\xab\x97\x9f\xdc\x54\x1e\x03"
    "\xa0\x4a\x6c\xbe\xa7\x45\xbb\x0e\x49\x21\x8d\xb5\x5b\xfe\xfc\xfd"
    "\xca\xc9\x45\xd7\xad\xfd\x15\xb1\x6c\x03\x26\xde\x66\x09\xf6\xba"
    "\x89\x2c\xd1\x55\x01\x4f\xaa\x19\xda\xc7\x66\x5a\xc0\x82\x2d\xb2"
    "\x89\xe2\x94\x20\xca\x96\xdd\xa6\x9e\xc2\xc5\xf2\xf0\x59\x2d\x2e"
    "\x23\x12\x9f\x32\x93\xac\x8a\x03\x65\x46\x4e\x84\x71\xcb\x81\x13"
    "\x27\xcf\x40\x7f\x92\x0e\x57\x73\x24\xa6\xcf\x6b\x2a\x38\x27\xd3"
    "\xb5\xe3\xcf\x2a\x51\x7d\x0c\xde\x4f\x3d\x16\x54\xb1\x11\xe1\x7b"
    "\x3e\x71\xda\x5d\x5b\x21\x64\x50\x8f\xeb\x7e\x4d\x79\xfd\x0e\x3f"
    "\xb2\x80\x7e\x38\x92\x94\x9d\x58\x96\x31\x5e\xb9\xd4\x0e\x74\x4a"
    "\x76\x84\x45\x32\xf3\x20\xfa\x7d\x9e\xe6\x66\xd8\x52\xbe\xd2\xdd"
    "\xc7\x5b\x35\x89\x4d\xab\xea\x16\xdb\xad\x0d\x18\x05\xa7\xe5\x04"
    "\x94\x83\xaa\x3b\xf8\x59\x0b\x5c\xca\xd7\x7f\xfd\xdb\x19\xc6\x43"
    "\xba\x1a\x0d\x6e\x62\xd5\x8b\x37\x44\x06\xae\x9d\x32\xcf\xde\x87"
    "\x27\xe7\xaf\x3a\x36\xc5\x56\xda\x0c\x1a\xae\x33\xf2\xb1\xb2\xea"
    "\xcd\x09\x8c\x37\xcd\x44\xc6\x54\x45\xee\x12\xe0\xf3\xb3\x15\x92"
    "\xdf\xe3\xc7\x72\x39\x66\x2f\xba\xca\x94\xb8\xc9\x13\x64\xcb\xcc"
    "\xab\x8c\xb9\xa6\xb2\xa8\x47\x67\x6c\x6b\x63\xbb\x26\x22\xae\x48"
    "\x42\xa0\x44\x07\x63\xcf\x0b\x42\x1b\x8f\x0c\xff\x05\xcf\x21\xae"
    "\x1e\x9d\xb3\x92\xdd\xd4\x32\x11\xd9\x73\x1d\x16\x5d\x6f\x46\x40"
    "\xdb\x5a\xe0\xc9\xb4\xcb\x7b\xb8\xdc\x37\xda\x6d\x63\x8c\x59\x12"
    "\x13\xcc\x19\x6c\xe0\x6c\x57\xcc\xb4\xd2\x86\xf3\xbd\xbf\x10\xfb"
    "\xa6\x49\x95\x1c\x2d\x13\xbe\xa0\xa1\xe9\x7a\xc0\xb4\xa7\x35\xec"
    "\x94\xfc\xe4\x88\x04\xe0\x36\x2e\x53\x9e\xb1\x6c\x78\x2d\x06\xaf"
    "\x51\xf6\xe6\x67\x24\x08\x27\xe2\xd4\x85\x96\x0d\x01\x49\xe7\xac"
    "\x23\xf6\xf6\xb5\x49\xb8\x90\x6d\x35\x5d\xb3\x19\x46\xe0\xf2\x6d"
    "\x00\xd6\xbf\x02\xa8\x0b\x61\x55\x04\x65\x12\x0f\x70\x11\x6d\x0d"
    "\xa5\xeb\x42\x66\x29\x85\x83\xde\xed\x55\xbe\xf1\x8b\x56\x64\xbb"
    "\x71\xd2\x8f\xc5\x1a\xc9\xe6\xde\x3d\x4d\xe1\x6f\x95\x5e\x3a\x24"
    "\x78\x72\x41\xdd\x0b\xfc\xa6\x6c\xc3\x44\xd3\xa6\x24\x9a\x5e\xaf"
    "\x91\xe5\x75\x2a\x2b\xc8\x31\x64\xc0\xa8\x0e\x67\x54\x6e\x07\x4c"
    "\x35\xae\xb8\x4d\x26\x55\x50\x2e\xa8\x0d\xa1\xd9\xbb\x42\xea\xfb"
    "\x5d\xd4\xcf\xf9\xee\x30\x82\x05\x41\x06\x09\x2a\x86\x48\x86\xf7"
    "\x0d\x01\x07\x01\xa0\x82\x05\x32\x04\x82\x05\x2e\x30\x82\x05\x2a"
    "\x30\x82\x05\x26\x06\x0b\x2a\x86\x48\x86\xf7\x0d\x01\x0c\x0a\x01"
    "\x02\xa0\x82\x04\xee\x30\x82\x04\xea\x30\x1c\x06\x0a\x2a\x86\x48"
    "\x86\xf7\x0d\x01\x0c\x01\x03\x30\x0e\x04\x08\x89\x4d\xad\xd3\xc3"
    "\x4b\x6c\x69\x02\x02\x08\x00\x04\x82\x04\xc8\x56\x00\x17\x25\xd8"
    "\x40\xd3\x8f\x3c\xe3\x40\x89\x64\x36\x4a\xd3\x93\x82\xd0\x4d\xc0"
    "\x3c\x3f\xfc\x89\x50\x1d\xb9\x2a\x44\x72\x5b\x81\x70\x78\x9f\xfb"
    "\x4f\xea\xcb\x46\xde\xe1\xc0\x4e\x7b\x77\x5c\xab\x49\xcd\x4e\x99"
    "\xc8\x13\xce\x2a\x57\x60\x35\x1f\x4d\xa4\x8c\x5e\x2a\x2f\x45\xc3"
    "\x1b\x69\x49\x37\x8c\x1d\xc8\xcf\x21\x1c\x32\x3e\x08\x9b\x88\x01"
    "\xf5\x3f\x54\xce\x08\xf5\x59\xa6\xd8\xa2\xca\xeb\x5c\x27\xf8\xbf"
    "\x3d\x7d\x52\xed\x37\x79\x3b\x57\xff\x91\xb8\x29\x4d\x10\xe6\xb1"
    "\xb2\xa7\x75\x21\x32\x35\xa4\x52\xc1\x0b\xf0\xe0\xd1\xf9\x5f\x3b"
    "\x79\x4f\x72\x67\x62\x1e\x34\xd2\x8d\xfc\x43\x9b\x27\x1e\x6a\x72"
    "\x9e\x5a\xa5\x78\x13\x14\xd6\x4d\x31\x12\xfd\xd5\xa5\xdd\x64\x4d"
    "\xf4\x3a\xdc\x0b\xba\x28\xd6\xc6\x91\xec\x04\xb0\x3e\x04\xe5\xa7"
    "\x4e\x89\xce\xc9\x30\xbc\x09\x2b\xb8\xfa\x95\x8a\xca\xcf\x63\x63"
    "\xf1\xf7\x91\xd0\x9d\x54\xba\xfe\x29\xfe\xe6\x1e\x20\x20\x46\x2e"
    "\x57\x1d\xe6\x85\xf8\x4d\x5c\x26\x0a\x49\x2e\x68\xbd\x68\x04\x50"
    "\x85\x2a\x8a\xed\x12\x30\x14\x0b\x81\xa6\xaa\x20\x5b\x50\x7c\x59"
    "\x22\x3c\xaf\xaa\xed\x5b\x6f\x3d\x05\x99\xcf\x54\xe9\x8d\xab\x34"
    "\x44\x52\x55\x8d\x75\x6c\x2a\x36\x8e\xcf\x64\x9f\xb2\xd5\x60\x2d"
    "\xcd\x56\x5e\xb2\x72\xb4\x83\x5e\xf7\x90\xae\x87\x60\x75\x59\x36"
    "\x57\x7a\x0a\x34\x18\xe0\x03\x1b\x59\xbe\xed\x6a\x72\x5a\xbf\xdc"
    "\xcf\x06\x8b\x8d\x7d\x10\x24\x43\x83\x49\xa8\x76\x95\x14\xa6\x3d"
    "\x67\x42\x86\x82\x34\x65\x53\x21\xc5\x0c\x27\xd6\x1a\xfd\xba\x32"
    "\xe9\x98\xf2\xd8\xf3\xbc\xa0\x5e\xf6\xbf\xb5\x2d\xf0\x86\xcd\xa7"
    "\x7b\x60\xaa\x8f\xf4\x4f\xfc\x9e\x3a\xd0\x1e\x01\x97\xab\x0c\x1c"
    "\x5c\x99\x41\x7f\x9b\x58\x14\xb0\xcd\xb8\xda\x70\xde\x85\x51\x13"
    "\x27\x7c\x1f\x2c\x9d\xcc\xf6\x9f\x07\x7b\x8a\x99\x4f\x13\xad\xd7"
    "\xc4\x1a\x20\x03\x2d\x6e\xba\x46\xbc\x4c\x44\xd0\x53\xb8\x88\x3a"
    "\xa5\x35\xc8\xd8\xf0\x03\xa2\x61\xc3\x28\xa2\xb5\x22\x1f\x3b\x16"
    "\xc2\x99\x95\xf4\xb8\xde\xc6\x64\x02\x7d\xe1\xbe\x8e\xaa\xc9\x76"
    "\x41\xc9\xaa\x4b\x92\x9a\xab\x72\x49\xf1\x94\x4f\xee\xd4\x29\xb7"
    "\x87\x66\x1a\x99\x95\x51\x71\x8d\x5f\x6b\xda\xbe\x1e\x0b\x1a\x34"
    "\xab\xea\x8a\xee\x9f\x61\xc6\xe4\x37\x23\xd8\xda\x28\xfd\x8b\x39"
    "\x35\x49\x70\xb9\x70\xbb\x5d\xe9\x58\x74\x0a\x34\xff\xf4\xcc\xc0"
    "\x00\xbf\x0e\x0a\xe4\x05\x44\x15\xf9\x2f\x1f\xea\xed\x63\x4e\x28"
    "\x51\x33\xb4\x6f\x94\x49\x5b\xbf\x85\x8f\xd8\xc1\xf2\xca\x72\x68"
    "\xde\xb7\x6e\x6f\xa4\x5b\xf1\xb3\x59\xd9\xb4\xb8\x2f\xb5\xd1\xc6"
    "\x01\x47\x7b\xd8\xb0\xd1\xaa\x60\x00\xa6\xd2\xef\xd7\xa9\xec\xe2"
    "\x6c\x36\x78\x5f\x58\x7f\x6f\x76\xab\xb4\x1c\xad\x34\xef\x04\x4d"
    "\xd7\xeb\x28\x75\x9b\x5f\x94\x86\x1b\x4a\x2f\x49\xe5\x43\x26\xdb"
    "\x6b\x52\x1a\xab\x76\x26\xba\x47\x9d\xf0\xfd\xea\x38\x5e\x5a\xff"
    "\x53\x6f\x85\x7b\x9d\x8b\x9c\xa7\x67\xde\xb8\xa0\xf0\xc6\xa9\xdd"
    "\x1a\x20\x75\x38\x57\xdc\xa8\x35\x27\x67\x0b\x04\x12\x03\x35\x6f"
    "\x02\xd4\x3d\x3b\x8a\xb9\xf3\x5a\x67\x9b\x16\xc1\x54\xf9\xe0\x54"
    "\xb1\x18\xcf\x59\xf3\x35\x40\x96\x31\xf6\x97\xc7\x1e\x1b\xe0\xf2"
    "\x80\xbb\x2d\xfc\xf8\xcd\xcb\x5e\x90\x3c\x89\x07\x8e\x16\x00\x66"
    "\x16\xf4\x1c\xc6\xd1\xf6\x02\x57\x17\x3a\x12\x2c\x45\x73\x78\x50"
    "\x76\xb2\x44\xab\xb1\xfc\x81\x9b\x75\x4d\x16\x09\x85\xad\xf1\x04"
    "\xd8\xa3\x7a\xe3\xe4\x7a\x7a\x2e\xd7\x32\x5c\xa0\x84\xff\x08\xe8"
    "\xfa\xb0\x64\x88\xd2\xa9\xef\xd7\xf9\xb2\x48\x55\xee\x72\x78\xdd"
    "\x21\x2e\xb1\xde\xa3\x08\x73\x6b\x20\xac\x42\x18\x41\x54\x1b\xfb"
    "\x92\xf6\xb1\xdb\xfa\xbd\x37\x55\x8f\x16\x4f\xfa\x57\x53\xda\x2b"
    "\x10\xa2\x0b\x7c\xd1\x51\x96\x37\x5a\x80\x65\xb9\x4f\xae\x18\xde"
    "\xc1\xb0\x2d\x07\x47\x6c\x2e\x05\xdd\xd2\x32\xf4\x23\x3c\xa8\xfb"
    "\x62\x83\x43\xab\x80\xee\x7d\x14\x76\xb3\x19\x9f\xcf\xf5\xbe\xa1"
    "\xe4\x81\x72\x1b\xa5\x32\xa8\x4a\xbb\xa9\x16\x55\x8c\x3b\x2c\x28"
    "\xfe\x02\x28\xc9\x2d\xc1\xab\x5c\x37\xd7\xa4\xf5\xd8\x55\x8d\xf6"
    "\x8b\xda\xf5\x73\x28\x0e\x70\xd4\xba\xf9\x1f\x7a\xa7\x10\x3c\x7e"
    "\x77\x9c\xa2\x70\xb5\x48\x07\x23\x99\x7f\x18\xd7\x01\x35\xa6\x66"
    "\xad\x79\xb2\x86\x80\x71\x41\x34\xbe\xc2\xa8\x2f\x74\xcc\x4f\xc9"
    "\xf1\xa0\xef\x8a\xc3\x2d\x78\x46\xa7\x10\x11\x8e\x93\xed\x56\x8a"
    "\xd1\xb7\x86\x8c\x3c\x0c\xe2\x4f\x24\x3f\x34\x44\xfe\x5c\xbb\xc9"
    "\x84\x12\x8b\x83\xf0\xb5\x69\xb7\xfa\xfa\x80\x5f\xab\x3b\xa5\xc1"
    "\x62\xcc\x74\x32\x55\xf3\x00\x22\x78\x86\x64\x51\x97\x80\xdd\xcd"
    "\xf6\xee\x8e\x67\xd6\xfb\xe8\xb0\xcc\x01\xbd\xd2\xb7\x39\x21\x9d"
    "\x65\x0b\x58\xc6\xd6\x00\x82\x41\x5c\xf3\x83\xec\x94\x3d\xf7\x40"
    "\xb9\x6f\x0a\xaf\x1e\x69\xe6\x68\xb7\x64\x4c\x94\x23\xa1\xc6\x33"
    "\x08\x94\x17\x0f\xd4\x85\xf5\x0f\xf0\x32\x0a\x85\xab\x83\xf0\x80"
    "\x2e\xfb\x52\x67\x98\x2b\x49\xeb\x06\x69\xed\x04\x35\x81\x2b\x0a"
    "\x4a\xb2\x36\xf2\xf3\xd4\x32\x29\x4c\x4f\x42\xbb\xbd\x85\x57\x66"
    "\xf4\x3c\xfc\xb1\xe9\xd6\xbf\xd5\xee\x19\xbe\x76\x20\x98\x94\xbd"
    "\x19\x9d\x32\xf4\xdd\x84\xc4\x8b\x16\xa8\x87\xc3\x01\x8d\x5b\x8b"
    "\x0a\x87\xea\x7c\x06\x93\x8c\x51\x41\x01\xcf\xb9\xc8\x84\x0d\xbd"
    "\x0f\x2c\xa3\xb6\x18\x8e\x05\x57\xf2\x02\x80\x3c\xa1\xbe\x62\x56"
    "\xe5\x4a\x69\xb7\xe4\x32\xe3\x78\xae\xf7\xab\xec\x9c\xfe\x31\xce"
    "\x97\xd9\xe6\x1a\xe7\xe8\x36\xf8\x4a\xb0\x61\xe4\x5e\x72\xb1\x4c"
    "\x49\x40\x14\x9e\x43\x25\xa3\x17\xa1\xa9\x52\x0e\xea\x84\xad\xb0"
    "\x46\x95\x19\xd9\x94\xc7\xb8\xc3\xf6\xee\x1f\x65\x5c\x22\xbb\x45"
    "\xdd\xf4\x4d\x31\x25\x30\x23\x06\x09\x2a\x86\x48\x86\xf7\x0d\x01"
    "\x09\x15\x31\x16\x04\x14\xcd\x75\xd0\x77\xbd\xc3\x72\x4f\xca\xa4"
    "\x8a\x1d\x28\xb7\x69\xb9\xd7\x71\x82\xea\x30\x31\x30\x21\x30\x09"
    "\x06\x05\x2b\x0e\x03\x02\x1a\x05\x00\x04\x14\x43\xb4\x6c\x6f\x63"
    "\xeb\xb2\x94\xae\xdc\xd8\xd4\x1e\xcf\xa9\xe6\x69\x8a\xa0\xed\x04"
    "\x08\xd5\xf2\x47\x3a\x43\xfa\xb2\xed\x02\x02\x08\x00",
    2493);

const yxiva::apns::p8_token current::p8{
    R"(-----BEGIN PRIVATE KEY-----
MIIEvgIBADANBgkqhkiG9w0BAQEFAASCBKgwggSkAgEAAoIBAQDNTmFVBSU3WN8r
pkcEGr0H2K3wSEl3eqv0tShb0pfVQoRXPslYtfpMrFcCZqeCFNdx2xJNUjeOEqMa
a783x/NQw4gd/dqo+P6BiwAeHLT6e4JD0afaDXs2xSllc4DPcSwR63LNYmxJwCDK
UwqmMpzLiGh8TZ05FeIxotuWSfZJ3oiFHgXjuaFySFMg4PUoplzgnN3sNc/JEBTx
j/Suq3/NRBW41+H80QiM6hYDGRagBmJ1jOSCAqlay5E7EgTqtJkWMHyf6qf76/5G
Qxl2ZHcVRaDv+BISzL+tC1aMBaoUCgdEH0h7tWxv6g4PHF48K1Hzd6X2f40m+kwm
N6X1FoZDAgMBAAECggEAcT/WMubms5QjbtmG/P2e3lApCo1lcTHK3u57N8OAG5RY
miO5P2UKj/bU/C8SIZBLk7OLcflzSbyAGhfrX+K9FgT19c1NK75F7cF2LCYbsrwk
jDJ/S6n6GKlDf4HqQQOlkrD1H9Q19ay0qcaK7kim8yLouAC8RHwMzOKQDGP6Jp5M
awqfHop7NzwW+D22r1sLlya3YnuLhTKt7BeONuQDYwT3wl1/jhkJXhaaiTsLDafC
HJ9neP3hQ0ibXJKK/wfkDKI2pukusBQMfUz3Hq9r4lzSaanCHIMJZpWASDy5SyAk
XiFZ9YpDeYUVk4JTITFXaGp+MF4YfLH7ED7s/MAOCQKBgQDpb4lR9tGqTeo4IpT+
xwgHVA8mPJFqkLgYPliqLvn9ubOetFyNtjSOjJ5fk933zkUFpyhnj0wKWO/lDNTx
vhprD2Kxxg76yVZUrwAmm1ARGGTpZlgDuiYXVfr6Ran/hB6aL3lYQxX7k5P+WRhq
yfHObeW4scYN6td7a6nsyJ80dwKBgQDhJsRlHYI24HKY+8G6Dedr+XamSvHuAaNu
fFpZUSjrgRNm11mhHKny/kN6yCmJBXjg/f9HqF7IEjliJKpJlW3dH/ZWqIta39cZ
BXzNAER0tFoi6NkDbpLxjOXAp9kIIUFgcgwAh8qYSM/z9jr4D0I2af0Xs4SvVVQN
2MUiOosrlQKBgQCepnRgpWv33/5oanBy+9gkCuhE53M5RuQnbfAasdSrsZM3MIOa
uJPdsgpNERURvzm/zsyyHXzjp3Bbbq4nIjmYO5urEN3LahznbwyeMH5pL/gdfLYi
ylfrEj+B6ujPxP57cfp5lxaQpcs2WV0z7s4Nqzkunnvpp4sN0a1qRSuWnwKBgBLu
1p8GHzgMIbbZnM3Pc4Fx5ddQAA4alyqoTFSsbXt4TDBetYvFxEjJttTxVgTOr/6D
fV8COo4qcibdnHA5n5ZzpJxe6fEHV6gj/P2kIx2JOPcsi2stdi/XwE42DxsO6Nd6
X6X/xTqPDOWf9A9+APxgZ0cB8otn6nyLmPAq6h8NAoGBAMmG3EkFAV8W/lwk2Sdk
PtauWMdTttXc1l3/09olkSa4F8PaasJDQ0G21qF0qlXXSXg6zlHb+iWd4LZ5OZaA
4QG2tXjmJ17ijQh84AL98yKPEyM4sGQc1FI5D58OuvgJQLZRAipGPYfLzbSxVevA
MM+idAVqKyPvKwhBr5HQ6ou4
-----END PRIVATE KEY-----
)",
    "12asdg3",
    "ABsdgasdCD",
    "test",
    "development"
};

const string backup::pem =
    R"(-----BEGIN PRIVATE KEY-----
MIIEvQIBADANBgkqhkiG9w0BAQEFAASCBKcwggSjAgEAAoIBAQDNV70itpdrHOje
bqYJl3yTgILX3oUxYZOLzHyJWeaE0oPdyhEp6kHfsSXuIsR3eQkqZXfiiiYaApk1
BzcVexba6zr8aFwYAoUBoUn1rDx6Ud1aPTZXRH+Nr0he5WdmkrI8XlofVeCOUNP3
eZlMZ86l09SMuzatpNmdsezTE/rmzHhf2nPVrcqv1P6BkJLdQMiwMLUQxDVznzIK
He7ZIdI0L3Rgvl00gS0EPfEqa0gVuDkjVLR6HBhSJKPB2jhnC3PeaH49jEHsCx/U
I5cx8vK8S4q5uSNLRGXKvLHMn1iREa6ZFMjMMEfAUWv4Es+sXDEHaMbMUB8v1Com
9vN8avzzAgMBAAECggEAVwTwnQ5d+sGEmgawDG6LF4Mlb8/YCAY+3capnBdiDjWc
NlhUj7u7cM2Esnc/ZRYA4Vugw3siDHG/dW56VHsWQ23eUGr1ZCNlNDyiuB7mfjvF
H2HDbpYTq/p1I+n5uMlACT8+1/qvr//m5AA/C9cOBs5PS9hmoo2yvxZQs17LPBT1
g5pFfUZtb975h3mh+wqvn6h79JzKfC6RAIBL+gHjY8FlggJY6xPb0X0QBNJjcc66
/vwuVBHh6OXsdru+uWbGpkKPi2/4OeWjn6ncXJmjSU8f9JippkSnNVNcbB50dUR9
6b90ZMmCWuIRaLLC3gPOjr3AUOE2HCiqtiD657JlcQKBgQD5kdRhcR/F3RwKGB5Z
Fe5hRA5mcQN3qaeTigS9rIfhwlOvABUlNPSX6PA7Jn0KRpJRtRQqwSAEsyTnkq5l
F/aS5+A2QO+HkkthmJlqLVLOHU29C6NZ/WCAx+pQwg9uZRgUcUCO+hBD8iaGSMIu
7BGfcN6Eh9Z3xh3bzFH7+A58OQKBgQDSoi/V/CODJbt1TnSi8e27jFIhq0I/Xrhd
ytTzLIdhbZCni7VeQatAFJghDQ++smWUtvVy2fjzuvK+gqASygCuvG6VmU5yMlFB
ngQWYBwvZZuwY5xL2oLKyYUg+YBlbpA05fA1iqcdnsRP36HIl3N/rFiVswH0WUUf
4YJ+Ju3aiwKBgQDTN/LRTVagtVPHz0XQC9mmhFzToo5WtkVXfxYoGV8fWMJOd/lt
dfYUo1a+GQL2sWegs2qQDMc2cw+m1eO3N6K+DnUaNbcDvHMD9IzS6e0M+IwUlUb5
IJKOK6RKGRPKUNEia/9Bgq5FcynbuLRv1efMU1BgRj7jO8+FeNViNTy9MQKBgEI/
CFSAzJp+z4qBdmWZOW/3Hg3A00EjBVFabcddhea6kvhfFFfkx3uL/pzGguEzMoen
U2Mf8xCVB6urYbUHG1fZo94hLW71LGE61YZEDBlMF79OVOX2f5RP9TL9xcghEfHR
L89nIFBMkumcKEHygDzzc5kBh+fV+8ufHW+Xxs8pAoGAS2gpRlbwMyWwTAi1jAZP
bOmUw1BUTshGFcLivi8n6x4ZEGfNcbrpk3XSSXxMmlrC3z/9dM/w21v7H5LfIVMn
yTADfRpIQwloGscDaBFwFgBojh+cbX9+Zlpx7+eZXqpTC00kLlFDd7jlzw09sx9g
9C/3JHRZCFjZPCeGyjyQqpA=
-----END PRIVATE KEY-----
-----BEGIN CERTIFICATE-----
MIIDdDCCAlwCCQCVjL3QmqNOwTANBgkqhkiG9w0BAQsFADB8MQswCQYDVQQGEwJV
UzELMAkGA1UECAwCNTIxEDAOBgNVBAcMB05vd2hlcmUxDTALBgNVBAoMBG1haWwx
GTAXBgNVBAsMEHhpdmFzZXJ2ZXJfdXRlc3QxFTATBgoJkiaJk/IsZAEBDAV0b3Bp
YzENMAsGA1UEAwwEcm9vdDAeFw0yMDAyMDQxNTE3NTdaFw0zMDAyMDExNTE3NTda
MHwxCzAJBgNVBAYTAlVTMQswCQYDVQQIDAI1MjEQMA4GA1UEBwwHTm93aGVyZTEN
MAsGA1UECgwEbWFpbDEZMBcGA1UECwwQeGl2YXNlcnZlcl91dGVzdDEVMBMGCgmS
JomT8ixkAQEMBXRvcGljMQ0wCwYDVQQDDARyb290MIIBIjANBgkqhkiG9w0BAQEF
AAOCAQ8AMIIBCgKCAQEAzVe9IraXaxzo3m6mCZd8k4CC196FMWGTi8x8iVnmhNKD
3coRKepB37El7iLEd3kJKmV34oomGgKZNQc3FXsW2us6/GhcGAKFAaFJ9aw8elHd
Wj02V0R/ja9IXuVnZpKyPF5aH1XgjlDT93mZTGfOpdPUjLs2raTZnbHs0xP65sx4
X9pz1a3Kr9T+gZCS3UDIsDC1EMQ1c58yCh3u2SHSNC90YL5dNIEtBD3xKmtIFbg5
I1S0ehwYUiSjwdo4Zwtz3mh+PYxB7Asf1COXMfLyvEuKubkjS0RlyryxzJ9YkRGu
mRTIzDBHwFFr+BLPrFwxB2jGzFAfL9QqJvbzfGr88wIDAQABMA0GCSqGSIb3DQEB
CwUAA4IBAQBTtZfx4+REwPrfHlfMelZ1t04zkyEjmF6NKTFx4El2YKNBgIzT3bsP
xXw6jV5EDxJog06RJbnByGz1NkcRlyS23uYIal9xe3L9umwdBtvwTH08qUeL3mWa
psV/cLiPGNKuUP3No8GEUNQS1o0yOPqwYVRkR3laELqUigdE38E2rPnk7EGdBof2
UNib+HVJ3Ojf7l8eyd/NDuxfZf5lK8RUcxaZBOWK4lETL51XfmwVpul7qooZqDXW
jnKL02DFB7USRvyA3rAY+d77ePuiojw/tb6WCln6ZiHwAT2stjHwm4JzYzH+3b8T
/8HokSBVvOrLne9wRc1GUAySsUuxza5U
-----END CERTIFICATE-----
)";

const string backup::p12(
    "\x30\x82\x09\xb9\x02\x01\x03\x30\x82\x09\x7f\x06\x09\x2a\x86\x48"
    "\x86\xf7\x0d\x01\x07\x01\xa0\x82\x09\x70\x04\x82\x09\x6c\x30\x82"
    "\x09\x68\x30\x82\x04\x1f\x06\x09\x2a\x86\x48\x86\xf7\x0d\x01\x07"
    "\x06\xa0\x82\x04\x10\x30\x82\x04\x0c\x02\x01\x00\x30\x82\x04\x05"
    "\x06\x09\x2a\x86\x48\x86\xf7\x0d\x01\x07\x01\x30\x1c\x06\x0a\x2a"
    "\x86\x48\x86\xf7\x0d\x01\x0c\x01\x06\x30\x0e\x04\x08\xd0\x8f\x26"
    "\x48\x62\x14\xe8\x9d\x02\x02\x08\x00\x80\x82\x03\xd8\x9a\x2a\x72"
    "\xb9\x6b\x2d\x0d\xe8\x15\xb8\xd6\xea\x05\x7f\xc2\x83\xd8\xac\xf3"
    "\x78\xab\x74\x08\xda\x88\x28\xf1\xdb\x03\xf9\x8d\x21\x02\x2d\x63"
    "\x56\x0d\x43\xf0\xf7\xc2\x55\xde\xaa\xf2\x7f\x0e\x36\x82\xe5\xd7"
    "\xf7\xf9\xfa\x57\x5d\xe8\x32\x7a\xdd\xf2\xe5\x2e\xa5\x53\x00\x28"
    "\x9d\xc4\x40\xac\xa1\x25\xa9\x7f\xe4\x6b\x8f\xfa\x1a\x47\x95\x82"
    "\xce\x8e\x44\x10\x54\xf9\xba\x24\xa2\x59\x8d\x4f\x4f\x8d\x19\x72"
    "\x8e\xcc\x5a\x5d\xe2\xf3\x62\x78\x1c\xb7\xe6\x2b\xbb\x55\xec\xcc"
    "\xad\x08\xd3\x31\x29\xd8\x78\x31\x56\xd6\xc3\x9e\xab\xfb\x32\x9b"
    "\x25\xd3\xd8\x5e\x92\x7f\x0d\x1e\x9f\x2e\x90\x98\xb9\xf5\xdc\x6d"
    "\x93\x9d\xdf\x68\x94\x5f\x2f\x7f\xf5\x92\x7c\x86\x13\x06\x1a\x1e"
    "\x58\x76\x66\x0c\xc7\x53\xae\xc8\x8d\x8a\x1f\x26\x33\x25\xee\xcd"
    "\x9e\x05\xc4\x6f\x51\x12\x34\x45\x7a\x2e\x1b\xef\x44\xe0\xe7\x24"
    "\xba\xac\x4a\x14\x61\x69\xa2\x2f\xcf\x8f\xd6\x15\x59\x17\xd7\xf0"
    "\xfb\x2c\x0b\xb1\x1b\x01\xac\x42\x7a\xd5\x76\x9d\x5f\xa5\x56\x99"
    "\xb3\xc0\xf9\xb5\xdd\xb5\x28\x98\x7d\xad\xd0\x51\x51\xd9\x0f\x94"
    "\x37\x69\xc9\x2f\x96\xea\xef\x56\x34\x5e\x8c\x11\xfa\x58\x63\x7a"
    "\xa5\xcc\x55\xa8\xa0\xbb\x45\x2a\x8f\x3c\xd9\x3a\xc7\x1a\x94\x94"
    "\xad\x07\x24\xe1\x98\x1b\xaa\x65\x00\xb8\x4d\xc9\xbe\x76\x21\x53"
    "\x15\x12\xd9\x59\x36\x4e\xab\x34\x22\xf6\xb6\x43\xa8\x10\x31\x4d"
    "\x40\x98\x43\x7f\xe7\x74\x93\x18\xe2\x5d\x17\xdc\x8b\x67\x3a\x5a"
    "\xe3\x95\x58\xe4\x2a\xe2\x53\xb1\xad\xf9\x39\x77\xef\xeb\x88\x67"
    "\x13\x75\xb5\x22\x99\x7c\x04\x20\x6a\xfb\x2a\xd2\xd8\x3d\x1e\xe8"
    "\x0f\x2a\x59\x9d\xe4\x0e\xcf\x09\x0f\xe0\xec\x1b\xa3\xde\x64\x1b"
    "\x63\x00\x41\xab\xb7\x5b\x24\x25\xbf\xa4\x57\xa5\xd0\xdb\x2e\x4f"
    "\xe1\x86\x6d\x1c\x8b\x93\xb8\x19\xd3\x28\xb9\x1a\xa2\xd2\x4f\xd9"
    "\x9b\x01\xce\xb5\x65\x2b\xcf\x09\xed\xf9\xbe\x2f\xfb\x85\x09\xf9"
    "\x9d\x4d\x70\x78\xf5\x0c\x9f\x6b\xb6\x97\xa9\x8d\x97\xe3\xc6\xa2"
    "\xd3\x9a\x30\x30\x5b\x14\xbc\x1e\x35\x5e\x7a\xf9\xe0\xcd\x2d\x4f"
    "\x77\xed\xb6\xbf\x5e\xed\x87\x91\xf9\xb0\x62\xa8\xf0\xfd\xbd\xac"
    "\x0c\xc6\x77\x21\xaf\xc9\xf0\xea\xf4\x9b\x3c\xe2\x65\xc0\x16\x54"
    "\xf4\xb0\x3d\x6c\x6b\xd3\x87\x59\x3b\x70\x3a\x28\xaf\x29\x1a\x7e"
    "\x90\xd0\xb1\x4a\xd6\x9d\x1d\xf9\x25\xd0\x47\x86\xc1\xea\x40\x9c"
    "\x4a\x90\x66\x1d\xe8\xff\xee\xf0\x26\xc6\xe8\x5a\x63\x84\xd6\x7b"
    "\x21\x0e\xbd\xec\x9c\xd2\x7d\x98\xcf\xd2\x7f\xeb\x02\x35\x1f\x22"
    "\x5b\x0e\x89\x6d\x7d\xb7\xcc\xee\x65\x15\x37\x2c\xfa\x78\x60\xa7"
    "\xff\xba\xb6\x76\x60\x95\x39\xa0\x28\xa3\xff\x1b\x19\x96\x43\x2f"
    "\xba\xcb\x58\xba\x79\x95\x23\x8e\x1a\x6f\xe1\x3b\xd4\x48\x59\x36"
    "\xbd\x1b\x29\x44\x91\x63\x66\x42\xb6\x97\x27\x4a\xb1\xfb\x8c\xe4"
    "\x4e\x54\xd9\xc8\x4d\x5f\x09\x1d\xd5\x91\xe0\x3d\x29\x78\x34\x95"
    "\x3b\x53\x6e\xef\x49\xe3\x2e\xd5\x9a\x54\x9e\xf6\x49\xa6\x8c\x8c"
    "\xb9\x63\x9b\xb8\x0f\x29\x27\xa9\x85\x19\x82\x98\xf5\xa0\xb8\xfd"
    "\x18\xa1\x7b\x33\xd9\x0c\x38\x9a\x9e\x54\xdf\x69\xcb\xcd\xc0\x95"
    "\xaf\x93\xf9\x5b\x81\xf0\x06\x70\xf4\x36\xcd\x27\xab\x73\xea\xf3"
    "\x5c\x6a\x80\xfa\xa0\x72\x2c\xc2\xe2\xc8\xcd\x53\x9d\x95\xe8\x3f"
    "\xf5\x98\xe8\xd0\x7e\xc1\xd0\x81\xe8\x8a\xe8\x9b\xa8\xb6\xca\xc4"
    "\x45\xca\x8f\x3f\x49\xc2\xf1\xcb\xc9\x83\x04\x3f\x32\x13\xf2\xaa"
    "\xf8\xa8\xd6\x0a\x6d\xa4\x29\xfa\x7f\x7e\xfe\x3c\xd8\x15\xd7\x07"
    "\xc4\x2a\x90\x10\x82\xb4\x1c\x1d\x8b\x4c\x45\xf1\x03\xc2\x07\x83"
    "\x7f\xe6\x55\x36\x4b\xca\xf4\x23\xfd\xa7\x76\x7e\x7b\x88\x8d\xff"
    "\xfc\x37\xea\x45\xe7\x55\x69\xc4\xe2\xe0\x1d\xbb\x17\x68\x59\xb7"
    "\x22\x36\x5c\x18\x7e\x04\x58\x0e\xb7\xf9\x34\xa6\x88\x15\x56\x8f"
    "\x7d\xd3\x0a\xe0\xc3\x0c\x8f\xea\xb3\xbc\x00\x1c\xc1\xbb\x50\x37"
    "\xc4\x66\x13\x75\x5e\xf8\x7c\x40\x6e\x44\x4b\x23\x43\x38\xe3\x16"
    "\x15\xd4\xe2\x15\x4f\x1c\x11\xdf\x36\x23\x8e\x6f\x2c\x7d\x15\x82"
    "\x3a\x62\xdb\xde\xf2\x0b\x99\xd5\xd2\xd7\x1a\xc3\x04\x6e\x64\xbc"
    "\x69\xec\x93\x97\x27\x09\x55\x82\x18\xb8\x1c\xc7\x48\x49\x50\xdb"
    "\xfb\xb2\x0b\xfb\xe0\xcb\xcb\x22\x56\x72\x78\xdd\xa3\xa7\xa9\xb5"
    "\xbf\xa6\xd0\xaf\x01\xb1\xe9\x7a\xb6\x07\xc9\x4e\xec\xda\xf7\xb4"
    "\xe8\x70\x1a\xc7\x6d\xc2\x5c\xb6\xbd\x95\x75\xb0\x05\x88\xee\x00"
    "\x54\x1e\xf3\xa5\xec\x7a\xeb\x92\xf0\x12\xc3\xd8\xc4\xc1\xc4\x64"
    "\xec\xdb\x5c\x78\x60\x7d\x2b\xca\xea\x37\x4a\x23\xb5\xcc\x03\x9e"
    "\xa7\xbb\x20\xb7\x84\x30\x82\x05\x41\x06\x09\x2a\x86\x48\x86\xf7"
    "\x0d\x01\x07\x01\xa0\x82\x05\x32\x04\x82\x05\x2e\x30\x82\x05\x2a"
    "\x30\x82\x05\x26\x06\x0b\x2a\x86\x48\x86\xf7\x0d\x01\x0c\x0a\x01"
    "\x02\xa0\x82\x04\xee\x30\x82\x04\xea\x30\x1c\x06\x0a\x2a\x86\x48"
    "\x86\xf7\x0d\x01\x0c\x01\x03\x30\x0e\x04\x08\x20\xe8\x88\x40\x66"
    "\xf4\xf3\x8e\x02\x02\x08\x00\x04\x82\x04\xc8\x97\x88\x87\x10\x89"
    "\x23\x2b\x62\xc0\xf5\xf3\x93\x17\xf3\x6a\x46\xef\x13\x3b\xd4\x73"
    "\x11\xd3\xac\xe0\xd3\xc7\x6b\x1a\xc7\x61\x15\x7b\x60\xa7\xc9\x30"
    "\xbf\xeb\xc4\xf8\x50\xbf\x82\x55\xe8\xbd\xb2\xf0\x31\x19\x73\x86"
    "\xd1\x12\x1b\x6e\xdf\x7d\xe5\x9f\x17\x03\x6a\x94\xa0\xc6\xe4\x8a"
    "\x67\xf3\xd8\xa3\x4f\x70\xb4\xf7\x01\x60\xe5\x6b\xa8\xdc\xb9\xd9"
    "\x54\x3e\x5c\x16\x3a\x22\x9e\xc5\xe1\x80\x33\xa6\x62\xd4\x08\xd3"
    "\x2b\x00\x50\x6e\x4d\x71\x39\x0a\x17\x68\xa5\x16\x59\xb5\x79\xd3"
    "\x7c\x50\xc1\x02\x4d\x1a\x4a\x21\x44\x3a\x5c\xde\x52\x41\xd7\xcb"
    "\xca\x82\x4d\x11\x23\x7d\x52\x3d\x3a\xe0\x79\x11\x82\x90\xfe\x6d"
    "\x67\xe9\xda\x0b\x5c\x85\xae\x88\x10\x50\x00\xdc\xb7\xe2\xc9\xc7"
    "\xc0\x43\xb7\xf1\xec\x48\xce\x52\xd3\x09\x4b\x34\x77\x21\x2c\x65"
    "\xff\x60\xd3\x0f\x54\xf9\x4a\xaa\xbe\xa6\xa4\x1c\xbb\x95\xbd\x08"
    "\x66\x1d\x6d\x3d\x99\x8b\xca\x1b\xc5\xd4\x5f\x2e\xc4\x60\x1a\x4c"
    "\xbb\xdd\x3a\x67\xf3\xdc\x36\x43\x14\x55\xed\x5d\x0c\xc7\x7a\x0a"
    "\xeb\xe8\xb9\x6b\x05\x6c\x51\xc9\xb7\x74\xcf\x65\x52\xfd\xe6\x0d"
    "\x50\x13\xa5\x4e\xa5\x29\xb3\xb5\xbe\xed\xc9\x6c\x02\x11\x99\x51"
    "\x5d\xbf\x30\xa8\x21\x2c\x86\x9b\xd5\x53\xd7\x44\x1d\xad\x24\x21"
    "\x05\x83\x10\x05\xbf\x8f\x7f\x5f\xb1\x8e\xbd\x8e\x81\xa1\xb6\xbb"
    "\xfd\xcb\xc0\xb0\x50\x7d\x7b\xb2\x80\x73\x3e\xed\xb4\x61\x97\x6e"
    "\xed\x34\x18\x50\xd4\xf0\x39\xed\x3d\xa3\xc9\xc2\x47\x13\xcd\x71"
    "\x17\xe3\x62\x8d\x11\xca\x54\xb3\xce\x8b\xc2\xa8\x3e\x64\x26\x3a"
    "\xbe\x13\xb6\x60\x64\x1c\x9b\xfb\x53\x92\xdb\xe6\x9f\xf1\xe0\xc2"
    "\x87\x58\x8d\x8f\xaa\x5f\x84\xae\xe7\x66\xda\xbd\x89\x18\x16\xff"
    "\x50\x80\xca\xe6\xe1\x79\xe7\x2a\x8b\xc5\x94\x62\x9a\xca\x4a\xed"
    "\x6b\xe2\x16\xb1\x0b\xd9\x7e\xfc\x92\xba\x92\xd9\xfc\x13\x60\xf8"
    "\xbe\xb0\xae\xfa\xea\xba\xed\xe0\x3a\x4c\x8b\x49\xb1\x7e\x51\x71"
    "\xea\xb5\x1c\x24\x19\xac\x72\x97\x8a\xe4\x81\x30\x0c\x13\xb9\xe5"
    "\x94\x02\x26\xf2\xe5\x6d\xa1\x61\x06\x01\x7c\x77\xc5\x7e\x8a\x69"
    "\xe5\xd3\xdb\xf4\xac\xad\xad\x61\x2f\x4d\x6b\xd5\x12\xe8\x99\xdb"
    "\xf9\x9e\xcc\x36\xa6\xc2\x79\x9e\xa4\x93\xe7\xf8\x7c\x14\x0e\xd5"
    "\x76\xb2\x36\x66\x1e\x80\x27\xc7\x72\xa4\x2a\x1b\x0e\x4d\x10\x3a"
    "\xef\xe0\x42\xc7\xa9\x82\x17\x4b\x96\xd8\xb9\xda\xdf\x26\x79\xe2"
    "\x55\x67\x07\xa4\x66\xb3\x91\xce\x77\x91\x69\x37\xd2\xf9\x7c\xa5"
    "\xf8\xd5\xd7\xc0\x83\xb8\xd2\x97\x81\x93\x3a\xa2\x1e\xc6\xde\x0a"
    "\x7b\xeb\x56\xcd\xbe\x44\xd6\x08\xed\xc7\x70\x0b\x1e\x78\x6e\x0c"
    "\xa2\xf8\xc7\x7a\xd8\x85\x2c\x89\xb8\x6c\x73\x83\x92\xf2\x76\xf1"
    "\xf8\xdd\x97\xb0\x5f\xf3\xf2\xd8\x60\xf4\xdc\xff\x63\xf3\xdc\x1f"
    "\x61\xd3\xeb\x80\x4f\x1d\x2b\xef\xe7\x65\x24\x73\xfd\x0b\x94\x6f"
    "\x92\x06\x45\x85\xc9\x5b\x25\x38\x4f\xae\x84\x41\x56\xa0\x11\x22"
    "\x75\x8d\x08\xfb\x76\xab\xe0\x58\x52\x65\xa7\xc7\xc8\x58\x10\xab"
    "\x21\xac\x75\xd6\x12\xa3\x0b\x2b\x8b\x46\x7d\xa3\xea\x57\x84\xc6"
    "\x71\xed\x22\x7e\x78\xe3\x5f\xd5\xa9\x57\x31\x40\xea\xa8\xc8\xd0"
    "\x46\x21\x14\x53\xb0\x24\x72\x94\xcb\xc7\x59\xb1\xce\x38\xd7\x78"
    "\x82\x84\xab\xe5\x24\x78\x03\xb4\xa0\x01\xa2\xf3\x16\x94\xe2\x9b"
    "\xb2\x34\x34\x89\x96\xaa\xd0\xb8\x0d\x3c\x1d\xd0\x9d\x10\x7e\x61"
    "\xe1\xb7\x11\x72\x99\x02\xf9\xb8\x14\x4f\xf1\x2b\xd4\x52\xc3\x1a"
    "\x1c\x1f\x00\x33\xce\x69\xb4\x44\xbe\xff\xe7\x94\xcd\xf6\xcd\xff"
    "\xb7\x05\xb9\x0e\xd0\x1f\x1c\x4c\xfb\x76\x0a\x97\xa5\xea\x46\x5f"
    "\x0f\xf6\x00\xa8\xd4\x95\xec\x3c\x1c\x86\x17\x24\x79\x33\xcc\xef"
    "\x5b\x20\x9e\xd4\xdf\x7e\xd3\x1e\x24\x06\x29\x1d\x88\x60\x5e\x35"
    "\xcc\x95\x46\x7d\xcf\x8a\x73\x86\xab\x0f\x28\xd5\xc6\x5b\xba\x7b"
    "\xca\x64\x55\x7e\x19\x56\x1c\x33\x72\xbf\xab\xf7\x3a\x08\xec\x6a"
    "\xfa\x63\x37\x2c\xa1\xab\x0e\x85\x80\x3f\xa6\xc2\x61\x88\x64\x14"
    "\xc7\x5e\xea\xd0\x4d\xe5\x7b\xa5\x48\xf3\x38\x52\x61\x5c\x21\xfb"
    "\xa7\x23\x37\xc1\xce\x5a\x2a\x4c\x86\x44\xfc\xe8\xa8\xb8\x52\x89"
    "\x3f\xaf\x17\xba\x2a\xf0\x71\x90\x53\xe8\xbc\x57\xb9\xed\xd9\x70"
    "\x22\xcd\x50\x94\x37\xc0\x1d\x40\xdc\x81\xd0\x14\x10\x7d\x75\xb8"
    "\x39\x7f\x7c\x38\x65\xdd\xe2\xea\x65\x7c\x09\xd4\x34\xc6\x3c\x94"
    "\x8b\x96\xd2\x45\xc0\x1a\xe8\xc6\xee\x60\x2b\xa3\x31\xcc\x1d\x44"
    "\x7b\x93\x00\xb3\xb6\x5f\xdb\x28\x73\xaf\x73\x48\x9c\xad\x21\xa5"
    "\x68\xa4\x3b\x4b\x72\x6d\xcb\xb7\x8c\x20\xe8\xe7\x11\xe3\xa8\x6f"
    "\x56\x04\x6e\x8e\x71\x2f\x01\x32\x5c\x5e\xa8\xe5\x1e\xb4\x69\x6d"
    "\x92\x35\xaf\xe3\xc2\x55\x19\x4c\x74\x11\xcb\x7b\xa0\x00\xc9\xde"
    "\x1f\x2e\x55\x41\x99\xe0\x68\x24\x84\x5e\xa4\xd8\xa3\xeb\x31\x6a"
    "\x6f\x0b\x6c\x3f\x02\x54\xd0\x8e\x6f\xd4\xbd\x18\xb8\x51\x2f\x5c"
    "\x3e\xde\xe6\x36\x6e\x92\x5d\xd0\x5d\x57\xb1\x97\x24\x96\x1b\x9f"
    "\x1a\x86\xac\x54\x85\x4e\x58\xfc\x1c\x50\xa6\x27\x43\xb5\x17\xa2"
    "\x12\xb1\x13\x96\x72\x4e\x97\x1d\xd9\x0d\xb3\x0f\x0f\xf1\x6b\x2f"
    "\xc1\x31\x43\x1a\xf2\x27\x43\x78\x35\x7a\x5f\x46\x61\x92\xd7\x3b"
    "\x47\x58\x53\x19\x71\xec\x84\x36\x42\x3d\x28\x7f\x56\xf7\xf9\x5b"
    "\x29\x4c\x94\x84\xb5\x3f\x0a\xc5\x87\xec\xe8\x44\x6a\x12\xe0\x11"
    "\x3a\x6a\xb8\x63\x98\x5d\x6c\x09\x22\xfb\xf5\xfa\x40\x29\x59\x27"
    "\x2b\xd8\xaa\x91\xa7\x5f\x73\x8b\xde\x60\x98\xd5\x88\xb5\x8f\xa3"
    "\x7a\x2c\xcd\x0e\x3b\x24\x8f\xe0\x78\x24\xcb\xb6\x38\x1e\x69\x12"
    "\xa9\xf6\x8b\x9d\xc7\xb1\xa8\x2f\x1a\x95\xd0\xef\xec\xc2\x39\xdb"
    "\x9c\x52\x5a\x30\x9f\x9d\x85\xaa\x71\x5a\xcd\xae\x58\x64\x00\xfa"
    "\xf5\x02\x0d\x31\x25\x30\x23\x06\x09\x2a\x86\x48\x86\xf7\x0d\x01"
    "\x09\x15\x31\x16\x04\x14\x86\xfa\xbd\xf8\xd0\xe6\xd2\xbb\x3a\x68"
    "\x70\x24\x51\xdd\xa7\x1f\xa0\xb3\x28\x28\x30\x31\x30\x21\x30\x09"
    "\x06\x05\x2b\x0e\x03\x02\x1a\x05\x00\x04\x14\xb0\x26\xfb\x3f\x50"
    "\x6c\xdf\x38\xff\x54\x72\x58\x6c\x83\xfc\x18\xe9\xdc\xb0\xe4\x04"
    "\x08\xd1\x12\x5a\xd0\x0e\x52\x41\x72\x02\x02\x08\x00",
    2493);

const yxiva::apns::p8_token backup::p8{
    R"(-----BEGIN PRIVATE KEY-----
MIIEvQIBADANBgkqhkiG9w0BAQEFAASCBKcwggSjAgEAAoIBAQDNV70itpdrHOje
bqYJl3yTgILX3oUxYZOLzHyJWeaE0oPdyhEp6kHfsSXuIsR3eQkqZXfiiiYaApk1
BzcVexba6zr8aFwYAoUBoUn1rDx6Ud1aPTZXRH+Nr0he5WdmkrI8XlofVeCOUNP3
eZlMZ86l09SMuzatpNmdsezTE/rmzHhf2nPVrcqv1P6BkJLdQMiwMLUQxDVznzIK
He7ZIdI0L3Rgvl00gS0EPfEqa0gVuDkjVLR6HBhSJKPB2jhnC3PeaH49jEHsCx/U
I5cx8vK8S4q5uSNLRGXKvLHMn1iREa6ZFMjMMEfAUWv4Es+sXDEHaMbMUB8v1Com
9vN8avzzAgMBAAECggEAVwTwnQ5d+sGEmgawDG6LF4Mlb8/YCAY+3capnBdiDjWc
NlhUj7u7cM2Esnc/ZRYA4Vugw3siDHG/dW56VHsWQ23eUGr1ZCNlNDyiuB7mfjvF
H2HDbpYTq/p1I+n5uMlACT8+1/qvr//m5AA/C9cOBs5PS9hmoo2yvxZQs17LPBT1
g5pFfUZtb975h3mh+wqvn6h79JzKfC6RAIBL+gHjY8FlggJY6xPb0X0QBNJjcc66
/vwuVBHh6OXsdru+uWbGpkKPi2/4OeWjn6ncXJmjSU8f9JippkSnNVNcbB50dUR9
6b90ZMmCWuIRaLLC3gPOjr3AUOE2HCiqtiD657JlcQKBgQD5kdRhcR/F3RwKGB5Z
Fe5hRA5mcQN3qaeTigS9rIfhwlOvABUlNPSX6PA7Jn0KRpJRtRQqwSAEsyTnkq5l
F/aS5+A2QO+HkkthmJlqLVLOHU29C6NZ/WCAx+pQwg9uZRgUcUCO+hBD8iaGSMIu
7BGfcN6Eh9Z3xh3bzFH7+A58OQKBgQDSoi/V/CODJbt1TnSi8e27jFIhq0I/Xrhd
ytTzLIdhbZCni7VeQatAFJghDQ++smWUtvVy2fjzuvK+gqASygCuvG6VmU5yMlFB
ngQWYBwvZZuwY5xL2oLKyYUg+YBlbpA05fA1iqcdnsRP36HIl3N/rFiVswH0WUUf
4YJ+Ju3aiwKBgQDTN/LRTVagtVPHz0XQC9mmhFzToo5WtkVXfxYoGV8fWMJOd/lt
dfYUo1a+GQL2sWegs2qQDMc2cw+m1eO3N6K+DnUaNbcDvHMD9IzS6e0M+IwUlUb5
IJKOK6RKGRPKUNEia/9Bgq5FcynbuLRv1efMU1BgRj7jO8+FeNViNTy9MQKBgEI/
CFSAzJp+z4qBdmWZOW/3Hg3A00EjBVFabcddhea6kvhfFFfkx3uL/pzGguEzMoen
U2Mf8xCVB6urYbUHG1fZo94hLW71LGE61YZEDBlMF79OVOX2f5RP9TL9xcghEfHR
L89nIFBMkumcKEHygDzzc5kBh+fV+8ufHW+Xxs8pAoGAS2gpRlbwMyWwTAi1jAZP
bOmUw1BUTshGFcLivi8n6x4ZEGfNcbrpk3XSSXxMmlrC3z/9dM/w21v7H5LfIVMn
yTADfRpIQwloGscDaBFwFgBojh+cbX9+Zlpx7+eZXqpTC00kLlFDd7jlzw09sx9g
9C/3JHRZCFjZPCeGyjyQqpA=
-----END PRIVATE KEY-----
)",
    "123345",
    "ABCD4365",
    "mail.ya",
    "production"
};

const string fresh::pem =
    R"(-----BEGIN PRIVATE KEY-----
MIIEvQIBADANBgkqhkiG9w0BAQEFAASCBKcwggSjAgEAAoIBAQDcIr9xxJdYYTTB
N8PLw1WkldQmNf/K4wnAfALyEB4AkjMS6lYU2HfwzUOeFD+jFY7hUXXhD0FQvU2b
w4e9seF/Jj+Yx67KTdToRVGqqhv2Hu7WGyD4qyXqK9LXBPya6IJUGCKicS1R/871
wzDaxwneWcnF+hqQZcTrh+jE3AgQc4mm6uzLL5K/3jPzji6ip9hhU60y5r7qIyzA
bqvaV5C4Va8ItuR24ugxJ8Lj1LRwOJM1upuGbcGik3Mb/zN/724sAN6pRtKUWs1/
8g/WbBeiRHx8tmsd5uu1WCPlLXirKcYdVz+vbn7Xoq8EIXy31bYWvlG/tAhMNFjJ
mUYVOojLAgMBAAECggEAEiX/OIbnn06sRb2uM69P8Ti0+8154CUlmavfpksz3gIv
mcCeBivuf3i8uVyb/nr5g5PurvZOATUKH+xX1YjDuXLSzWqw+1a9VcTZPcO2eG0G
bbpg1n9tqcehIz7zAGZo0DwJJvFDtj3rtxE3huHf62flnZ/H7t2lBXH3+7rUI17r
9qRzsuUIoA3OKM/7o4Om7UdyPmb/L0Rg3f+3jXsqgc15jLTYMDBkBGEtFydotaye
n/IwiE3pU0tUPa3SJ0XQ76gsHHv7qRMhZrvOUDUN9J4Rny7HVPzAtNQ8cYMXeKSE
mMrFS0XdtSDMieo3CteappsnIo35RSX6uMUV+H64UQKBgQDullQ9bLBubVKjPJCF
RjY3mtI4qXYtPXOH+pyrkCck2FvIwrDcsB1LSLPCTW5OTBY6uBWd8k6M8zra3qGw
UWhFdlbvjbrsltx0X3wDr/2PTDqCwdZn/YbMhiFiRFz03OBvWqta5V2+wRcFpu10
Hs8E/69wM6qABuaUBNmbsjUaYwKBgQDsM62mEOYrzfPXSInvgkmeE/TBic6v/Gbc
JLJrUHTvW+DpuUh2sajwBnvMQ11bDM6b70Xr4c5L5FWhty3a4TQXD7roTQ4kY2uQ
DERXrMl/HMELcEc7MHskv3HT4b+l4qkt81ye/rK/SGnxKPQeV8yURbzgbHnttTTB
8ARJIh+weQKBgAopHsdK5KyfPUC0aJc0aUSu9DrzhJPipRPqaVepGPGpwSbom62a
VnUJUJZ6w7LT7LJteoYydoPsK9p8VrhyjxU+N++VwpsmxZazJk3l5c9/eJJu/NmK
4328TJmpvaez7F7xppCFahugj4FvHXevhDN7TBZkLOOs547wMfA+g6JvAoGAO+kz
v26D41J+6fk1O3HxmE35g+mfalX8BRStyLiEoH3Og5oCflC7x/3tAreHg2OxOC4t
rxH8UJWe/sDGucDrhHnx76Az2Rj6nHPwFcj25u31LSCSgLngUkbY6WRWRw5B2Szk
LlPuJMSKUqgPm26rrmQt70K0fo26zwJBJAeqwAkCgYEAnZ9EvNIh+J5yJqoRpavI
lx/kcFLveNfly3dSsL927plyxM08IbY7KHzmeDMGGgks5ZCClRRQXI+Ay/gisNkK
ACAOXPRoKfA2cqZ6CKelKs1zORwNDIv0Gc1Nb8trOjHv/hgurBvybwHrYhrcKxv8
hkxZ0rJvNkOg5Zegg9PLRYU=
-----END PRIVATE KEY-----
-----BEGIN CERTIFICATE-----
MIIDdDCCAlwCCQDpMABbclTIuTANBgkqhkiG9w0BAQsFADB8MQswCQYDVQQGEwJV
UzELMAkGA1UECAwCNTIxEDAOBgNVBAcMB05vd2hlcmUxDTALBgNVBAoMBG1haWwx
GTAXBgNVBAsMEHhpdmFzZXJ2ZXJfdXRlc3QxFTATBgoJkiaJk/IsZAEBDAV0b3Bp
YzENMAsGA1UEAwwEcm9vdDAeFw0yMDAyMDQxNTIxMTVaFw0zMDAyMDExNTIxMTVa
MHwxCzAJBgNVBAYTAlVTMQswCQYDVQQIDAI1MjEQMA4GA1UEBwwHTm93aGVyZTEN
MAsGA1UECgwEbWFpbDEZMBcGA1UECwwQeGl2YXNlcnZlcl91dGVzdDEVMBMGCgmS
JomT8ixkAQEMBXRvcGljMQ0wCwYDVQQDDARyb290MIIBIjANBgkqhkiG9w0BAQEF
AAOCAQ8AMIIBCgKCAQEA3CK/ccSXWGE0wTfDy8NVpJXUJjX/yuMJwHwC8hAeAJIz
EupWFNh38M1DnhQ/oxWO4VF14Q9BUL1Nm8OHvbHhfyY/mMeuyk3U6EVRqqob9h7u
1hsg+Ksl6ivS1wT8muiCVBgionEtUf/O9cMw2scJ3lnJxfoakGXE64foxNwIEHOJ
pursyy+Sv94z844uoqfYYVOtMua+6iMswG6r2leQuFWvCLbkduLoMSfC49S0cDiT
Nbqbhm3BopNzG/8zf+9uLADeqUbSlFrNf/IP1mwXokR8fLZrHebrtVgj5S14qynG
HVc/r25+16KvBCF8t9W2Fr5Rv7QITDRYyZlGFTqIywIDAQABMA0GCSqGSIb3DQEB
CwUAA4IBAQC1DvZ7qto0r87BT8gBK/D4448/g+JQ7oXEKrKDQZ0bVwpN7DuPkg4P
4C5S/vSHiRCzb+0IliMN7VNvjK9gNH7+RKMSY8x1sW05H7lQCOaleTrSiZJSYDMC
Lj/OcDnAXk06GVx8TqawCxSqTnm9RhC47UK3y9QjCZZ4tk8LkojtP/LKUGr7+CDW
uGtggDhW4vhMKetafRM/EVPrsXsWZcHdBdIJhPmrXEQjxT/IPqGfPRhnJDE7/sqF
6xSX1+gAP49sLhKrvOUPGYtwVOjI8Fqe1nBta3/3ilz7+L2Ntd+Im2DpaD9GWYnJ
1ZfWQh/YldFR4sNogiKKosiT6hU/qnMA
-----END CERTIFICATE-----
)";

const string fresh::p12(
    "\x30\x82\x09\xb9\x02\x01\x03\x30\x82\x09\x7f\x06\x09\x2a\x86\x48"
    "\x86\xf7\x0d\x01\x07\x01\xa0\x82\x09\x70\x04\x82\x09\x6c\x30\x82"
    "\x09\x68\x30\x82\x04\x1f\x06\x09\x2a\x86\x48\x86\xf7\x0d\x01\x07"
    "\x06\xa0\x82\x04\x10\x30\x82\x04\x0c\x02\x01\x00\x30\x82\x04\x05"
    "\x06\x09\x2a\x86\x48\x86\xf7\x0d\x01\x07\x01\x30\x1c\x06\x0a\x2a"
    "\x86\x48\x86\xf7\x0d\x01\x0c\x01\x06\x30\x0e\x04\x08\xfe\x2f\x7b"
    "\x74\x0a\x86\xe8\x71\x02\x02\x08\x00\x80\x82\x03\xd8\x0c\xb1\x54"
    "\x8f\x3d\xfa\x1c\x27\xf2\x20\xe4\x82\xec\xca\x5f\x01\x54\x12\x4c"
    "\x8d\x28\x4f\x55\x24\x22\x37\x23\x88\xaa\x62\xa8\x47\xdf\xc6\x73"
    "\x8a\xf2\xa5\xb6\x73\x7b\x0b\xe0\xc2\x04\xc9\x30\x7e\xfa\xe1\xed"
    "\x71\xce\x44\x4c\xc7\xef\x77\x51\x20\xeb\x0b\xeb\x2d\x77\x8e\x3b"
    "\xc9\x7b\xf2\x11\x16\x18\x50\x84\x05\xad\xeb\x0d\xc6\x51\x9b\x26"
    "\x72\x2c\xfc\xca\xe2\x94\x90\x96\x68\xfd\xc7\x5a\xac\x46\x15\x08"
    "\x16\xb4\x71\xb2\x72\xd6\x69\xd1\xea\xa0\x4c\x52\x45\x00\x31\x7f"
    "\x4b\x88\x64\xbc\x27\x8f\x36\x8d\x38\x1d\xee\xd8\xc0\x02\xd0\xfe"
    "\x1c\x6c\xc6\x2c\x50\x55\x8f\xd5\xa8\x78\x5e\x8a\x62\x98\xde\x2c"
    "\x09\xea\x91\x4d\x6a\xdf\x62\xac\x53\x48\x9d\x37\x84\xb5\x1d\xf2"
    "\xe5\xfb\xf6\xa9\xa4\x57\x20\x93\x72\x7b\xf4\x6c\x3c\xcb\x58\xe0"
    "\x1f\x64\x55\x1f\xf7\x72\xda\x62\x03\x4e\xa4\x60\x61\x96\xde\x10"
    "\xc7\x21\x51\xe4\x23\x3f\xd1\xe6\x64\x82\x0d\x83\x55\xd0\x3f\x90"
    "\xb2\x8c\x5f\x1c\xdf\xb3\x39\x65\x62\xd8\xb2\x97\xc4\xde\x42\x91"
    "\xfd\x7a\x20\x99\x95\x53\xb3\xed\x1d\xc4\x22\x84\x3f\x11\x04\x60"
    "\x9b\xd9\x64\xea\x9d\x1c\xac\xf8\x3f\x7d\x7c\x70\x36\x36\x2d\x26"
    "\x77\x1e\xfb\x9a\xbd\x5e\xc0\xa2\x1e\x50\x51\x40\x1d\xe2\x2c\x87"
    "\x3b\x7b\x26\xa5\x5f\x96\x17\x2b\x30\x5d\x66\xa0\xfd\x84\x8c\x20"
    "\xc5\x50\xfa\x4c\x66\xb6\xd2\x2f\x08\x28\xa0\xd0\x11\x89\x39\x41"
    "\x15\x3d\x1c\x91\xec\x97\xda\x2f\x88\x16\x3e\x03\x7b\x07\x80\x43"
    "\xf6\x73\x82\x98\xc3\x4e\x8b\x2f\x6b\x3d\xd2\xce\xfc\x20\x37\x48"
    "\xd3\x6c\xff\x22\xe5\x08\xb5\xb4\x68\x56\x1d\x7c\x57\x7d\x72\x35"
    "\x49\x36\xe9\xa1\xd0\x7a\x7b\xd1\xbb\x7e\xb4\x19\x2a\x3c\x25\x98"
    "\xc3\xcb\x06\xea\xd3\x94\x95\x86\x20\x3e\x33\x8c\xbc\x6e\x46\x2a"
    "\x59\x96\xeb\x0e\xdf\xd8\x97\xde\x12\xbe\x46\x74\x7b\x7e\xf3\x58"
    "\x65\xcd\x26\xa4\x12\x95\x7a\x91\xbd\x12\x6b\x1c\x40\xfc\x03\xb1"
    "\x2f\x2c\x75\xc2\x28\x21\x91\xfa\xe2\x74\x4f\xc6\x3b\xe5\x1e\x92"
    "\x16\x9a\xa8\xe0\xd4\xe7\xa3\x12\x0d\x02\x44\x64\xa8\xee\xc9\x04"
    "\x64\xb3\x6c\xed\x48\xd3\x72\x08\x18\x76\x7a\x97\x1a\xcc\x80\x8b"
    "\xc1\xeb\xc3\xd7\x8c\xd8\x51\xce\xbe\x3c\x6f\xde\xef\x8c\x8d\xad"
    "\x60\x99\x6d\xd5\xe7\x36\x12\xe1\x61\x67\xcc\xbc\x8e\xb9\xf2\x62"
    "\x1b\x3e\x8e\x7c\xfe\x2c\xc6\x8a\x7f\x64\xc5\x72\x10\xe7\x72\x25"
    "\xa4\xac\x39\xb2\x6c\xa8\x23\xaf\x15\xe6\x49\x53\xa7\xb3\xeb\x81"
    "\xc8\x39\xca\x3f\xf8\x86\x90\x28\xe5\xe7\x6d\x0f\xdc\xac\x19\x42"
    "\xff\x84\xdf\xc5\x57\xa5\x6b\x7b\x13\x12\x75\x73\xb3\x36\xc7\xaf"
    "\x2e\xfe\x0d\xea\x12\x65\xe9\xd9\xa8\x14\x08\xfd\xc8\x07\x5f\x90"
    "\xef\xe3\x87\x43\xdc\xf5\x26\x18\xe7\x47\x58\x39\x14\x64\x3f\xd5"
    "\x62\x00\x43\x85\x2d\x1f\xad\xa6\x33\xb3\x8c\x51\xdd\x41\xd2\x19"
    "\x22\x32\xdb\xb6\xc5\x9b\x0b\xff\x00\x03\x27\xdb\xe8\x04\xbd\xc4"
    "\x4b\x12\x3b\x3a\x05\x24\x26\x50\xf3\x99\xba\x08\xeb\xbc\xc7\x81"
    "\xb9\x08\x35\xec\xb9\x58\x4c\x6d\xcb\x85\x82\xb1\xd5\x5a\xaa\xcf"
    "\x3a\xa9\x57\xdf\xab\x3a\xfc\x4e\xa2\x29\xc2\xd3\xf1\xca\x02\x13"
    "\x3f\x72\xc4\x60\xb6\xc1\xff\x58\x87\x5a\xe8\x6f\x23\xdd\xe2\x19"
    "\xb7\xf9\x1f\xb1\x0d\x02\xce\x94\xdc\xdf\x0e\x01\x71\x16\x6a\x66"
    "\xe2\xa7\x93\xb6\x72\x67\xe4\x91\x9d\x1b\x38\x60\x94\xc5\xf9\x37"
    "\x90\xca\x7b\x54\x4b\x74\xe0\xcb\x55\x76\xa2\x01\x59\x3e\x9e\x54"
    "\xf8\x64\x51\x3e\xfb\x96\xab\x86\x2f\x48\x1e\x3b\x7c\x3a\x29\xc5"
    "\xde\x58\xd1\xa0\x4d\x70\xac\x5b\x94\xba\x8e\x7c\x76\x65\x35\xf3"
    "\x45\x4e\xbf\x26\x75\xd9\x35\xe0\xdd\x67\x45\x5f\x43\x64\x54\xd6"
    "\x59\xdd\x2a\xde\xd3\x27\x98\xea\xcc\xb3\x5b\x4d\x5b\xe9\xbe\x41"
    "\xcd\x82\x7f\xc0\x22\x4b\xe3\x0a\x91\x01\x95\xa0\xa3\x9b\x8e\x17"
    "\xe2\xf2\xb1\x24\x34\xa5\xfe\x07\x50\xc5\x6b\x5d\xb7\x76\xb4\xb4"
    "\xdc\x24\xc0\x07\xa9\xe8\x61\x1d\x79\xab\x02\x02\x9b\x03\xb5\xf9"
    "\x03\xbe\x2e\x28\xdb\xeb\x8e\x4e\xa6\x15\xdb\xfb\x0e\xbe\x2e\xd5"
    "\x98\x10\xd5\x52\x97\xe5\xac\x50\x82\xd6\x97\x25\xa1\xe7\xc8\x94"
    "\xfa\x30\x70\x6d\x2d\xa5\x58\x64\xd8\x03\x27\x48\x4b\x8e\x6c\x70"
    "\x75\x6e\x1d\x4f\x30\xc4\x6e\xe9\x7d\xe5\x5a\x90\xf7\x57\x8b\x44"
    "\xf9\xdc\xbd\x49\xfb\xe9\xb0\xb2\x12\x9c\x76\x89\x2c\x87\x0c\xd5"
    "\x9e\x31\x26\x14\x8d\x58\x33\x40\xed\x2d\xa0\x06\x6d\x86\x05\x3f"
    "\x7a\x5a\x72\xff\x83\x6e\x0e\x77\x1d\x68\x23\xd3\x9a\x7f\xfb\xe6"
    "\xde\xe4\xd9\x0e\x30\xb9\xbc\x60\x24\x8f\xce\xd3\x4f\x52\x77\x4d"
    "\x15\x94\x85\x51\x7c\x30\x82\x05\x41\x06\x09\x2a\x86\x48\x86\xf7"
    "\x0d\x01\x07\x01\xa0\x82\x05\x32\x04\x82\x05\x2e\x30\x82\x05\x2a"
    "\x30\x82\x05\x26\x06\x0b\x2a\x86\x48\x86\xf7\x0d\x01\x0c\x0a\x01"
    "\x02\xa0\x82\x04\xee\x30\x82\x04\xea\x30\x1c\x06\x0a\x2a\x86\x48"
    "\x86\xf7\x0d\x01\x0c\x01\x03\x30\x0e\x04\x08\xc8\x9a\x9f\x4d\x5e"
    "\x82\xb9\x74\x02\x02\x08\x00\x04\x82\x04\xc8\x2f\x97\x3a\x1a\xd4"
    "\xde\xe8\xdc\xa9\x0c\xcc\xfe\xff\xdb\xb8\xf3\x65\x32\x08\x5b\x9b"
    "\xd0\x23\xfb\xff\x30\xae\x08\xaf\x90\x15\x7f\x9d\x6e\xfb\x07\x8d"
    "\x34\x2d\xa7\x6c\xa0\x99\xc3\xa5\xcb\x18\x34\xe5\xd4\xd5\xe9\xe4"
    "\xd6\x2c\x17\xc5\x11\x01\x60\x78\xae\x56\xee\xb1\xa6\x6f\x23\x8a"
    "\x03\xfe\x0e\x7d\xe3\x96\xd4\xcd\x4c\xc5\x55\xaa\x06\xde\xd5\x35"
    "\x5e\xc7\x6c\x68\xbb\xff\x6a\x68\xd7\xba\x83\x88\x67\x39\x53\x99"
    "\xe3\xdf\x74\x31\x6c\x57\xe3\x96\x25\x87\x7d\xe2\x0d\x0b\xf0\x8e"
    "\xb1\x3c\x45\x08\x74\x98\x52\x37\x4d\x7d\xd7\x20\x2d\xb9\x04\x72"
    "\x8f\x76\x71\x43\xc7\x19\x68\xde\xcb\x12\xfd\xd2\x45\xfd\x1f\x54"
    "\x5a\x52\x6d\xf3\xcc\x8d\xd8\x1c\x19\x84\x59\x11\xe0\x24\x7f\xbf"
    "\x13\xad\x41\xe5\xa3\x5a\x86\xae\x48\x0c\x50\x0d\x78\xdf\xbd\x8d"
    "\xd1\x29\x65\x37\xd8\x4b\x30\xac\x31\xac\xd9\xad\xbb\x9b\x8e\x3c"
    "\x55\x2a\xe0\xd8\x25\x22\x92\x60\xbd\x23\x8c\xc7\x12\xcc\xb8\xd6"
    "\x53\x45\x8f\x72\x38\x1c\x3e\xe3\x90\x08\x7b\xd1\xe0\xd7\xb6\xbd"
    "\xaa\xdd\xe8\xb7\x50\x0e\x69\x38\xb6\xb4\x03\x7b\xff\xfa\x1c\x3f"
    "\xb2\x6c\x8c\x25\xe4\xd2\x7d\xfc\x89\x68\xa4\x89\x84\xa1\x86\xee"
    "\xb7\x47\x04\x44\xde\x05\x06\xed\x7b\xe0\x31\xf8\xbc\x52\xb7\x1c"
    "\xfa\xef\x8a\x78\xd2\x9e\xe2\x9c\x7d\x9e\x62\x5c\x47\xa1\x97\xc9"
    "\x76\x6c\x70\xe4\x41\xee\x05\xea\x4f\x01\x46\x1b\x6c\x9b\x94\x54"
    "\x9e\xa5\xd5\xd0\x68\xdc\x98\x3b\xce\x4b\xa3\x7f\x38\x53\x8b\xdd"
    "\xf5\xe7\x94\xda\xa1\x62\xf4\xfa\x2d\x1a\x4c\x20\x1b\x05\x5e\x5b"
    "\x56\x77\xfe\xc2\xe3\xe3\xe9\x02\xd4\x24\xb7\x55\xfc\x45\xf1\xfc"
    "\x87\x47\xe7\x6f\x92\x62\x7a\x95\x4a\xb7\x5e\xd2\x5d\x5c\xee\x1f"
    "\xd1\xa2\xcd\xb5\x88\x46\x28\x0d\xe1\x2b\x60\x9b\xd9\x58\xe1\xb5"
    "\xe6\x18\x61\x4f\x70\x05\x24\x71\xd1\x6c\xfa\x47\x73\xb9\xfb\xd5"
    "\x12\x63\x79\xaa\x6c\xc5\x15\xcd\x97\x74\x23\x42\x47\x24\x6d\x28"
    "\x87\x4d\x72\x5e\x88\x26\xba\xc2\x78\xed\xdb\xe4\xc4\xf1\xa7\x22"
    "\x76\x7d\x0f\x50\x3d\x89\x16\xa9\x73\x8e\x2f\xdd\xbf\x6d\x68\xda"
    "\x46\x3f\x89\x1f\xbb\x2e\x1f\x83\xb3\x67\xef\x63\x63\xa6\xd1\xe8"
    "\xca\xe4\x6d\xac\xa4\xff\xda\x0e\xca\x6f\xde\xcf\x7a\xe1\xe4\xa1"
    "\xc7\xd2\x70\x07\x54\x6d\x9f\xce\x17\x1c\x2a\x26\xb5\x8c\x09\x58"
    "\x47\x73\xe5\xfc\x74\xe1\x9e\xc0\x0f\x3f\xe4\x75\x14\x93\x6e\x2b"
    "\x91\x85\x5e\x07\x15\x16\x68\x09\xc7\x26\x61\xeb\xa7\x23\x51\x58"
    "\x29\x7d\xa0\xc7\x5c\x2f\xa1\x27\x33\xe0\x9e\x39\x99\x82\x5f\x35"
    "\x9e\xca\x84\x70\xdc\x19\x7e\x7a\xaa\x53\xb0\xdc\x32\x59\x0f\x55"
    "\xec\x88\x5d\xd3\x84\xdc\x0c\x7c\x1e\xd3\x3a\x1f\x50\x83\xc4\xf8"
    "\x5e\x90\x93\x86\x1e\xd6\xda\x9a\xe4\x36\x1f\xdf\x7d\x40\x18\x38"
    "\xd8\x71\xb0\xf8\xfb\x97\xfa\x39\x5b\xb0\x53\x96\x42\xff\x02\x6f"
    "\x0c\xbf\x16\x49\xdc\x38\x4f\xb4\x4a\x4d\x76\x94\x32\x58\xd5\x8f"
    "\x38\xf9\x93\x75\x65\xc5\xd6\x06\x3a\x54\x44\x41\xc3\x70\x40\x8e"
    "\x27\x3b\x15\xe4\x1a\xdd\xd8\xb5\xcc\xaf\x40\x8e\x49\xcd\x97\xb3"
    "\x8d\x66\xc6\x90\xff\xd2\x03\x89\xa8\x41\x42\x4a\xae\x6d\xdd\xef"
    "\x52\x97\x6c\xec\xca\x3d\x69\x44\xe2\x9d\x91\x76\xcd\xc5\x40\x22"
    "\xc1\x5a\x33\xed\xe5\xf5\x6d\x1b\x0a\x2f\x40\x47\xa8\xe1\xbb\xe3"
    "\xb4\x20\x42\xbc\x48\x32\xa2\x31\x45\x05\x16\x98\xf7\x88\x27\xd1"
    "\x58\xe2\xb6\x30\x79\xb9\x43\x6a\x04\x34\x63\x39\xf2\x03\x03\x72"
    "\xa4\x4f\x81\xbd\x6f\x25\x6e\x4f\x42\xbc\xa5\x25\x7d\x3a\x7e\x1e"
    "\xc3\x2a\xf1\x64\x0c\xf7\x2d\x8b\x94\x1e\x23\xbb\x8e\x98\x21\x25"
    "\x94\xd1\x8e\x11\xd8\x2c\x8c\x6a\xcd\x19\xd7\x91\x18\xa1\xcd\xb8"
    "\xae\x20\x95\x9e\xcd\x79\x58\x7d\x28\x12\x59\x94\x21\xa7\x32\xa3"
    "\xd8\x4c\x24\x75\x32\x4b\x73\xef\xf7\x2d\x67\x8a\xe4\x93\xb6\xc3"
    "\xde\x5c\xdf\x1b\x2b\xd9\xf0\x7a\x52\x7b\x53\xdf\x74\x9d\x2e\x76"
    "\x88\xbb\x7d\x16\x3f\x05\x1c\xd9\xcd\x80\xef\x79\x0c\xd8\x72\x1e"
    "\x12\xfe\x13\xf2\xaf\x48\x42\x17\x18\xa6\x80\x02\x53\x20\xc3\xc3"
    "\x50\xd9\xa7\x47\x26\x70\xe7\x60\xe2\x9f\x85\x2c\x21\xee\x30\xde"
    "\x2f\x82\x70\xb7\x08\x02\x3b\x2e\xe8\xec\x4a\xbd\xbb\x79\xf8\x7b"
    "\x30\x4b\x23\x40\x61\xb7\x52\x4d\x5e\x82\x98\xd7\xd4\xf7\x7a\x8b"
    "\x3f\xc7\x12\x66\x24\x31\xb7\xeb\x8c\x4e\xcf\xef\xe3\x5a\xbd\x19"
    "\xbd\x6a\x4a\xe0\x09\xeb\x08\x2a\x0a\xf3\x11\x5f\x93\x83\xdc\x89"
    "\xfe\x54\x81\xdf\x0c\x76\xf1\x33\x9a\x6b\xb8\x28\xed\x56\xca\x3a"
    "\x00\xbf\xbd\x3c\x74\x83\x07\x58\xe5\x61\xf2\xe0\x0e\xb5\x68\x44"
    "\x63\x15\xb9\x54\x44\xf0\xe0\xdf\x25\x19\x36\x35\xe1\x31\xb9\xbf"
    "\xb5\x70\x6d\x73\x70\xe7\x54\x15\x0e\xbe\x68\xe9\xd2\x8e\xbf\x85"
    "\xf1\xad\xe0\xb0\x48\x0f\x28\x08\x5d\x4f\x9e\x64\x59\x9d\xce\xd4"
    "\x9d\x21\x7d\x34\x96\xeb\x10\x81\x04\xd9\x49\x68\x89\x07\x97\x55"
    "\x39\xaa\x24\xf3\x70\x1e\xb7\xe6\xa9\xf4\x1f\xcd\x24\xbc\xdc\x33"
    "\x9b\x8e\x0e\xa0\xa5\x6d\x59\xc6\xdf\x1e\xc9\x10\x0d\xf4\x2c\xb8"
    "\x4b\x70\x2f\xbf\x21\x8f\x9d\x67\x0c\x1e\x09\xcb\x61\x7d\xba\x12"
    "\x4e\x8c\x54\x16\x77\x7f\x3b\x01\x0b\x15\x4d\x05\x5e\x3f\x85\x16"
    "\x94\x9c\x57\xf3\xe4\xd6\x4b\x20\xe0\x76\x07\xa2\x67\x72\x62\xc6"
    "\x3f\x88\xc4\x1c\xf0\x6c\xdf\x76\xb1\x88\x24\x0a\x9c\xae\x3b\x0c"
    "\x61\x6d\x5f\xcb\x53\xba\x0d\xed\xfa\x31\x89\xdd\xcb\x1d\xa6\x3b"
    "\x1b\x14\x59\x8f\x8e\x59\x78\x32\xff\xdd\x5b\xbe\x0f\xe5\x04\x8f"
    "\x3b\xbf\x81\xce\x8f\x09\x45\x73\x30\x21\xef\x82\x68\xb5\xb1\x04"
    "\xcc\xca\x8f\xd1\xce\x34\x89\x82\xeb\xec\x0f\x6f\xb8\x7d\x95\x28"
    "\x5b\x63\x71\x74\xfd\x30\xd1\xe8\xfd\xfc\x51\x21\xe4\x4e\x01\x8e"
    "\x60\x70\xce\x31\x25\x30\x23\x06\x09\x2a\x86\x48\x86\xf7\x0d\x01"
    "\x09\x15\x31\x16\x04\x14\x1b\x52\x26\x9f\x45\xb2\xb1\x3c\x80\x05"
    "\x00\x3a\x2b\xf9\x51\x29\x69\x42\x48\x8c\x30\x31\x30\x21\x30\x09"
    "\x06\x05\x2b\x0e\x03\x02\x1a\x05\x00\x04\x14\x04\xf5\x4d\x8e\xf8"
    "\xa6\xfb\xce\x66\x52\xaf\xc0\xb5\xe4\x98\xb3\x1e\xf1\xd4\xe0\x04"
    "\x08\xe6\x11\xd6\x25\x41\x84\x48\x92\x02\x02\x08\x00",
    2493);

const yxiva::apns::p8_token fresh::p8{
    R"(-----BEGIN PRIVATE KEY-----
MIIEvQIBADANBgkqhkiG9w0BAQEFAASCBKcwggSjAgEAAoIBAQDcIr9xxJdYYTTB
N8PLw1WkldQmNf/K4wnAfALyEB4AkjMS6lYU2HfwzUOeFD+jFY7hUXXhD0FQvU2b
w4e9seF/Jj+Yx67KTdToRVGqqhv2Hu7WGyD4qyXqK9LXBPya6IJUGCKicS1R/871
wzDaxwneWcnF+hqQZcTrh+jE3AgQc4mm6uzLL5K/3jPzji6ip9hhU60y5r7qIyzA
bqvaV5C4Va8ItuR24ugxJ8Lj1LRwOJM1upuGbcGik3Mb/zN/724sAN6pRtKUWs1/
8g/WbBeiRHx8tmsd5uu1WCPlLXirKcYdVz+vbn7Xoq8EIXy31bYWvlG/tAhMNFjJ
mUYVOojLAgMBAAECggEAEiX/OIbnn06sRb2uM69P8Ti0+8154CUlmavfpksz3gIv
mcCeBivuf3i8uVyb/nr5g5PurvZOATUKH+xX1YjDuXLSzWqw+1a9VcTZPcO2eG0G
bbpg1n9tqcehIz7zAGZo0DwJJvFDtj3rtxE3huHf62flnZ/H7t2lBXH3+7rUI17r
9qRzsuUIoA3OKM/7o4Om7UdyPmb/L0Rg3f+3jXsqgc15jLTYMDBkBGEtFydotaye
n/IwiE3pU0tUPa3SJ0XQ76gsHHv7qRMhZrvOUDUN9J4Rny7HVPzAtNQ8cYMXeKSE
mMrFS0XdtSDMieo3CteappsnIo35RSX6uMUV+H64UQKBgQDullQ9bLBubVKjPJCF
RjY3mtI4qXYtPXOH+pyrkCck2FvIwrDcsB1LSLPCTW5OTBY6uBWd8k6M8zra3qGw
UWhFdlbvjbrsltx0X3wDr/2PTDqCwdZn/YbMhiFiRFz03OBvWqta5V2+wRcFpu10
Hs8E/69wM6qABuaUBNmbsjUaYwKBgQDsM62mEOYrzfPXSInvgkmeE/TBic6v/Gbc
JLJrUHTvW+DpuUh2sajwBnvMQ11bDM6b70Xr4c5L5FWhty3a4TQXD7roTQ4kY2uQ
DERXrMl/HMELcEc7MHskv3HT4b+l4qkt81ye/rK/SGnxKPQeV8yURbzgbHnttTTB
8ARJIh+weQKBgAopHsdK5KyfPUC0aJc0aUSu9DrzhJPipRPqaVepGPGpwSbom62a
VnUJUJZ6w7LT7LJteoYydoPsK9p8VrhyjxU+N++VwpsmxZazJk3l5c9/eJJu/NmK
4328TJmpvaez7F7xppCFahugj4FvHXevhDN7TBZkLOOs547wMfA+g6JvAoGAO+kz
v26D41J+6fk1O3HxmE35g+mfalX8BRStyLiEoH3Og5oCflC7x/3tAreHg2OxOC4t
rxH8UJWe/sDGucDrhHnx76Az2Rj6nHPwFcj25u31LSCSgLngUkbY6WRWRw5B2Szk
LlPuJMSKUqgPm26rrmQt70K0fo26zwJBJAeqwAkCgYEAnZ9EvNIh+J5yJqoRpavI
lx/kcFLveNfly3dSsL927plyxM08IbY7KHzmeDMGGgks5ZCClRRQXI+Ay/gisNkK
ACAOXPRoKfA2cqZ6CKelKs1zORwNDIv0Gc1Nb8trOjHv/hgurBvybwHrYhrcKxv8
hkxZ0rJvNkOg5Zegg9PLRYU=
-----END PRIVATE KEY-----
)",
    "123",
    "ABCD",
    "mail",
    "production"
};

}
