#pragma once

#include <ymod_xconf/conf_list.h>
#include <ymod_pq/response_handler.h>
#include <ymod_pq/unescape_bytea.h>
#include <string>

namespace ymod_xconf {

struct list_handler : public ymod_pq::response_handler
{
    enum COLUMNS
    {
        REVISION = 0,
        TYPE,
        CONFIGURATION,
        TOKEN,
        NAME,
        OWNER,
        ENVIRONMENT,

        COUNT
    };

    conf_list_ptr configurations;
    bool exception_occured;

    list_handler(revision_t rev)
    {
        configurations = std::make_shared<conf_list>();
        configurations->max_revision = rev;
    }

    void handle_cell(unsigned /*row*/, unsigned col, const string& value, bool is_null)
    {
        if (is_null) return;
        try
        {
            switch (col)
            {
            case REVISION:
            {
                revision_t revision = boost::lexical_cast<revision_t>(value);
                if (configurations->max_revision < revision)
                {
                    configurations->max_revision = revision;
                }
                configurations->items.back().revision = revision;
            }
            break;
            case TYPE:
                if (auto res = resolve_type(value, configurations->items.back().type))
                {
                }
                else
                {
                    throw std::runtime_error(res.error_reason);
                }
                break;
            case CONFIGURATION:
                configurations->items.back().configuration = ymod_pq::unescape_bytea(value);
                break;
            case TOKEN:
                configurations->items.back().token = value;
                break;
            case NAME:
                configurations->items.back().name = value;
                break;
            case OWNER:
                configurations->items.back().owner = value;
                break;
            case ENVIRONMENT:
                configurations->items.back().environment = value;
                break;
            default:
                break;
            }
        }
        catch (const boost::bad_lexical_cast& ex)
        {
            YLOG_G(error) << "bad cast error: column=" << col << " exception=\"" << ex.what()
                          << "\""
                          << " value=\"" << value << "\"";
            exception_occured = true;
        }
        catch (const std::exception& ex)
        {
            YLOG_G(error) << "error on select: column=" << col << " exception=\"" << ex.what()
                          << "\""
                          << " value=\"" << value << "\"";
            exception_occured = true;
        }
    }

    unsigned column_count() const
    {
        return COLUMNS::COUNT;
    }

    void handle_row_begin(unsigned /*row*/)
    {
        configurations->items.emplace_back();
        exception_occured = false;
    }

    void handle_row_end(unsigned /*row*/)
    {
        if (exception_occured)
        {
            configurations->items.pop_back();
        }
        else if (
            configurations->items.back().type == config_type::SERVICE ||
            configurations->items.back().type == config_type::MOBILE)
        {
            // Prevent stored value to leak into app.
            configurations->items.back().environment.clear();
        }
    }
};

} // conf
