#pragma once

#include <boost/system/error_code.hpp>
#include <initializer_list>

namespace ymod_xconf {
using boost::system::error_code;

namespace error {
namespace detail {

enum code : int
{
    success = 0,
    token_violation,
    bad_owner_id,
    transport_error,
    invalid_arg,
    forbidden_environment,
    stale_revision,

    COUNT
    // DO NOT FORGET TO EXTEND error_names
};

class error_category : public boost::system::error_category
{
public:
    const char* name() const noexcept
    {
        return "xconf";
    }

    string message(int ev) const
    {
        return error_names[ev];
    }

    static const error_category& instance()
    {
        static constexpr std::initializer_list<const char*> names = {
            "success",          "token violation",       "mismatched owner", "transport error",
            "invalid argument", "forbidden environment", "stale revision"
        };
        static_assert(
            names.size() == code::COUNT,
            "Error codes count doesn't correspond with error names count");
        static const error_category category(names);
        return category;
    }

private:
    error_category(std::initializer_list<const char*> names)
        : error_names(names.begin(), names.end())
    {
    }

    const std::vector<string> error_names;
};

inline boost::system::error_code make_error(code value)
{
    return boost::system::error_code(value, error_category::instance());
}

} // detail

static const error_code success = make_error(detail::code::success);
static const error_code token_violation = make_error(detail::code::token_violation);
static const error_code bad_owner_id = make_error(detail::code::bad_owner_id);
static const error_code transport_error = make_error(detail::code::transport_error);
static const error_code invalid_arg = make_error(detail::code::invalid_arg);
static const error_code forbidden_environment = make_error(detail::code::forbidden_environment);
static const error_code stale_revision = make_error(detail::code::stale_revision);

} // error
} // conf
