#pragma once

#include <ymod_xconf/xconf.h>
#include <yxiva/core/http_handlers.h>
#include <ymod_httpclient/cluster_client.h>
#include <yplatform/util/safe_call.h>
#include <string>

namespace ymod_xconf {

class http_client : public client
{
public:
    http_client(std::shared_ptr<yhttp::cluster_client> adaptor) : http_(adaptor)
    {
    }

protected:
    static const error_code error_from_response_body(const string& body)
    {
        if (body == error::token_violation.message()) return error::token_violation;
        if (body == error::bad_owner_id.message()) return error::bad_owner_id;
        if (body == error::stale_revision.message()) return error::stale_revision;
        return error::transport_error;
    }

    void put_impl(
        config_type type,
        const string& environment,
        const string& name,
        const string& owner,
        const string& token,
        revision_t revision,
        string_ptr value,
        const put_handler_t& handler,
        task_context_ptr ctx) override
    {
        if (name.empty() || owner.empty() || !value || type == config_type::ANY)
        {
            yplatform::safe_call(
                ctx, "xconf handler", handler, error::invalid_arg, INITIAL_REVISION);
            return;
        }
        auto url = "/put" +
            yhttp::url_encode({ { "name", name },
                                { "type", get_type_name(type) },
                                { "owner", owner },
                                { "token", token },
                                { "environment", environment },
                                { "revision", revision } });
        http_->async_run(
            ctx,
            yhttp::request::POST(url, std::move(*value)),
            [handler, ctx](auto ec, auto response) {
                if (ec)
                {
                    yplatform::safe_call(
                        ctx, "xconf handler", handler, error::transport_error, INITIAL_REVISION);
                }
                else if (response.status != 200)
                {
                    yplatform::safe_call(
                        ctx,
                        "xconf handler",
                        handler,
                        error_from_response_body(response.body),
                        INITIAL_REVISION);
                }
                else
                {
                    try
                    {
                        auto revision = boost::lexical_cast<revision_t>(response.body);
                        yplatform::safe_call(
                            ctx, "xconf handler", handler, error::success, revision);
                    }
                    catch (const std::exception& ex)
                    {
                        YLOG_CTX_GLOBAL(ctx, error) << "xconf put failed: " << ex.what();
                        yplatform::safe_call(
                            ctx,
                            "xconf handler",
                            handler,
                            error::transport_error,
                            INITIAL_REVISION);
                    }
                }
            });
    }

    void list_impl(
        config_type type,
        const string& environment,
        revision_t revision,
        const conf_list_handler_t& handler,
        task_context_ptr ctx) override
    {
        auto url = "/list" +
            yhttp::url_encode({ { "type", get_type_name(type) },
                                { "revision", revision },
                                { "environment", environment } });
        http_->async_run(
            ctx, yhttp::request::GET(url), [handler, revision, ctx](auto ec, auto response) {
                if (ec)
                {
                    yplatform::safe_call(
                        ctx, "xconf handler", handler, error::transport_error, nullptr);
                }
                else if (response.status != 200)
                {
                    yplatform::safe_call(
                        ctx,
                        "xconf handler",
                        handler,
                        error_from_response_body(response.body),
                        nullptr);
                }
                else
                {
                    try
                    {
                        auto configurations = std::make_shared<conf_list>();
                        configurations->deserialize_msgpack(response.body);
                        if (configurations->items.empty())
                        {
                            configurations->max_revision = revision;
                        }
                        yplatform::safe_call(
                            ctx, "xconf handler", handler, error::success, configurations);
                    }
                    catch (const std::exception& ex)
                    {
                        YLOG_CTX_GLOBAL(ctx, error) << "xconf list failed: " << ex.what();
                        yplatform::safe_call(
                            ctx, "xconf handler", handler, error::transport_error, nullptr);
                    }
                }
            });
    }

    std::shared_ptr<yhttp::cluster_client> http_;
};

} // conf
