#pragma once

#include <ymod_xconf/dbhandlers/put_handler.h>
#include <ymod_xconf/dbhandlers/list_handler.h>
#include <ymod_xconf/xconf.h>
#include <ymod_pq/call.h>
#include <string>

namespace ymod_xconf {

class pg_client : public client
{
public:
    pg_client(boost::shared_ptr<ymod_pq::call> pg_adaptor, const string& env)
        : pg_adaptor_(pg_adaptor), allowed_environment_(env)
    {
    }

protected:
    void put_impl(
        config_type type,
        const string& environment,
        const string& name,
        const string& owner,
        const string& token,
        revision_t revision,
        string_ptr value,
        const put_handler_t& handler,
        task_context_ptr ctx) override
    {
        static const string ALL_ENVIRONMENTS = "any";

        if (type == config_type::ANY)
        {
            yplatform::safe_call("xconf handler", handler, error::invalid_arg, INITIAL_REVISION);
            return;
        }
        if (environment_specific(type))
        {
            if (environment.empty())
            {
                yplatform::safe_call(
                    "xconf handler", handler, error::invalid_arg, INITIAL_REVISION);
                return;
            }
            else if (!allowed(environment))
            {
                yplatform::safe_call(
                    "xconf handler", handler, error::forbidden_environment, INITIAL_REVISION);
                return;
            }
        }
        else if (environment.size())
        {
            yplatform::safe_call("xconf handler", handler, error::invalid_arg, INITIAL_REVISION);
            return;
        }

        ymod_pq::bind_array_ptr args(new ymod_pq::bind_array);
        ymod_pq::push_const_string(args, get_type_name(type));
        ymod_pq::push_const_string(args, name);
        ymod_pq::push_const_string(args, owner);
        ymod_pq::push_cptr_byte_array(args, value->data(), value->size());
        if (token.empty())
        {
            ymod_pq::push_null(args, ymod_pq::bind_array::STRING);
        }
        else
        {
            ymod_pq::push_const_string(args, token);
        }
        ymod_pq::push_const_string(args, environment.size() ? environment : ALL_ENVIRONMENTS);
        if (revision > INITIAL_REVISION)
        {
            ymod_pq::push_const_string(args, std::to_string(revision));
        }
        else
        {
            ymod_pq::push_null(args, ymod_pq::bind_array::STRING);
        }

        auto dbhandler = boost::make_shared<put_handler>();
        auto res = pg_adaptor_->request(ctx, "", "put", args, dbhandler, true);
        res.add_callback([res, dbhandler, handler, value]() {
            try
            {
                res.get();
                yplatform::safe_call(
                    "xconf handler", handler, dbhandler->errc, dbhandler->revision);
            }
            catch (const std::exception& ex)
            {
                YLOG_G(error) << "pg_client::on_put error: exception=\"" << ex.what() << "\"";
                yplatform::safe_call(
                    "xconf handler", handler, error::transport_error, INITIAL_REVISION);
            }
        });
    }

    void list_impl(
        config_type type,
        const string& env,
        revision_t revision,
        const conf_list_handler_t& handler,
        task_context_ptr ctx) override
    {
        static const string STR_TRUE = "t";
        static const string STR_FALSE = "f";
        static const config_type SOME_TYPE = static_cast<config_type>(0);

        // If all possible envs are requested, return all that are allowed.
        auto& environment = env.empty() ? allowed_environment_ : env;
        // Specifically requesting disallowed envs is an error.
        if (!allowed(environment))
        {
            yplatform::safe_call("xconf handler", handler, error::forbidden_environment, nullptr);
            return;
        }

        bool any_type = (type == config_type::ANY);
        bool any_env = environment.empty();

        ymod_pq::bind_array_ptr args(new ymod_pq::bind_array);
        ymod_pq::push_const_double(args, static_cast<double>(revision));
        ymod_pq::push_const_string(args, any_type ? STR_TRUE : STR_FALSE);
        // Hack to use valid enum value ('any' is not valid for db).
        ymod_pq::push_const_string(args, get_type_name(any_type ? SOME_TYPE : type));
        ymod_pq::push_const_string(args, any_env ? STR_TRUE : STR_FALSE);
        ymod_pq::push_const_string(args, environment);

        auto dbhandler = boost::make_shared<list_handler>(revision);
        auto res = pg_adaptor_->request(ctx, "", "list", args, dbhandler, true);
        res.add_callback([res, dbhandler, handler]() {
            try
            {
                res.get();
                yplatform::safe_call(
                    "xconf handler", handler, error::success, dbhandler->configurations);
            }
            catch (const std::exception& ex)
            {
                YLOG_G(error) << "pg_client::on_list error: exception=\"" << ex.what() << "\"";
                yplatform::safe_call("xconf handler", handler, error::transport_error, nullptr);
            }
        });
    }

    bool environment_specific(config_type type)
    {
        return type == config_type::SEND_TOKEN || type == config_type::LISTEN_TOKEN;
    }

    bool allowed(const string& env)
    {
        return allowed_environment_.empty() || env == allowed_environment_;
    }

    boost::shared_ptr<ymod_pq::call> pg_adaptor_;
    string allowed_environment_;
};

} // conf
