#pragma once

#include <ymod_xconf/conf_list.h>
#include <yxiva/core/packing.hpp>
#include <mutex>
#include <map>
#include <string>

namespace ymod_xconf {

class fs_conf_dumper
{
    struct key_t
    {
        config_type type;
        string environment;
        string name;

        bool operator<(const key_t& k) const
        {
            return std::tie(type, environment, name) < std::tie(k.type, k.environment, k.name);
        }

        MSGPACK_DEFINE(type, environment, name);
    };
    using store_t = std::map<key_t, item>;

public:
    fs_conf_dumper(const string& path) : path_(path), stopped_(true)
    {
        if (path_.size() && path_.back() != '/') path_.append("/");
        path_.append("cached_config");
    }

    void start()
    {
        scoped_lock lock(guard_);
        stopped_ = false;

        std::fstream f(path_);
        if (f)
        {
            string packed_confs;
            f.seekg(0, std::ios::end);
            packed_confs.reserve(f.tellg());
            f.seekg(0, std::ios::beg);
            packed_confs.assign(
                std::istreambuf_iterator<char>(f), std::istreambuf_iterator<char>());
            f.close();

            try
            {
                yxiva::unpack(packed_confs, cached_configs_);
            }
            catch (const std::exception& ex)
            {
                YLOG_L(error) << "loading from dump error: exception=\"" << ex.what() << "\"";
            }
        }
    }

    conf_list_ptr get(config_type type, const string& environment)
    {
        scoped_lock lock(guard_);
        auto configs = std::make_shared<conf_list>();
        if (stopped_) return configs;

        auto begin = type == config_type::ANY ?
            cached_configs_.begin() :
            cached_configs_.lower_bound({ type, string{}, string{} });
        auto end = type == config_type::ANY ?
            cached_configs_.end() :
            cached_configs_.lower_bound({ inc(type), string{}, string{} });

        for (auto it = begin; it != end; ++it)
        {
            // Return items with matching environment along with environment-indifferent items.
            // TODO Do it smarter
            if (environment.size() && it->first.environment.size() &&
                it->first.environment != environment)
            {
                continue;
            }
            configs->items.push_back(it->second);
            if (it->second.revision > configs->max_revision)
                configs->max_revision = it->second.revision;
        }
        return configs;
    }

    void dump(conf_list_ptr confs)
    {
        scoped_lock lock(guard_);
        if (stopped_) return;

        for (auto& conf : confs->items)
        {
            cached_configs_[{ conf.type, conf.environment, conf.name }] = conf;
        }

        try
        {
            std::fstream f(path_, std::ofstream::out | std::ofstream::trunc);
            string packed = yxiva::pack(cached_configs_);
            f << packed;
            f.flush();
            f.close();
        }
        catch (const std::exception& ex)
        {
            YLOG_L(error) << "error while dumping to file: exception=\"" << ex.what() << "\"";
        }
    }

    void stop()
    {
        scoped_lock lock(guard_);
        stopped_ = true;
    }

private:
    config_type inc(config_type type)
    {
        return static_cast<config_type>(
            static_cast<typename std::underlying_type<config_type>::type>(type) + 1);
    }

private:
    string path_;
    store_t cached_configs_;
    bool stopped_;
    mutex guard_;
};

} // conf
