#include "local_conf_storage_impl.h"

#include <ymod_xconf/http_client.h>
#include <ymod_xconf/fake_client.h>
#include <yplatform/find.h>
#include <boost/filesystem.hpp>

namespace ymod_xconf {

local_conf_storage_impl::local_conf_storage_impl(const yplatform::ptree& module_conf)
{
    reactor_ =
        yplatform::find<yplatform::reactor, std::shared_ptr>(module_conf.get("reactor", "global"));
    load_setting(module_conf);

    if (not settings_.condb_http_api_host.empty())
    {
        auto cluster_client = std::make_shared<yhttp::cluster_client>(*reactor_, settings_.http);
        confdb_ = std::make_shared<http_client>(cluster_client);
    }
    else
    {
        confdb_ = std::make_shared<fake_client>();
        YLOG_L(error) << "not active: missing xconf api host";
    }
    if (settings_.dump_to_filesystem)
        confs_dumper_ = std::make_shared<fs_conf_dumper>(settings_.dump_dir_path);
}

void local_conf_storage_impl::start()
{
    if (settings_.poll_interval == time_duration::max())
    {
        YLOG_L(debug) << "polling disabled (poll_interval not set)";
        return;
    }

    if (confs_dumper_)
    {
        confs_dumper_->start();
    }

    for (auto& handlers_info : handlers_by_conf_props_)
    {
        auto conf_props = handlers_info.first;

        revision_t restored_revision = 0;
        if (confs_dumper_)
        {
            auto dumped_conf_list = confs_dumper_->get(conf_props.first, conf_props.second);
            if (!dumped_conf_list->items.empty())
                call_handlers(conf_props.first, conf_props.second, dumped_conf_list);
            restored_revision = dumped_conf_list->max_revision;
        }

        auto poller = find_or_create_poller(conf_props.first, conf_props.second);
        poller->start(
            restored_revision, [this, conf_props](const error_code& errc, conf_list_ptr confs) {
                if (errc)
                {
                    YLOG_L(error) << "error while polling configurations: error=\""
                                  << errc.message() << "\"";
                }
                else
                {
                    if (!confs->items.empty())
                    {
                        call_handlers(conf_props.first, conf_props.second, confs);
                        if (confs_dumper_) confs_dumper_->dump(confs);
                    }
                }
            });
    }
}

void local_conf_storage_impl::stop()
{
    for (auto poller : pollers_)
        poller.second->stop();
    pollers_.clear();
    if (confs_dumper_) confs_dumper_->stop();
}

void local_conf_storage_impl::subscribe_updates_impl(
    config_type type,
    const string& environment,
    no_error_conf_list_handler_t&& handler)
{
    handlers_by_conf_props_[{ type, environment }].push_back(std::move(handler));
}

void local_conf_storage_impl::put_impl(
    config_type type,
    const string& environment,
    const string& name,
    const string& owner,
    const string& token,
    revision_t revision,
    string_ptr value,
    const put_handler_t& handler,
    task_context_ptr ctx)
{
    if (environment.size())
    {
        confdb_->put(
            type,
            resolve_environment(environment),
            name,
            owner,
            token,
            revision,
            value,
            handler,
            ctx);
    }
    else
    {
        confdb_->put(type, name, owner, token, revision, value, handler, ctx);
    }
}

void local_conf_storage_impl::list_impl(
    config_type type,
    const string& environment,
    revision_t revision,
    const conf_list_handler_t& handler,
    task_context_ptr ctx)
{
    // @todo add dirty read: try use dumper cache
    if (environment.size())
    {
        confdb_->list(type, resolve_environment(environment), revision, handler, ctx);
    }
    else
    {
        confdb_->list(type, revision, handler, ctx);
    }
}

timer_based_poller_ptr local_conf_storage_impl::find_or_create_poller(
    config_type type,
    const string& environment)
{
    auto ipoller = pollers_.find({ type, environment });
    if (ipoller == pollers_.end())
    {
        auto poller = std::make_shared<timer_based_poller>(
            *reactor_->io(), confdb_, settings_.poll_interval, type, environment);
        pollers_[{ type, environment }] = poller;
        return poller;
    }
    return ipoller->second;
}

void local_conf_storage_impl::load_setting(const yplatform::ptree& conf)
{
    settings_.poll_interval = conf.get("poll_interval", time_duration::max());
    settings_.condb_http_api_host = conf.get<string>("host", "");
    settings_.dump_to_filesystem = conf.get<bool>("dump_to_filesystem", true);
    settings_.dump_dir_path = conf.get<string>("dump_dir", "");
    if (auto http_node = conf.get_child_optional("http"))
    {
        settings_.http.parse_ptree(*http_node);
    }
    settings_.http.nodes = { settings_.condb_http_api_host };

    if (settings_.dump_to_filesystem)
    {
        if (settings_.dump_dir_path.empty())
        {
            throw std::runtime_error("Dump directory path not set for xconf poller");
        }
        else
        {
            boost::filesystem::path dir(settings_.dump_dir_path);
            if (!boost::filesystem::exists(dir))
            {
                boost::filesystem::create_directories(dir);
            }
        }
    }
}

void local_conf_storage_impl::call_handlers(
    config_type type,
    const string& environment,
    conf_list_ptr confs)
{
    for (auto& handler : handlers_by_conf_props_[{ type, environment }])
    {
        try
        {
            handler(confs);
        }
        catch (const std::exception& ex)
        {
            YLOG_L(error) << "error while executing handler: exception=\"" << ex.what() << "\"";
        }
        catch (...)
        {
            YLOG_L(error) << "error while executing handler: exception=\"unknown\"";
        }
    }
}

} // conf

#include <yplatform/module_registration.h>
DEFINE_SERVICE_OBJECT(ymod_xconf::local_conf_storage_impl)
