#pragma once

#include "poller_task_context.h"
#include <ymod_xconf/xconf.h>
#include <ymod_xconf/types.h>
#include <yxiva/core/types.h>

namespace ymod_xconf {

class timer_based_poller : public std::enable_shared_from_this<timer_based_poller>
{
public:
    timer_based_poller(
        boost::asio::io_service& io,
        xconf_ptr confdb,
        time_duration poll_interval,
        config_type configuration_type,
        const string& environment)
        : timer_(io)
        , poll_interval_(poll_interval)
        , confdb_(confdb)
        , revision_(0)
        , listen_type_(configuration_type)
        , listen_env_(environment)
        , ctx_(new poller_task_context())
    {
    }

    ~timer_based_poller()
    {
        stop();
    }

    void start(revision_t start_rev, conf_list_handler_t handler)
    {
        revision_ = start_rev;
        handler_ = std::move(handler);
        make_poll_request();
    }

    void stop()
    {
        timer_.cancel();
    }

private:
    void make_poll_request()
    {
        namespace p = std::placeholders;
        if (listen_env_.empty())
        {
            confdb_->list(
                listen_type_,
                revision_,
                std::bind(&timer_based_poller::handle_poll, this->shared_from_this(), p::_1, p::_2),
                ctx_);
        }
        else
        {
            confdb_->list(
                listen_type_,
                resolve_environment(listen_env_),
                revision_,
                std::bind(&timer_based_poller::handle_poll, this->shared_from_this(), p::_1, p::_2),
                ctx_);
        }
    }

    void handle_poll(const error_code& errc, conf_list_ptr configurations)
    {
        yplatform::safe_call("xconf handler", handler_, errc, configurations);
        if (!errc) revision_ = std::max(revision_, configurations->max_revision);

        std::weak_ptr<timer_based_poller> wptr = this->shared_from_this();
        timer_.expires_from_now(poll_interval_);
        timer_.async_wait([wptr, this](const boost::system::error_code& err) {
            if (!err)
                if (auto ptr = wptr.lock()) make_poll_request();
        });
    }

private:
    steady_timer timer_;
    time_duration poll_interval_;
    xconf_ptr confdb_;
    conf_list_handler_t handler_;
    revision_t revision_;
    config_type listen_type_;
    string listen_env_;
    task_context_ptr ctx_;
};

typedef std::shared_ptr<timer_based_poller> timer_based_poller_ptr;

} // conf
