#pragma once


#include <ymod_xstore/types.h>
#include <yxiva/core/iabstract.h>
#include <ymod_pq/response_handler.h>
#include <boost/lexical_cast.hpp>
#include <libpq-fe.h>
#include <string>

namespace ymod_xstore
{
namespace pq_handler
{

inline string unescape_string(const string& value)
{
  size_t len = 0;
  auto bytes = PQunescapeBytea(reinterpret_cast<const unsigned char*> (value.data()), &len);
  if (!bytes) {
    throw std::runtime_error("PQunescapeBytea failed");
  }
  try {
    string result(reinterpret_cast<const char*> (bytes), len);
    PQfreemem(bytes);
    return result;
  }
  catch (...) {
    PQfreemem(bytes);
    throw;
  }
}

class notifications_list
  : public ymod_pq::response_handler, public yxiva::iabstract
{
private:
  entry current_;
  entries_list_ptr list_;

public:

  enum
  {
    uid,
    service,
    local_id,
    event_ts,
    tags,
    ttl,
    content,
    is_deleted,
    is_sticked,
    delayed_until_ts,

    column_cnt
  };

  notifications_list()
    : list_(new entries_list())
  {
  }

  void handle_cell(unsigned /*row*/, unsigned col, const string& value,
      bool is_null)
  {
    switch (col) {
      case uid:
        current_.uid = value;
        break;
      case service:
        current_.service = value;
        break;
      case local_id:
        current_.local_id = is_null || value.empty() ? 0 : boost::lexical_cast<local_id_t>(value);
        break;
      case event_ts:
        current_.event_ts = is_null || value.empty() ? 0 : boost::lexical_cast<std::time_t>(value);
        break;
      case tags:
        // ignore
        break;
      case ttl:
        current_.ttl = boost::lexical_cast<ttl_t>(value);
        break;
      case content:
        current_.content = unescape_string(value);
        break;
      case is_deleted:
        current_.is_deleted = (value == "t");
        break;
      case is_sticked:
        current_.is_sticked = (value == "t");
        break;
      case delayed_until_ts:
        current_.delayed_until_ts = is_null || value.empty() ? 0 : boost::lexical_cast<std::time_t>(value);
        break;
      default:
        YLOG_L(error) << "[pq_handlers::notifications_list] unknown column " << col;
    }
  }

  void handle_row_begin(unsigned /*row*/)
  {
    current_ = entry();
  }

  void handle_row_end(unsigned /*row*/)
  {
    list_->push_back(current_);
  }

  unsigned column_count() const
  {
    return column_cnt;
  }

  const entries_list_ptr& data() const
  {
    return list_;
  }
};

}
}
