#include "xstore_postgre.h"

#include "pq_handlers/ignore_response.h"
#include "pq_handlers/notifications_list.h"
#include "pq_handlers/get_counters.h"
#include "pq_handlers/get_range_counters.h"
#include "pq_handlers/insert.h"
#include "pq_handlers/lift_messages.h"
#include <yxiva/core/gid.h>
#include <yplatform/task_context.h>
#include <yplatform/hash/sha1.h>

using yplatform::task_context;
using yplatform::task_context_ptr;

#define SAFE_CALL(where, expr) \
  try { \
    (expr); \
  } catch (const std::exception & e) { \
    YLOG_CTX_GLOBAL(ctx, error) << where << " callback exception=" << e.what(); \
  } catch (...) { \
    YLOG_CTX_GLOBAL(ctx, error) << where << " callback exception=unknown"; \
  }

namespace ymod_xstore
{
namespace
{

// holds an entry while pg call is not finished
template<typename Handler, typename Callback>
void handle_write(task_context_ptr ctx, const std::shared_ptr<entry>& /*entry_holder*/,
    ymod_pq::future_result fres, Handler handler, const Callback& cb)
{
  try {
    fres.get();
    SAFE_CALL("handle_write", cb(error(), handler->local_id(), handler->event_ts(),
        handler->merge(), handler->transit_id()));
  }
  catch (const std::exception & exc) {
    SAFE_CALL("handle_write", cb(make_error(err_code_storage_fail, exc.what()), 0, 0, merge_type::none, ""));
  }
}

template<typename Handler, typename Callback, typename DataType>
void handle_read(task_context_ptr ctx, ymod_pq::future_result fres,
    Handler handler, const Callback& cb)
{
  try {
    fres.get();
    SAFE_CALL("handle_read", cb(error(), handler->data()));
  }
  catch (const std::exception & exc) {
    auto err = make_error(err_code_storage_fail, exc.what());
    SAFE_CALL("handle_read", cb(err, DataType()));
  }
}
}

void xstore_postgre::init(const yplatform::ptree& config)
{
  reload(config);
}

void xstore_postgre::reload(const yplatform::ptree& config)
{
  auto postgre_module_name = config.get("postgre_module", "mod_postgre_xstore");
  pq_caller_ = yplatform::find<ymod_pq::cluster_call> (postgre_module_name);
  auto shards_name = config.get("shards", "shards_xstore");
  shards_ = yplatform::find<yxiva::shard_config::storage, std::shared_ptr> (shards_name);
}

void xstore_postgre::start()
{
}

void xstore_postgre::stop()
{
}

void xstore_postgre::fini()
{
  pq_caller_.reset();
}

void xstore_postgre::write(task_context_ptr ctx, entry&& store_entry,
    const string& hash, const insert_callback_t& cb)
{
  try {
    auto entry_ptr = std::make_shared<entry>(std::move(store_entry));
    auto args = boost::make_shared<ymod_pq::bind_array>();
    ymod_pq::push_const_string(args, static_cast<string> (entry_ptr->uid));
    ymod_pq::push_const_string(args, entry_ptr->service);
    ymod_pq::push_const_time_t(args, entry_ptr->event_ts);
    ymod_pq::push_const_svector(args, entry_ptr->tags);
    ymod_pq::push_null(args, ymod_pq::bind_array::STRING); // empty 'extra data'
    ymod_pq::push_const_uint(args, entry_ptr->ttl);
    // doesn't make a copy, keep entry until executing the request
    ymod_pq::push_cptr_byte_array(args, entry_ptr->content.data(), int(entry_ptr->content.size()));
    ymod_pq::push_const_string(args, hash);
    ymod_pq::push_const_string(args, entry_ptr->transit_id);
    if (entry_ptr->topic.size()) {
      ymod_pq::push_const_string(args, entry_ptr->topic);
    } else {
      ymod_pq::push_null(args, ymod_pq::bind_array::STRING);
    }

    auto handler = boost::make_shared<pq_handler::insert>();
    auto shards = shards_->get();
    auto master = master_from_uid(*shards, entry_ptr->uid);
    if (!master) {
      cb(make_error(err_code_storage_fail,
          "failed to get shard conninfo for " + string(entry_ptr->uid)),
          0, 0, merge_type::none, "");
      return;
    }

    auto fres = pq_caller_->request(ctx, master->conninfo, "add_notification",
        args, handler, true, req_deadline, ymod_pq::request_target::master);
    fres.add_callback(boost::bind(handle_write<decltype(handler), decltype(cb)>,
        ctx, entry_ptr, fres, handler, cb));
  }
  catch (std::exception& e) {
    cb(make_error(err_code_call_fail, e.what()), 0, 0, merge_type::none, "");
  }
  catch (...) {
    cb(make_error(err_code_call_fail, "..."), 0, 0, merge_type::none, "");
  }
}

void xstore_postgre::read(task_context_ptr ctx, const user_id & uid,
    const string & service, const local_id_t local_id_from,
    const unsigned limit, const time_t begin_ts, const time_t end_ts,
    const entries_callback_t& cb)
{
  read_single_service_impl(ctx, uid, service, local_id_from, nullptr, limit, begin_ts, end_ts, cb);
}

void xstore_postgre::read(task_context_ptr ctx, const user_id & uid,
    const string & service, const local_id_t local_id_from,
    const local_id_t local_id_to, const unsigned limit, const time_t begin_ts,
    const time_t end_ts, const entries_callback_t& cb)
{
  read_single_service_impl(ctx, uid, service, local_id_from, &local_id_to, limit, begin_ts, end_ts, cb);
}

void xstore_postgre::read(task_context_ptr ctx, const user_id & uid,
    const std::vector<string> & services, const unsigned last_hours,
    const unsigned limit, const entries_callback_t& cb)
{
  try {
    auto args = boost::make_shared<ymod_pq::bind_array>();
    ymod_pq::push_const_string(args, static_cast<string> (uid));
    ymod_pq::push_const_svector(args, services);
    ymod_pq::push_const_uint(args, last_hours);
    if (limit) {
      ymod_pq::push_const_uint(args, limit);
    } else {
      ymod_pq::push_null(args, ymod_pq::bind_array::INT);
    }

    auto handler = boost::make_shared<pq_handler::notifications_list>();
    auto shards = shards_->get();
    auto master = master_from_uid(*shards, uid);
    if (!master) {
      cb(make_error(err_code_storage_fail,
          "failed to get shard conninfo for " + string(uid)),
          entries_list_ptr());
      return;
    }

    auto fres = pq_caller_->request(ctx, master->conninfo, "list_notifications_multi_service",
        args, handler, true, req_deadline, ymod_pq::request_target::master);
    fres.add_callback(boost::bind(handle_read<decltype(handler), decltype(cb), entries_list_ptr>, ctx,
        fres, handler, cb));
  }
  catch (std::exception& e) {
    cb(make_error(err_code_call_fail, e.what()), entries_list_ptr());
  }
  catch (...) {
    cb(make_error(err_code_call_fail, "..."), entries_list_ptr());
  }
}

void xstore_postgre::get_counters(task_context_ptr ctx, const user_id& uid,
    const std::vector<string> & services, const counters_callback_t& cb)
{
  try {
    auto args = boost::make_shared<ymod_pq::bind_array>();
    ymod_pq::push_const_string(args, static_cast<string> (uid));
    ymod_pq::push_const_svector(args, services);

    auto handler = boost::make_shared<pq_handler::get_counters>();
    auto shards = shards_->get();
    auto master = master_from_uid(*shards, uid);
    if (!master) {
      cb(make_error(err_code_storage_fail,
          "failed to get shard conninfo for " + string(uid)),
          counter_pack());
      return;
    }

    auto fres = pq_caller_->request(ctx, master->conninfo, "get_counters", args,
        handler, true, req_deadline, ymod_pq::request_target::master);
    fres.add_callback(boost::bind(handle_read<decltype(handler), decltype(cb), counter_pack>,
        ctx, fres, handler, cb));
  }
  catch (std::exception& e) {
    cb(make_error(err_code_call_fail, e.what()), counter_pack());
  }
  catch (...) {
    cb(make_error(err_code_call_fail, "..."), counter_pack());
  }
}

void xstore_postgre::read_single_service_impl(task_context_ptr ctx, const user_id & uid, const string & service,
    const local_id_t local_id_from, const local_id_t* local_id_to,
    const unsigned limit, const time_t begin_ts, const time_t end_ts,
    const entries_callback_t& cb)
{
  time_t now = time(nullptr);
  try {
    auto args = boost::make_shared<ymod_pq::bind_array>();
    ymod_pq::push_const_string(args, static_cast<string> (uid));
    ymod_pq::push_const_string(args, service);
    ymod_pq::push_const_string(args, std::to_string(local_id_from));
    if (local_id_to) {
      ymod_pq::push_const_string(args, std::to_string(*local_id_to));
    } else {
      ymod_pq::push_null(args, ymod_pq::bind_array::INT);
    }
    if (limit) {
      ymod_pq::push_const_uint(args, limit);
    } else {
      ymod_pq::push_null(args, ymod_pq::bind_array::INT);
    }
    ymod_pq::push_const_time_t(args, begin_ts > 0 ? begin_ts : now - 24 * 60 * 60);
    ymod_pq::push_const_time_t(args, end_ts > 0 ? end_ts : now + 60 * 60);

    auto handler = boost::make_shared<pq_handler::notifications_list>();
    auto shards = shards_->get();
    auto master = master_from_uid(*shards, uid);
    if (!master) {
      cb(make_error(err_code_storage_fail,
          "failed to get shard conninfo for " + string(uid)),
          entries_list_ptr());
      return;
    }
    auto fres = pq_caller_->request(ctx, master->conninfo, "list_notifications_by_local_id",
        args, handler, true, req_deadline, ymod_pq::request_target::master);
    fres.add_callback(boost::bind(handle_read<decltype(handler), decltype(cb), entries_list_ptr>, ctx,
        fres, handler, cb));
  }
  catch (std::exception& e) {
    cb(make_error(err_code_call_fail, e.what()), entries_list_ptr());
  }
  catch (...) {
    cb(make_error(err_code_call_fail, "..."), entries_list_ptr());
  }
}

void xstore_postgre::get_range_counters(task_context_ptr ctx,
    const user_id& uid, const std::vector<string> & services, time_t begin_ts,
    local_id_t local_id_from, size_t count, const range_counters_list_callback_t& cb)
{
  try {
    auto args = boost::make_shared<ymod_pq::bind_array>();
    ymod_pq::push_const_string(args, static_cast<string> (uid));
    ymod_pq::push_const_svector(args, services);
    ymod_pq::push_const_time_t(args, begin_ts);
    ymod_pq::push_const_string(args, std::to_string(local_id_from));
    ymod_pq::push_const_string(args, std::to_string(count));

    auto handler = boost::make_shared<pq_handler::get_range_counters>();
    auto shards = shards_->get();
    auto master = master_from_uid(*shards, uid);
    if (!master) {
      cb(make_error(err_code_storage_fail,
          "failed to get shard conninfo for " + string(uid)),
          range_counters_list());
      return;
    }

    auto fres = pq_caller_->request(ctx, master->conninfo, "get_range_counters", args,
        handler, true, req_deadline, ymod_pq::request_target::master);
    fres.add_callback(boost::bind(handle_read<decltype(handler), decltype(cb), range_counters_list>,
        ctx, fres, handler, cb));
  }
  catch (std::exception& e) {
    cb(make_error(err_code_call_fail, e.what()), range_counters_list());
  }
  catch (...) {
    cb(make_error(err_code_call_fail, "..."), range_counters_list());
  }
}

void xstore_postgre::lift_messages(task_context_ptr ctx, const string& uid,
    const string& service, local_id_t local_id_from, size_t count,
    const lift_messages_callback_t& cb)
{
  try {
    auto args = boost::make_shared<ymod_pq::bind_array>();
    ymod_pq::push_const_string(args, uid);
    ymod_pq::push_const_string(args, service);
    ymod_pq::push_const_string(args, std::to_string(local_id_from));
    ymod_pq::push_const_string(args, std::to_string(count));

    auto handler = boost::make_shared<pq_handler::lift_messages>();
    auto shards = shards_->get();
    auto master = master_from_uid(*shards, uid);
    if (!master) {
      cb(make_error(err_code_storage_fail,
          "failed to get shard conninfo for " + string(uid)), 0);
      return;
    }

    auto fres = pq_caller_->request(ctx, master->conninfo, "lift_messages", args,
        handler, true, req_deadline, ymod_pq::request_target::master);
    fres.add_callback(boost::bind(handle_read<decltype(handler), decltype(cb), unsigned long>,
        ctx, fres, handler, cb));
  }
  catch (std::exception& e) {
    cb(make_error(err_code_call_fail, e.what()), 0);
  }
  catch (...) {
    cb(make_error(err_code_call_fail, "..."), 0);
  }
}

std::shared_ptr<yxiva::shard_config::storage> xstore_postgre::shards()
{
  return shards_;
}

bool xstore_postgre::in_fallback()
{
  auto pq_health = pq_caller_->get_health();
  return std::any_of(pq_health.begin(), pq_health.end(),
    [](auto& db_cluster) {return db_cluster.second.master.in_fallback;});
}

void xstore_postgre::enable_fallback(task_context_ptr ctx)
{
  auto shards = shards_->get();
  for (auto& shard : *shards) {
      pq_caller_->enable_fallback(ctx, shard.master.conninfo, ymod_pq::request_target::master);
  }
}

void xstore_postgre::disable_fallback(task_context_ptr ctx)
{
  auto shards = shards_->get();
  for (auto& shard : *shards) {
      pq_caller_->disable_fallback(ctx, shard.master.conninfo, ymod_pq::request_target::master);
  }
}

}
