#pragma once

#include "operations.h"
#include <ymod_xtasks/types.h>
#include <ymod_xtasks/task.h>
#include <ymod_xtasks/results.h>
#include <msgpack.hpp>
#include <unordered_map>
#include <unordered_set>
#include <deque>
#include <map>
#include <vector>

namespace ymod_xtasks {

struct pending_part_t
{
    pending_part_t()
      : notification(false), subscription(false)
    {
    }
    bool notification; // unused
    bool subscription; // unused

    MSGPACK_DEFINE(notification, subscription)
};

struct active_part_t
{
    active_part_t()
      : worker(), start_time(0)
    {
    }
    string worker;
    time_t start_time;

    MSGPACK_DEFINE(worker, start_time)
};

struct delay_part_t
{
    delay_part_t()
      : delay_time(0), delay_sec(0), flags(delay_flags::none)
    {
    }
    time_t delay_time;
    time_t delay_sec;
    delay_flags_t flags;

    MSGPACK_DEFINE(delay_time, delay_sec, flags)
};

struct task_pack
{
    task_pack(const ymod_xtasks::task& task, delay_flags_t delay_flags)
      : task_(task), pending_(false),
        active_(false), delayed_(false),
        delay_flags_(delay_flags)
    {
    }

    task_pack()
      : task_(), pending_(false),
        active_(false), delayed_(false),
        delay_flags_(delay_flags::none)
    {
    }

    void update_local_id(local_id_t local_id)
    {
      if (local_id != 0 && task_.local_id > local_id) {
        task_.local_id = local_id;
      }
    }

    const ::ymod_xtasks::task& task() const
    {
        return task_;
    }

    void set_pending()
    {
        pending_ = true;
    }

    void reset_pending()
    {
        pending_ = false;
    }

    bool is_pending() const
    {
        return pending_;
    }

    bool wakeup_on_create() const
    {
        return (delay_flags_ & delay_flags::wakeup_on_create)
            || (delay_part_.flags & delay_flags::wakeup_on_create);
    }

    bool ignore_delay_if_pending() const
    {
        return delay_flags_ & delay_flags::ignore_if_pending
            || (delay_part_.flags & delay_flags::ignore_if_pending);
    }

    void update_delay_flags(delay_flags_t v)
    {
        delay_flags_ |= v;
    }

    void activate(const string& worker, time_t start_time)
    {
        reset_pending();
        delay_flags_ = delay_flags::none;
        active_ = true;
        active_part_.worker = worker;
        active_part_.start_time = start_time;
    }

    void cleanup_active()
    {
        assert(active_);
        set_pending();
        reset_active();
    }

    void reset_active()
    {
        assert(active_);
        active_= false;
        active_part_ = active_part_t();
    }

    bool is_active() const
    {
        return active_;
    }

    time_t active_start_time() const
    {
        assert(active_);
        return active_part_.start_time;
    }

    const string& worker() const
    {
        assert(active_);
        return active_part_.worker;
    }

    void delay(time_t delay_time, time_t delay_sec, delay_flags_t flags)
    {
        assert(active_);
        assert(!delayed_);
        reset_active();
        delayed_ = true;
        delay_part_.delay_time = delay_time;
        delay_part_.delay_sec = delay_sec;
        delay_part_.flags = flags;
    }

    void reset_delay()
    {
        assert(delayed_);
        delayed_ = false;
        delay_part_ = delay_part_t();
    }

    void wakeup_delayed()
    {
        assert(delayed_);
        set_pending();
        reset_delay();
    }

    bool is_delayed() const
    {
        return delayed_;
    }

    time_t delay_start_time() const
    {
        assert(delayed_);
        return delay_part_.delay_time;
    }

    time_t delay_sec() const
    {
        assert(delayed_);
        return delay_part_.delay_sec;
    }

    ymod_xtasks::task task_;
    bool pending_;
    bool active_;
    bool delayed_;

    pending_part_t pending_part_; // unused
    active_part_t active_part_;
    delay_part_t delay_part_;

    delay_flags_t delay_flags_;

    MSGPACK_DEFINE(task_, pending_, active_, delayed_, pending_part_, active_part_, delay_part_, delay_flags_)
};

#if 0
inline bool operator<(const delayed_info& info, const task_id_t& id) {
    return info.task.id < id;
};
inline bool operator<(const task_id_t& id, const delayed_info& info) {
    return id < info.task.id;
};
inline bool operator<(const active_info& info, const task_id_t& id) {
    return info.task.id < id;
};
inline bool operator<(const task_id_t& id, const active_info& info) {
    return id < info.task.id;
};
inline bool operator<(const worker_info& info, const string& name) {
    return info.worker < name;
};
inline bool operator<(const string& name, const worker_info& info) {
    return name < info.worker;
};
#endif


struct data
{
    data() : rev(-1) {}
    typedef std::deque<string> pending_queue_t;
    typedef std::unordered_map<string, task_pack> tasks_t;
    typedef std::unordered_map<string, worker_info> workers_t;

    iid_t rev;
    pending_queue_t pending_queue;
    tasks_t tasks;
    workers_t workers;

    void reset_tasks_and_workers()
    {
      pending_queue.clear();
      tasks.clear();
      workers.clear();
    }

    MSGPACK_DEFINE(rev, pending_queue, tasks, workers);
};

typedef std::shared_ptr<data> data_ptr;

struct data_index
{
    std::unordered_set<string> active;
    std::unordered_set<string> delayed;

    // TODO priority_queue for delayed - ?

    void clear() {
        active.clear();
        delayed.clear();
    }
};

typedef std::shared_ptr<data_index> data_index_ptr;

void build_index(data_ptr data, data_index_ptr index);

//TODO move to data structure
struct domain_settings
{
  domain_settings() : max_tasks_count(1000000000) {}

  size_t max_tasks_count;

  void parse(const yplatform::ptree& conf)
  {
      max_tasks_count = conf.get("max_tasks_count", max_tasks_count);
  }
};

}
