#pragma once

#include "data.h"
#include "operations.h"
#include "results.h"
#include <ymod_xtasks/error.h>
#include <algorithm>

namespace ymod_xtasks {

class data_processor
{
public:
    data_processor(data_ptr data, data_index_ptr index, const domain_settings& settings)
      : data_(data), index_(index), settings_(settings)
    {
    }

    proc_result<create_task_result> apply(const create_task_args& draft)
    {
        proc_result<create_task_result> result;

        if (data_->tasks.size() >= settings_.max_tasks_count) {
          result.error = make_lite_error(err_code_tasks_limit, "");
          return result;
        }

        // optimization: not delayed and nothing to send - ignore
        if (draft.hint == "new_subscription") {
            bool need_diff = (draft.local_id != 0);
            if (!need_diff) {
                result.data.action = "ignore";
                return result;
            }
        }

        auto id = make_id(draft);

        auto itask = data_->tasks.find(id);
        if (exists(itask)) {
            itask->second.update_local_id(draft.local_id);
            itask->second.update_delay_flags(draft.delay_flags);
            if (is_delayed(itask)) {
                if (itask->second.wakeup_on_create()) {
                    move_delayed_to_pending(itask);
                }
                result.data.action = "attach";
                return result;
            }
        } else {
            itask = create_new_task_pack(
                create_task_from_draft(id, draft), draft.delay_flags);
        }

        insert_pending_queue(itask);

        itask->second.set_pending();
        result.data.action = "insert";

        return result;
    }

    proc_result<get_tasks_result> apply(const get_tasks_args& args)
    {
        const string& worker = args.worker;
        const size_t count = args.count;
        const time_t start_time = args.ts;

        proc_result<get_tasks_result> result;

        // TODO check limits for active tasks or workers - not here
        // TODO IDEA check worker's productivity - not here

        auto qcur = data_->pending_queue.begin();
        while (qcur != data_->pending_queue.end() && result.data.tasks.size() < count) {
            auto id = *qcur;

            auto itask = data_->tasks.find(id);
            if (exists(itask) && !is_pending(itask)) {
                qcur = drop_pending_queue_item(qcur);
                continue;
            }

            if (exists(itask) && !is_active(itask)) {
                result.data.tasks.push_back(itask->second.task());
                activate(itask, worker, start_time);
                qcur = drop_pending_queue_item(qcur);
                continue;
            }

            ++qcur;
        }

        if (result.data.tasks.size())
            worker_retained_tasks(worker, result.data.tasks.size(), start_time);

        return result;
    }

    proc_result<fin_task_result> apply(const fin_task_args& args)
    {
        const string& worker = args.worker;
        const task_id_t id = args.task_id;
        const time_t time = args.ts;

        proc_result<fin_task_result> result;

        auto itask = data_->tasks.find(id);
        if (exists(itask) && is_active(itask)) {
            if (itask->second.worker() == worker) {
                deactivate(itask);
                worker_released_tasks(worker, 1, time);
                result.data.success = true;
                try_to_erase(itask);
            } else {
                result.error = make_lite_error(err_code_worker_mismatch,
                    "worker=" + worker + ";real-worker=" + itask->second.worker() + ";id=" + id);
            }
        } else {
            result.error = make_lite_error(err_code_no_such_task, id);
        }

        return result;
    }

    proc_result<delay_task_result> apply(const delay_task_args& args)
    {
        const string& worker = args.worker;
        const task_id_t id = args.task_id;
        const time_t delay_sec = args.delay_sec;
        const time_t time = args.ts;
        const delay_flags_t flags = args.flags;

        proc_result<delay_task_result> result;

        auto itask = data_->tasks.find(id);
        if (exists(itask) && is_active(itask)) {
            if (itask->second.worker() == worker) {
                if (is_pending(itask)) {
                  if (itask->second.ignore_delay_if_pending()
                        || (flags & delay_flags::ignore_if_pending)) {
                    // Do not delay, instead deactivate task,
                    // it remains pending.
                    deactivate(itask);
                  } else {
                    // We do want to delay this task,
                    // reset it's pending status so
                    // it will not become active prematurely.
                    itask->second.reset_pending();
                    delay(itask, time, delay_sec, flags);
                  }
                } else {
                    delay(itask, time, delay_sec, flags);
                }
                worker_released_tasks(worker, 1, time);
                result.data.success = true;
            } else {
                result.error = make_lite_error(err_code_worker_mismatch,
                    "worker=" + worker + ";real-worker=" + itask->second.worker() + ";id=" + id);
            }
        } else {
            result.error = make_lite_error(err_code_no_such_task, id);
        }

        return result;
    }

    proc_result<alive_result> apply(const alive_args& args)
    {
        const string& worker = args.worker;
        const time_t time = args.ts;

        proc_result<alive_result> result;
        worker_update_time(worker, time);
        result.data.success = true;
        return result;
    }

    proc_result<cleanup_active_result> apply(const cleanup_active_args& args)
    {
        const time_t time = args.ts;
        const time_t activity_timeout = args.max_activity_time;

        proc_result<cleanup_active_result> result;

        // move all timed out active tasks back to the pending queue
        auto cur = index_->active.begin();
        while (cur != index_->active.end()) {
            auto itask = data_->tasks.find(*cur);
            assert(exists(itask));

            auto expire_time = itask->second.active_start_time() + activity_timeout;
            if (expire_time < time) {
                worker_released_tasks(itask->second.worker(), 1, 0);
                index_->active.erase(cur++);  // ! increment before erase
                move_active_to_pending_no_index_update(itask);
                ++result.data.count;
            } else {
                ++cur;
            }
        }
        return result;
    }


    proc_result<cleanup_active_result> apply(const cleanup_workers_args& args)
    {
        const time_t time = args.ts;
        const time_t activity_timeout = args.max_activity_time;

        proc_result<cleanup_active_result> result;

        // move all timed out active tasks back to the pending queue
        auto cur = index_->active.begin();
        while (cur != index_->active.end()) {
            auto itask = data_->tasks.find(*cur);
            assert(exists(itask));

            auto expire_time = worker_get_time(itask->second.worker()) + activity_timeout;
            if (expire_time < time) {
                worker_released_tasks(itask->second.worker(), 1, 0);
                index_->active.erase(cur++);  // ! increment before erase
                move_active_to_pending_no_index_update(itask);
                ++result.data.count;
            } else {
                ++cur;
            }
        }
        return result;
    }

    proc_result<wakeup_delayed_result> apply(const wakeup_delayed_args& args)
    {
        const time_t time = args.ts;

        proc_result<wakeup_delayed_result> result;

        // move all ready delayed tasks to the pending queue
        auto cur = index_->delayed.begin();
        while (cur != index_->delayed.end()) {
            auto itask = data_->tasks.find(*cur);
            assert(exists(itask));

            auto expire_time = itask->second.delay_start_time() + itask->second.delay_sec();
            if (expire_time < time) {
                index_->delayed.erase(cur++);  // ! increment before erase
                move_delayed_to_pending_no_index_update(itask);
                ++result.data.count;
            } else {
                ++cur;
            }
        }
        return result;
    }

    proc_result<clear_result> clear()
    {
        proc_result<clear_result> result;

        result.data.count = data_->tasks.size();
        data_->reset_tasks_and_workers();
        index_->clear();
        return result;
    }

private:
    task create_task_from_draft(const task_id_t& id, const task_draft& draft)
    {
        task new_task;
        new_task.id = id;
        new_task.uid = draft.uid;
        new_task.service = draft.service;
        new_task.local_id = draft.local_id;
        return new_task;
    }

    task_id_t make_id(const task_draft& draft)
    {
        task_id_t result;
        result.reserve(draft.uid.size() + draft.service.size() + 2);
        result += draft.uid;
        result += "##";
        result += draft.service;
        return result;
    }

    void move_active_to_pending(data::tasks_t::iterator itask)
    {
        auto active_pos = index_->active.find(itask->first);
        assert(active_pos != index_->active.end());
        index_->active.erase(active_pos);

        move_active_to_pending_no_index_update(itask);
    }

    void move_active_to_pending_no_index_update(data::tasks_t::iterator itask)
    {
        assert(exists(itask));
        insert_pending_queue(itask);
        itask->second.cleanup_active();
    }

    void move_delayed_to_pending(data::tasks_t::iterator itask)
    {
        auto delayed_pos = index_->delayed.find(itask->first);
        assert(delayed_pos != index_->delayed.end());
        index_->delayed.erase(delayed_pos);

        move_delayed_to_pending_no_index_update(itask);
    }

    void move_delayed_to_pending_no_index_update(data::tasks_t::iterator itask)
    {
        assert (exists(itask));
        insert_pending_queue(itask);
        itask->second.wakeup_delayed();
    }

    bool exists(data::tasks_t::iterator itask)
    {
        return itask != data_->tasks.end();
    }

    bool is_finished(data::tasks_t::iterator itask)
    {
        return !is_pending(itask) && !is_active(itask) && !is_delayed(itask);
    }

    void try_to_erase(data::tasks_t::iterator itask)
    {
        if (is_finished(itask)) data_->tasks.erase(itask);
    }

    bool is_active(const task_id_t& id)
    {
        return index_->active.count(id);
    }

    bool is_delayed(const task_id_t& id)
    {
        return index_->delayed.count(id);
    }

    bool is_pending(data::tasks_t::iterator itask)
    {
        return itask->second.is_pending();
    }

    bool is_active(data::tasks_t::iterator itask)
    {
        return itask->second.is_active();
    }

    bool is_delayed(data::tasks_t::iterator itask)
    {
        return itask->second.is_delayed();
    }

    data::tasks_t::iterator create_new_task_pack(const task& task, delay_flags_t delay_flags)
    {
        return data_->tasks.insert({task.id, task_pack(task, delay_flags)}).first;
    }

    void insert_pending_queue(data::tasks_t::iterator itask)
    {
        if (!is_pending(itask)) {
            data_->pending_queue.push_back(itask->first);
        }
    }

    data::pending_queue_t::iterator drop_pending_queue_item(data::pending_queue_t::iterator cur)
    {
        return data_->pending_queue.erase(cur);
    }

    void activate(data::tasks_t::iterator itask, const string& worker, const time_t start_time)
    {
        auto index_pos = index_->active.find(itask->first);
        assert(index_pos == index_->active.end());

        itask->second.activate(worker, start_time);
        index_->active.insert(index_pos, itask->first);
    }

    void deactivate(data::tasks_t::iterator itask)
    {
        assert (itask != data_->tasks.end());
        auto index_pos = index_->active.find(itask->first);
        assert(index_pos != index_->active.end());
        index_->active.erase(index_pos);
        itask->second.reset_active();
    }

    void delay(data::tasks_t::iterator itask, const time_t start_time,
            const time_t delay_sec, const delay_flags_t flags)
    {
        auto active_pos = index_->active.find(itask->first);
        assert(active_pos != index_->active.end());
        auto delayed_pos = index_->delayed.find(itask->first);
        assert(delayed_pos == index_->delayed.end());

        assert (itask != data_->tasks.end());
        itask->second.delay(start_time, delay_sec, flags);

        index_->active.erase(active_pos);
        index_->delayed.insert(delayed_pos, itask->first);
    }

    void worker_update_time(const string& worker, const time_t time)
    {
        auto pos = data_->workers.find(worker);
        if (pos != data_->workers.end()) {
            pos->second.update_last_seen(time);
        }
    }

    time_t worker_get_time(const string& worker)
    {
        auto pos = data_->workers.find(worker);
        if (pos != data_->workers.end()) {
            return pos->second.last_seen;
        }
        return 0;
    }

    void worker_retained_tasks(const string& worker, const size_t count, const time_t time)
    {
        auto pos = data_->workers.find(worker);
        if (pos == data_->workers.end()) {
            data_->workers.insert(pos, {worker, worker_info({worker, static_cast<size_t>(count), time})});
        } else {
            pos->second.update_last_seen(time);
            pos->second.active_tasks += count;
        }
    }

    void worker_released_tasks(const string& worker, const size_t count, const time_t time)
    {
        auto pos = data_->workers.find(worker);
        if (pos == data_->workers.end()) {
            YLOG_G(error) << "worker_released_task no such worker";
        } else {
            pos->second.update_last_seen(time);
            pos->second.active_tasks -= count;
            if (pos->second.active_tasks == 0) {
                data_->workers.erase(pos);
            }
        }
    }

    void update_local_id(local_id_t & dest, const local_id_t source)
    {
        if (source != 0 && dest < source) {
            dest = source;
        }
    }

    data_ptr data_;
    data_index_ptr index_;
    domain_settings settings_;
};

}
