#include "database.h"
#include "snapshot.h"
#include "data.h"

namespace ymod_xtasks {

void database::init(const yplatform::ptree& conf)
{
    settings.parse(conf);
    domain_settings_.parse(conf);

    namespace filesystem = boost::filesystem;
    if (!filesystem::exists(settings.root_dir)) {
        filesystem::create_directories(settings.root_dir);
        filesystem::create_directories(settings.snapshots_dir);
    } else if (!filesystem::is_directory(settings.root_dir)) {
        throw std::runtime_error("xtasks database root is not a rirectory");
    }
    if (!filesystem::exists(settings.snapshots_dir)) {
        YLOG_L(warning) << "no snapshot directory found";
        filesystem::create_directories(settings.snapshots_dir);
    }

    timers_.reset(new timers::queue(yplatform::global_net_reactor));
    snapshot_timer_ = timers_->create_timer();

    try_to_load_from_snapshot();
}

void database::start()
{
    stop_ = false;
    if (settings.auto_snapshoting) {
        snapshot_timer_->async_wait(settings.snapshot_interval,
          boost::bind(&database::handle_snapshot_timer, shared_from(this)));
    }
}

void database::stop()
{
    stop_ = true;
    snapshot_timer_->cancel();
}

void database::fini()
{
    handle_snapshot_timer();
}

bool database::is_ok()
{
  scoped_lock lock(mutex_);
  return !corrupted_;
}

ymod_paxos::iid_t database::get_revision()
{
  scoped_lock lock(mutex_);
  return (sync_active_ || corrupted_ || !exec_wrapper_) ? -1 : exec_wrapper_->get_revision();
}

bool database::is_modifying(const operation& op)
{
    auto code = static_cast<opcodes::type> (op.code());
    return code != opcodes::none && code != opcodes::get_all_counters;
}

void database::apply(ymod_paxos::iid_t iid, operation op, std::weak_ptr<ymod_paxos::icaller> caller)
{
    scoped_lock lock(mutex_);
    if (sync_active_ || corrupted_ || !exec_wrapper_)
        throw std::logic_error("xtasks database is not ready to apply operations");
    auto exec_wrapper = exec_wrapper_;
    lock.unlock();

    try {
        exec_wrapper->apply(iid, std::move(op), caller);
        if (!exec_wrapper->is_ok()) {
            lock.lock();
            if (exec_wrapper_ && !exec_wrapper_->is_ok()) corrupted_ = true;
        }
    } catch (...) {
        if (!exec_wrapper->is_ok()) {
            lock.lock();
            if (exec_wrapper_ && !exec_wrapper_->is_ok()) corrupted_ = true;
        }
        throw;
    }
}

database::sync_manager_ptr database::start_sync(const serialized_data_type& serialized_snapshot)
{
    Snapshot snapshot;
    ymod_paxos::unpack(serialized_snapshot, snapshot);
    auto ret = std::make_shared<sync_manager>(snapshot);

    scoped_lock lock(mutex_);
    if (exec_wrapper_) exec_wrapper_.reset();
    sync_active_ = true;

    return ret;
}

void database::finish_sync(sync_manager_ptr manager_base)
{
    scoped_lock lock(mutex_);
    if (!sync_active_) throw std::logic_error("sync is not active now");
    auto manager = dynamic_cast<sync_manager*>(manager_base.get());
    if (!manager) {
        throw std::runtime_error("sync_manager cast failed");
    }

    // TODO Verify data (compare tasks and workers)
    auto& data =  manager->snapshot_.data;
    auto index = std::make_shared<data_index>();
    build_index(data, index);
    YLOG_L(notice) << "applied snapshot: revision=" << data->rev
        << " timestamp=" << manager->snapshot_.timestamp
        << " pending=" << data->pending_queue.size()
        << " tasks=" << data->tasks.size()
        << " active=" << index->active.size()
        << " delayed=" << index->delayed.size()
        << " workers=" << data->workers.size();

    exec_wrapper_ = std::make_shared<exec_wrapper>(data, index,
        domain_settings_, logger());

    sync_active_ = false;
    corrupted_ = false;
}

boost::optional<Snapshot> database::create_snapshot()
{
    boost::optional<Snapshot> result;
    scoped_lock lock(mutex_);
    if (!sync_active_ && !corrupted_ && exec_wrapper_ && exec_wrapper_->is_ok()) {
        auto exec_wrapper = exec_wrapper_;
        lock.unlock();
        result = exec_wrapper->create_snapshot();
    }
    return result;
}

database::serialized_data_type database::get_snapshot()
{
    scoped_lock lock(mutex_);
    if (sync_active_ || corrupted_ || !exec_wrapper_) throw std::domain_error("sync denied from corrupted database");
    auto exec_wrapper = exec_wrapper_;
    lock.unlock();
    return exec_wrapper->get_snapshot();
}

database::serialized_data_type database::get_delta(const serialized_data_type&)
{
    throw std::logic_error("chunked sync is not supported by xtasks database");
}

void database::handle_snapshot_timer()
{
    scoped_lock lock(mutex_);
    if (!corrupted_ && !sync_active_ && exec_wrapper_ && exec_wrapper_->is_ok()) {
        YLOG_L(debug) << "creating snapshot";
        auto snapshot = exec_wrapper_->create_snapshot();
        SnapshotFilesInfo files_info(settings.snapshots_dir, snapshot.timestamp);
        lock.unlock();
        write_snapshot(snapshot, files_info);
        lock.lock();
    }
    if (!stop_) {
        snapshot_timer_->async_wait(settings.snapshot_interval,
            boost::bind(&database::handle_snapshot_timer, shared_from(this)));
    }
}

void database::try_to_load_from_snapshot()
{
    corrupted_ = true;
    auto init_data = std::make_shared<data>();
    auto init_data_index = std::make_shared<data_index>();

    SnapshotList snapshot_list = list_snapshots(settings.snapshots_dir);
    if (snapshot_list.empty()) {
        YLOG_L(notice) << "no snapshots found";
        corrupted_ = false;
    }

    for (auto i = 0U; i < std::min(snapshot_list.size(),
          static_cast<size_t>(settings.load_snapshot_retries)); ++i)
    {
        SnapshotFilesInfo files_info = *(snapshot_list.rbegin() + i);
        try {
            Snapshot s = read_snapshot(files_info);
            init_data = s.data;
            build_index(init_data, init_data_index);
            corrupted_ = false;
            YLOG_L(notice) << "loaded from the snapshot: revision="
                << s.data->rev << " ts=" << s.timestamp;
            break;
        } catch (const std::exception & e) {
            YLOG_L(warning) << "failed to load last snapshot: file="
                << files_info.snapshot_filename() << " exception=" << e.what();
        }
    }

    if (!corrupted_) {
        exec_wrapper_ = std::make_shared<exec_wrapper>(init_data, init_data_index,
            domain_settings_, logger());
        return;
    }

    YLOG_L(error) << "no valid snapshots found, starting as empty DB";
}

}
