#pragma once

#include "messages.h"
#include "snapshot_io.h"
#include "operations.h"
#include "sync_manager.h"
#include "exec_wrapper.h"
#include <ymod_xtasks/types.h>
#include <ymod_paxos/db.h>
#include <yplatform/time_traits.h>
#include <sintimers/queue.h>
#include <boost/filesystem.hpp>
#include <exception>

namespace ymod_xtasks {

class database : public ymod_paxos::abstract_database, public yplatform::module
{
public:
    void init(const yplatform::ptree& conf);
    void start();
    void stop();
    void fini();
    bool is_ok() override;
    ymod_paxos::iid_t get_revision() override;
    bool is_modifying(const operation& op) override;
    void apply(ymod_paxos::iid_t iid, operation op, std::weak_ptr<ymod_paxos::icaller> caller) override;
    sync_manager_ptr start_sync(const serialized_data_type& snapshot) override;
    void finish_sync(sync_manager_ptr sync_manager) override;
    boost::optional<Snapshot> create_snapshot();
    serialized_data_type get_snapshot() override;
    serialized_data_type get_delta(const serialized_data_type& request) override;

private:
    void handle_snapshot_timer();
    void try_to_load_from_snapshot();

    std::shared_ptr<exec_wrapper> exec_wrapper_;

    struct settings_t
    {
        settings_t() : auto_snapshoting(true), snapshot_interval(5) {}
        boost::filesystem::path root_dir;
        boost::filesystem::path snapshots_dir;
        bool auto_snapshoting;
        yplatform::time_traits::duration snapshot_interval;
        size_t load_snapshot_retries;

        void parse(const yplatform::ptree& conf)
        {
            load_snapshot_retries = conf.get<size_t>("load_snapshot_deep", 3U);
            auto_snapshoting = conf.get("auto_snapshoting", true);
            snapshot_interval = conf.get<yplatform::time_traits::duration>("snapshot_interval");
            root_dir = conf.get<std::string>("root_dir");
            snapshots_dir = root_dir.native() + "/snapshots";
        }
    } settings;

    mutex mutex_;
    bool stop_ = true;
    domain_settings domain_settings_;
    timers::queue_ptr timers_;
    timers::timer_ptr snapshot_timer_;
    bool corrupted_ = false;
    bool sync_active_ = false;
};

}
