#pragma once

#include "data.h"
#include "messages.h"
#include "snapshot.h"
#include "operations.h"
#include "data_processor.h"
#include "results.h"
#include <ymod_xtasks/types.h>
#include <ymod_paxos/db.h> // data types definition
#include <ymod_paxos/caller.h>
#include <ymod_paxos/packing.hpp>
#include <exception>

#define DB_CHECK_CORRUPTED \
  if (corrupted_) { \
    if (caller) caller->set_error(op.uniq_id(), \
    ymod_paxos::error(ymod_paxos::ErrorCode_DB_IS_CORRUPTED, "db is corrupted")); \
    return;\
  }


#define ARG_CHECK_STRING(name) \
    if (name == "") { \
        update_version(rev); \
        set_caller_error(caller, op, string("empty string argument: ") + #name); \
        return; \
    }


#define ARG_FIND_STRING(name) \
    auto name = ymod_xtasks::extract<string>(op, #name, ""); \
    ARG_CHECK_STRING(name)

namespace ymod_xtasks
{

class exec_wrapper : public yplatform::log::contains_logger
{
  using serialized_data_type = ymod_paxos::abstract_database::serialized_data_type;
  using sync_manager_ptr = ymod_paxos::abstract_database::sync_manager_ptr;

public:

  exec_wrapper(data_ptr data, data_index_ptr index,
      domain_settings const& settings,
      const yplatform::log::source & logger)
  : yplatform::log::contains_logger(logger)
  {
    corrupted_ = false;
    data_ = data;
    index_ = index;
    processor_ = boost::make_shared<data_processor>(data_, index_, settings);
    YLOG_L(debug) << "exec wrapper created with data rev=" << data_->rev;
  }

  void apply(iid_t rev, operation op, std::weak_ptr<ymod_paxos::icaller> wcaller)
  {
    YLOG_L(debug) << "apply operation=" << opcodes::symbolic(op.code())
        << " rev=" << rev << " uniq_id=" << op.uniq_id();

    std::shared_ptr<ymod_paxos::icaller> caller = wcaller.lock();

    try {
      switch (op.code()) {
        case opcodes::create_task:
          apply_impl<opcodes::create_task>(rev, op, caller);
          break;

        case opcodes::get_tasks:
          apply_impl<opcodes::get_tasks>(rev, op, caller);
          break;

        case opcodes::fin_task:
          apply_impl<opcodes::fin_task>(rev, op, caller);
          break;

        case opcodes::delay_task:
          apply_impl<opcodes::delay_task>(rev, op, caller);
          break;

        case opcodes::alive:
          apply_impl<opcodes::alive>(rev, op, caller);
          break;

        case opcodes::cleanup_active:
          apply_impl<opcodes::cleanup_active>(rev, op, caller);
          break;

        case opcodes::cleanup_workers:
          apply_impl<opcodes::cleanup_workers>(rev, op, caller);
          break;

        case opcodes::wakeup_delayed:
          apply_impl<opcodes::wakeup_delayed>(rev, op, caller);
          break;

        case opcodes::get_all_counters:
          get_all_counters(rev, op, caller);
          break;

        case opcodes::clear:
          clear(rev, op, caller);
          break;
      }
    }
    catch (const std::exception& e) {
      YLOG_G(error) << "db is corrupted exception=" << e.what();
      scoped_lock lock(mutex_);
      corrupted_ = true;
    }
    catch (...) {
      YLOG_G(error) << "db is corrupted exception=unknown";
      scoped_lock lock(mutex_);
      corrupted_ = true;
    }
  }

  serialized_data_type get_snapshot()
  {
    serialized_data_type result = ymod_paxos::pack(get_snapshot_i());
    return result;
  }

  Snapshot create_snapshot()
  {
    scoped_lock lock(mutex_);
    return get_snapshot_i();
  }

  iid_t get_revision()
  {
    scoped_lock lock(mutex_);
    return data_->rev;
  }

  bool is_ok()
  {
    scoped_lock lock(mutex_);
    return !corrupted_;
  }

private:

  void check_is_ok_i()
  {
    if (corrupted_) {
      throw std::runtime_error("db is corrupted");
    }
  }

  Snapshot get_snapshot_i()
  {
    if (corrupted_) {
      throw std::runtime_error("create snapshot failed: db is corrupted");
    }
    Snapshot result;
    result.data = std::make_shared<data>(*data_);
    result.timestamp = std::time(0);
    return result;
  }

  void update_version(iid_t rev)
  {
    scoped_lock lock(mutex_);
    update_version_i(rev);
  }

  void update_version_i(iid_t rev)
  {
    if (rev != -1) {
      assert(data_->rev + 1 == rev);
      data_->rev = rev;
    }
  }

  template <typename Result>
  void set_caller_result(std::shared_ptr<ymod_paxos::icaller> caller,
      const operation& op, const proc_result<Result>& result)
  {
    try {
      if (caller) {
        caller->set_result(op.uniq_id(), pack_str(result));
      }
    }
    catch (const std::exception & e) {
      YLOG_G(error) << "xtasks database set_caller_result exception=" << e.what();
    }
    catch (...) {
      YLOG_G(error) << "xtasks database set_caller_result exception=unknown";
    }
  }

  void set_caller_error(std::shared_ptr<ymod_paxos::icaller> caller,
      const operation& op, const string& reason)
  {
    try {
      if (caller) {
        caller->set_attribute("error", reason);
        caller->set_result(op.uniq_id(), "");
      }
    }
    catch (const std::exception & e) {
      YLOG_G(error) << "xtasks database set_caller_error exception=" << e.what();
    }
    catch (...) {
      YLOG_G(error) << "xtasks database set_caller_error exception=unknown";
    }
  }

  template<unsigned OpCode>
  void apply_impl(iid_t rev, const operation& op, std::shared_ptr<ymod_paxos::icaller> caller)
  {
    typedef op_traits<OpCode> traits_t;
    typename traits_t::args args;
    try {
      args = op.convert_payload<typename traits_t::args>();
    } catch (const std::bad_cast&) {
      update_version(rev);
      set_caller_error(caller, op, "failed to unpack args for op=" + opcodes::symbolic(op.code()));
      return;
    }

    scoped_lock lock(mutex_);
    DB_CHECK_CORRUPTED

    auto result = processor_->apply(args);
    update_version_i(rev);

    lock.unlock();
    set_caller_result(caller, op, result);
  }

  void clear(iid_t rev, const operation& op,
      std::shared_ptr<ymod_paxos::icaller> caller)
  {
    scoped_lock lock(mutex_);
    DB_CHECK_CORRUPTED

    auto result = processor_->clear();
    update_version_i(rev);

    lock.unlock();
    set_caller_result(caller, op, result);
  }

  void get_all_counters(iid_t rev, const operation& op,
      std::shared_ptr<ymod_paxos::icaller> caller)
  {
    scoped_lock lock(mutex_);
    DB_CHECK_CORRUPTED

    proc_result<all_counters_result> result;
    result.data.total = data_->tasks.size();
    result.data.active = index_->active.size();
    result.data.pending = data_->pending_queue.size();
    result.data.delayed = index_->delayed.size();
    result.data.workers = data_->workers.size();

    update_version_i(rev);

    lock.unlock();
    set_caller_result(caller, op, result);
  }

  bool corrupted_;
  mutex mutex_;
  data_ptr data_;
  data_index_ptr index_;
  shared_ptr<data_processor> processor_;
};

}
