#include "snapshot_io.h"

#include <algorithm>
#include <boost/filesystem.hpp>
#include <boost/crc.hpp>
#include <ymod_paxos/packing.hpp>
#include <fstream>

namespace ymod_xtasks {

#define SIZE_CHECK_EDGE 1024*1024*1024

template<typename T>
void write_data(std::ostream & file, const T & t)
{
    file.write(reinterpret_cast<const char*> (&t), sizeof (t));
}

template<typename T>
inline void read_data(std::istream & file, T & t)
{
    file.read(reinterpret_cast<char*> (&t), sizeof(t));
}

template<typename T, typename Crc>
void write_data(std::ostream & file, const T & t, Crc & crc)
{
    file.write(reinterpret_cast<const char*> (&t), sizeof (t));
    crc.process_bytes(reinterpret_cast<const char*> (&t), sizeof (t));
}

template<typename T, typename Crc>
inline void read_data(std::istream & file, T & t, Crc & crc)
{
    file.read(reinterpret_cast<char*> (&t), sizeof(t));
    crc.process_bytes(reinterpret_cast<const char*> (&t), sizeof (t));
}

void write_snapshot(const Snapshot& snapshot, const SnapshotFilesInfo& files_info)
{
    std::ofstream file(files_info.snapshot_filename().native(), std::ofstream::trunc);
    if (!file.good()) {
        throw std::runtime_error("failed to open for write snapshot file");
    }
    std::ofstream sum_file(files_info.checksum_filename().native(), std::ofstream::trunc);
    if (!sum_file.good()) {
        throw std::runtime_error("failed to open for write snapshot file");
    }

    file.exceptions ( std::ofstream::failbit | std::ofstream::badbit );
    sum_file.exceptions ( std::ofstream::failbit | std::ofstream::badbit );

    boost::crc_32_type crc32;

    string data_portion = ymod_paxos::pack(snapshot);
    write_data(file, data_portion.size(), crc32);
    file.write(data_portion.data(), data_portion.size());
    crc32.process_bytes(data_portion.data(), data_portion.size());

    file.flush();
    file.close();

    auto sum = crc32.checksum();
    write_data(sum_file, sum);

    sum_file.flush();
    sum_file.close();
}

Snapshot read_snapshot(const SnapshotFilesInfo& files_info)
{
    Snapshot result;
    std::ifstream file(files_info.snapshot_filename().native());
    if (!file.good()) {
        throw std::runtime_error("failed to open stapshot file '"
                + files_info.snapshot_filename().native() + "'");
    }
    std::ifstream sum_file(files_info.checksum_filename().native());
    if (!sum_file.good()) {
        throw std::runtime_error("failed to open stapshot sum file '"
                + files_info.checksum_filename().native() + "'");
    }

    file.exceptions ( std::ifstream::failbit | std::ifstream::badbit );
    sum_file.exceptions ( std::ifstream::failbit | std::ifstream::badbit );

    boost::crc_32_type crc32;

    size_t pack_size = 0;
    read_data(file, pack_size, crc32);
    if (pack_size > SIZE_CHECK_EDGE) {
        throw std::runtime_error("read_snapshot detected extremely big node size");
    }
    std::vector<char> buf;
    buf.resize(pack_size);
    file.read(&buf[0], pack_size);
    crc32.process_bytes(buf.data(), buf.size());
    ymod_paxos::unpack(buf, result);

    auto sum = crc32.checksum();

    decltype(sum) sum_stored;
    read_data(sum_file, sum_stored);

    if (sum_stored != sum) {
        throw std::runtime_error("read snapshot failed: wrong checksum");
    }

    return result;
}

SnapshotList list_snapshots(const path& dir)
{
    using boost::filesystem::is_directory;
    using boost::filesystem::directory_iterator;
    using boost::filesystem::filesystem_error;

    try {
        if (!is_directory(dir)) {
            throw std::runtime_error("not a directory: '"
                    + dir.native() + "'");
        }
        SnapshotList list;

        auto it = directory_iterator(dir), end = directory_iterator();
        while (it != end) {
            const path& p = it->path();
            if (p.extension() == SNAPSHOT_EXTENSION) {
                time_t timestamp = boost::lexical_cast<time_t>(p.stem().native());
                list.push_back(SnapshotFilesInfo(dir, timestamp));
            }
            ++it;
        }
        std::sort(list.begin(), list.end());
        return list;
    } catch (const filesystem_error & ex) {
        throw std::runtime_error("read directory "
                + dir.native() + "' error: " + ex.what());
    }
}


}
