#pragma once

#include "database/snapshot_io.h"
#include "database/database.h"
#include "database/snapshot_to_json.h"
#include "caller.h"
#include "json_write.h"
#include <ymod_xtasks/xtasks.h>
#include <ymod_paxos/node.h>
#include <yplatform/module.h>
#include <yplatform/util/unique_id.h>
#include <yplatform/find.h>
#include <boost/filesystem.hpp>

namespace ymod_xtasks {

class request_context : public yplatform::task_context
{
public:
    const std::string& get_name() const override
    {
        static const std::string NAME = "xtasts_context";
        return NAME;
    }
};

class module : public xtasks, public yplatform::module
{
public:
    void init(const yplatform::ptree& conf)
    {
        auto ymod_paxos_name = conf.get<std::string>("paxos_node");
        auto database_name = conf.get<std::string>("database");
        paxos_node_ = yplatform::find<ymod_paxos::node, std::shared_ptr>(ymod_paxos_name);
        database_ = yplatform::find<database, std::shared_ptr>(database_name);

        reload(conf);
    }

    void reload(const yplatform::ptree& conf)
    {
      force_create_task_fails_ = conf.get<bool>("force_create_task_fails", false);
    }

    void create_task(const task_context_ptr& ctx, const task_draft& draft, const simple_callback& cb)
    {
        if (force_create_task_fails_) {
          cb(make_error(err_code::err_code_unknown, "forced error"));
          return;
        }
        operation op(opcodes::create_task, make_context(), draft);
        submit(ctx, std::move(op), std::make_shared<simple_caller<create_task_result>>(cb));
    }

    void get_tasks(const task_context_ptr& ctx, const string& worker, const unsigned count, const tasks_callback& cb)
    {
        operation op(opcodes::get_tasks, make_context(), get_tasks_args{worker, count, std::time(0)});
        submit(ctx, std::move(op), std::make_shared<tasks_caller<get_tasks_result>>(cb));
    }

    void fin_task(const task_context_ptr& ctx, const string& worker, const task_id_t& task_id, const simple_callback& cb)
    {
        operation op(opcodes::fin_task, make_context(), fin_task_args{worker, task_id, std::time(0)});
        submit(ctx, std::move(op), std::make_shared<simple_caller<fin_task_result>>(cb));
    }

    void delay_task(const task_context_ptr& ctx, const string& worker, const task_id_t& task_id, const time_t delay_sec, const simple_callback& cb)
    {
        operation op(opcodes::delay_task, make_context(), delay_task_args{worker,
            task_id, delay_sec, std::time(0), delay_flags::none});
        submit(ctx, std::move(op), std::make_shared<simple_caller<delay_task_result>>(cb));
    }

    void delay_task_shallow(const task_context_ptr& ctx, const string& worker, const task_id_t& task_id, const time_t delay_sec, const simple_callback& cb)
    {
        operation op(opcodes::delay_task, make_context(), delay_task_args{worker,
            task_id, delay_sec, std::time(0), delay_flags::wakeup_on_create | delay_flags::ignore_if_pending});
        submit(ctx, std::move(op), std::make_shared<simple_caller<delay_task_result>>(cb));
    }

    void alive(const task_context_ptr& ctx, const string& worker, const simple_callback& cb)
    {
        operation op(opcodes::alive, make_context(), alive_args{worker, std::time(0)});
        submit(ctx, std::move(op), std::make_shared<simple_caller<alive_result>>(cb));
    }

    void cleanup_active(const task_context_ptr& ctx, const time_t interval, const counter_callback& cb)
    {
        operation op(opcodes::cleanup_active, make_context(), cleanup_active_args{std::time(0),interval});
        submit(ctx, std::move(op), std::make_shared<counter_caller<cleanup_active_result>>(cb));
    }

    void cleanup_workers(const task_context_ptr& ctx, const time_t interval, const counter_callback& cb)
    {
        operation op(opcodes::cleanup_workers, make_context(), cleanup_workers_args{std::time(0),interval});
        submit(ctx, std::move(op), std::make_shared<counter_caller<cleanup_workers_result>>(cb));
    }

    void wakeup_delayed(const task_context_ptr& ctx, const counter_callback& cb)
    {
        operation op(opcodes::wakeup_delayed, make_context(), wakeup_delayed_args{std::time(0)});
        submit(ctx, std::move(op), std::make_shared<counter_caller<wakeup_delayed_result>>(cb));
    }

    void clear(const task_context_ptr& ctx, const counter_callback& cb)
    {
        operation op(opcodes::clear, make_context());
        submit(ctx, std::move(op), std::make_shared<counter_caller<clear_result>>(cb));
    }

    void get_all_counters(const task_context_ptr& ctx, const all_counters_list_callback& cb)
    {
        operation op(opcodes::get_all_counters, make_context());
        submit(ctx, std::move(op), std::make_shared<result_caller<all_counters_result>>(cb));
    }

    string get_summary_json()
    {
        auto snapshot = database_->create_snapshot();
        json_value json;
        if (snapshot) {
          json = snapshot_data_to_json(snapshot->data);
        } else {
          if (!database_->is_ok()) {
            json["error"] = "db is not initialized or corrupted";
          }
          else {
            json["error"] = "can't get snapshot";
          }
        }

        return json_write(json);
    }

    std::string make_context()
    {
        return yplatform::util::make_unique_id(reinterpret_cast<unsigned long long>(this));
    }

private:
    void submit(const task_context_ptr& ctx, operation op, std::shared_ptr<ymod_paxos::icaller> caller)
    {
        paxos_node_->perform(ctx, std::move(op), caller);
    }

    bool force_create_task_fails_;
    mutex m_;

    std::shared_ptr<database> database_;
    std::shared_ptr<ymod_paxos::node> paxos_node_;
};

}
