from .tasks import generate_tasks_panel
from mail.yasm.lib.micronaut.endpoints import Endpoint, EndpointSet, generate_endpoints_panel
from mail.yasm.lib.micronaut.jvm import generate_jvm_panel, GC, Executor, ExecutorType
from mail.yasm.lib.micronaut.jdbc import generate_jdbc_panel
from mail.yasm.lib.micronaut.brief import Configuration, generate_brief_panel

SERVICE_CONTROLLERS = [
    EndpointSet(name='grant',
                endpoints=[
                    Endpoint.server(method='POST', uri='/grant/user'),
                    Endpoint.server(method='POST', uri='/grant/group'),
                    Endpoint.server(method='POST', uri='/grant/role'),
                    Endpoint.server(method='POST', uri='/grant/user/resources_actions'),
                    Endpoint.server(method='GET',  uri='/grant/user/resource_types_actions'),
                    Endpoint.server(method='POST', uri='/grant/user/single_type_resources_actions')
                ]),
    EndpointSet(name='group',
                endpoints=[
                    Endpoint.server(method='POST',   uri='/group/add'),
                    Endpoint.server(method='POST',   uri='/group/create'),
                    Endpoint.server(method='POST',   uri='/group/update'),
                    Endpoint.server(method='DELETE', uri='/group'),
                    Endpoint.server(method='GET',    uri='/group/all'),
                    Endpoint.server(method='GET',    uri='/group/filter'),
                    Endpoint.server(method='POST',   uri='/group/add_user'),
                    Endpoint.server(method='POST',   uri='/group/remove_user')
                ]),
    EndpointSet(name='location',
                endpoints=[
                    Endpoint.server(method='POST',   uri='/location/create'),
                    Endpoint.server(method='POST',   uri='/location/update'),
                    Endpoint.server(method='DELETE', uri='/location'),
                    Endpoint.server(method='GET',    uri='/location/all'),
                    Endpoint.server(method='GET',    uri='/location/filter')
                ]),
    EndpointSet(name='resource',
                endpoints=[
                    Endpoint.server(method='POST',   uri='/resource/type/create'),
                    Endpoint.server(method='POST',   uri='/resource/type/get_or_create'),
                    Endpoint.server(method='GET',    uri='/resource/type'),
                    Endpoint.server(method='GET',    uri='/resource/types'),
                    Endpoint.server(method='GET',    uri='/resource/all'),
                    Endpoint.server(method='POST',   uri='/resource/create'),
                    Endpoint.server(method='POST',   uri='/resource/create/bulk'),
                    Endpoint.server(method='POST',   uri='/resource/add'),
                    Endpoint.server(method='POST',   uri='/resource/add/bulk'),
                    Endpoint.server(method='POST',   uri='/resource/update'),
                    Endpoint.server(method='POST',   uri='/resource/update/bulk'),
                    Endpoint.server(method='DELETE', uri='/resource')
                ]),
    EndpointSet(name='role',
                endpoints=[
                    Endpoint.server(method='POST',   uri='/role/create'),
                    Endpoint.server(method='POST',   uri='/role/update'),
                    Endpoint.server(method='DELETE', uri='/role'),
                    Endpoint.server(method='POST',   uri='/role/assign/user'),
                    Endpoint.server(method='POST',   uri='/role/remove/user'),
                    Endpoint.server(method='POST',   uri='/role/assign/group'),
                    Endpoint.server(method='POST',   uri='/role/remove/group')
                ]),
    EndpointSet(name='user',
                endpoints=[
                    Endpoint.server(method='POST',   uri='/user/create'),
                    Endpoint.server(method='POST',   uri='/user/create/batch'),
                    Endpoint.server(method='POST',   uri='/user/update'),
                    Endpoint.server(method='DELETE', uri='/user'),
                    Endpoint.server(method='GET',    uri='/user/filter'),
                    Endpoint.server(method='GET',    uri='/user/with_groups/filter')
                ])
]

WORKER_CONTROLLERS = [
    EndpointSet(name='task',
                endpoints=[
                    Endpoint.server(method='POST', uri='/task/execute')
                ])
]

CLIENTS = [
    EndpointSet(name='tvmtool',
                endpoints=[
                    Endpoint.client(method='GET', uri='/tvm/tickets'),
                    Endpoint.client(method='GET', uri='/tvm/checksrv'),
                    Endpoint.client(method='GET', uri='/tvm/checkusr')
                ])
]

YT_CLIENTS = [
    EndpointSet(name='staff',
                endpoints=[
                    Endpoint.client(method='GET', uri='/v3/rooms'),
                    Endpoint.client(method='GET', uri='/v3/persons'),
                    Endpoint.client(method='GET', uri='/v3/groups'),
                    Endpoint.client(method='GET', uri='/v3/offices')
                ])
]

SERVICE_EXECUTORS = [
    Executor(type=ExecutorType.FORK_JOIN, name='io'),
    Executor(type=ExecutorType.SCHEDULED, name='scheduled'),
    Executor(type=ExecutorType.FIXED, name='master'),
    Executor(type=ExecutorType.FIXED, name='slave')
]

WORKER_EXECUTORS = [
    Executor(type=ExecutorType.FORK_JOIN, name='io'),
    Executor(type=ExecutorType.SCHEDULED, name='scheduled'),
    Executor(type=ExecutorType.FIXED, name='master'),
    Executor(type=ExecutorType.FIXED, name='cerberus_worker_executor')
]

WORKER_TASKS = [
    'partman_maintenance'
]

YT_WORKER_TASKS = [
    'sync_staff'
]

ENVIRONMENTS = ['testing', 'production']
DEFAULT_ENV = 'testing'
DEFAULT_DOMAIN = 'yt'
DEFAULT_COMPONENT = 'cerberus'
EDITORS = ["imdex", "prez", "spirit-1984", "kirillovap", "kremenkov", "levromanov"]

YT_PROD_CID = 'mdbn5pm81em3vls837r8'
PUB_PROD_CID = 'mdb0u4ft92rufsvpp4vu'
YT_TEST_CID = 'mdblb41i2ls95dnahl0a'
PUB_TEST_CID = 'mdb6diuolop1hk47lvua'

CONFIGURATIONS = [
    Configuration(component='cerberus', domain='yt',
                  panel_name='Cerberus-Yt',
                  controller_endpoint_sets=SERVICE_CONTROLLERS,
                  client_endpoint_sets=CLIENTS,
                  data_source_names=['master', 'slave'],
                  gc=GC.G1,
                  executors=SERVICE_EXECUTORS,
                  mdb_cid_by_env={'testing': YT_TEST_CID, 'production': YT_PROD_CID},
                  controller_endpoints_panel_name='Cerberus-Yt-API',
                  client_endpoints_panel_name='Cerberus-Yt-Clients',
                  jdbc_panel_name='Cerberus-Yt-JDBC',
                  jvm_panel_name='Cerberus-Yt-JVM'),
    Configuration(component='cerberus', domain='public',
                  panel_name='Cerberus-Public',
                  controller_endpoint_sets=SERVICE_CONTROLLERS,
                  client_endpoint_sets=CLIENTS,
                  data_source_names=['master', 'slave'],
                  gc=GC.G1,
                  executors=SERVICE_EXECUTORS,
                  mdb_cid_by_env={'testing': PUB_TEST_CID, 'production': PUB_PROD_CID},
                  controller_endpoints_panel_name='Cerberus-Public-API',
                  client_endpoints_panel_name='Cerberus-Public-Clients',
                  jdbc_panel_name='Cerberus-Public-JDBC',
                  jvm_panel_name='Cerberus-Public-JVM'),
    Configuration(component='worker', domain='yt',
                  panel_name='Cerberus-Worker-Yt',
                  controller_endpoint_sets=WORKER_CONTROLLERS,
                  client_endpoint_sets=CLIENTS + YT_CLIENTS,
                  data_source_names=['master'],
                  gc=GC.Parallel,
                  executors=WORKER_EXECUTORS,
                  mdb_cid_by_env={'testing': YT_TEST_CID, 'production': YT_PROD_CID},
                  controller_endpoints_panel_name='Cerberus-Worker-Yt-API',
                  client_endpoints_panel_name='Cerberus-Worker-Yt-Clients',
                  jdbc_panel_name='Cerberus-Worker-Yt-JDBC',
                  jvm_panel_name='Cerberus-Worker-Yt-JVM'),
    Configuration(component='worker', domain='public',
                  panel_name='Cerberus-Worker-Public',
                  controller_endpoint_sets=WORKER_CONTROLLERS,
                  client_endpoint_sets=CLIENTS,
                  data_source_names=['master'],
                  gc=GC.Parallel,
                  executors=WORKER_EXECUTORS,
                  mdb_cid_by_env={'testing': PUB_TEST_CID, 'production': PUB_PROD_CID},
                  controller_endpoints_panel_name='Cerberus-Worker-Public-API',
                  client_endpoints_panel_name='Cerberus-Worker-Public-Clients',
                  jdbc_panel_name='Cerberus-Worker-Public-JDBC',
                  jvm_panel_name='Cerberus-Worker-Public-JVM'),
]


def panels_templates():
    yield 'Cerberus',\
          generate_brief_panel(name='Cerberus', editors_list=EDITORS, qloud_project_id='mail.cerberus-',
                               env_list=ENVIRONMENTS, default_env=DEFAULT_ENV, configurations=CONFIGURATIONS,
                               default_domain=DEFAULT_DOMAIN, default_component=DEFAULT_COMPONENT)

    for c in CONFIGURATIONS:
        qloud_project_id = 'mail.cerberus-' + c.domain

        panel_name = c.panel_name + '-API'
        yield panel_name, generate_endpoints_panel(name=panel_name,
                                                   endpoint_set_list=c.controller_endpoint_sets,
                                                   qloud_project_id=qloud_project_id,
                                                   qloud_component=c.component,
                                                   env_list=ENVIRONMENTS,
                                                   default_env=DEFAULT_ENV,
                                                   editors_list=EDITORS)

        panel_name = c.panel_name + '-Clients'
        yield panel_name, generate_endpoints_panel(name=panel_name,
                                                   endpoint_set_list=c.client_endpoint_sets,
                                                   qloud_project_id=qloud_project_id,
                                                   qloud_component=c.component,
                                                   env_list=ENVIRONMENTS,
                                                   default_env=DEFAULT_ENV,
                                                   editors_list=EDITORS)

        panel_name = c.panel_name + '-JVM'
        yield panel_name, generate_jvm_panel(name=panel_name,
                                             qloud_project_id=qloud_project_id,
                                             qloud_component=c.component,
                                             env_list=ENVIRONMENTS,
                                             default_env=DEFAULT_ENV,
                                             gc=c.gc,
                                             executors=c.executors,
                                             editors_list=EDITORS)

        panel_name = c.panel_name + '-JDBC'
        yield panel_name, generate_jdbc_panel(name=panel_name,
                                              qloud_project_id=qloud_project_id,
                                              qloud_component=c.component,
                                              env_list=ENVIRONMENTS,
                                              default_env=DEFAULT_ENV,
                                              data_source_names=c.data_source_names,
                                              editors_list=EDITORS)

        if c.component == 'worker':
            panel_name = c.panel_name + '-Tasks'
            tasks = WORKER_TASKS if c.domain == 'public' else WORKER_TASKS + YT_WORKER_TASKS
            yield panel_name, generate_tasks_panel(name=panel_name,
                                                   qloud_project_id=qloud_project_id,
                                                   qloud_component=c.component,
                                                   env_list=ENVIRONMENTS,
                                                   default_env=DEFAULT_ENV,
                                                   editors_list=EDITORS,
                                                   task_names=tasks)
