from textwrap import dedent


class FlapsConfig:
    def __init__(self, crit, stable):
        self.crit = crit
        self.stable = stable

    def to_jinja_dict(self):
        return {
            'critical': self.crit,
            'stable': self.stable
        }


class PhoneEscalation:
    def __init__(self, repeat, call_tries):
        self.repeat = repeat
        self.call_tries = call_tries

    def to_jinja_dict(self, responsibles):
        return {
            'template_name': 'phone_escalation',
            'template_kwargs': {
                'logins': responsibles,
                'repeat': self.repeat,
                'call_tries': self.call_tries,
            }
        }


class JugglerConfig:
    def __init__(self, notification_methods=None, flaps_config=None, phone_escalation=None, ignore_weekends=True,
                 namespace='mail.maildev', tags=None):
        self.notification_methods = notification_methods
        self.ignore_weekends = ignore_weekends
        self.flaps_config = flaps_config
        self.phone_escalation = phone_escalation
        self.namespace = namespace
        self.tags = tags or []

    def to_jinja_dict(self, responsibles):
        config = {
            'namespace': self.namespace,
            'tags': self.tags,
        }
        if self.flaps_config:
            config['flaps'] = self.flaps_config.to_jinja_dict()
        notifies = []
        if self.notification_methods:
            assert responsibles, "Can't set notification methods {} with no responsibles".format(self.notification_methods)
            notifies += [
                {
                    'template_name': 'on_status_change',
                    'template_kwargs': {
                        'status': [
                            {'from': 'OK', 'to': 'CRIT'},
                            {'from': 'CRIT', 'to': 'OK'},
                            {'from': 'WARN', 'to': 'CRIT'},
                            {'from': 'CRIT', 'to': 'WARN'},
                        ],
                        'login': responsibles[0:1],
                        'method': self.notification_methods,
                        'ignore_weekends': self.ignore_weekends
                    }
                }
            ]
        if self.phone_escalation:
            assert responsibles, "Can't set escalation with no responsibles"
            notifies += [self.phone_escalation.to_jinja_dict(responsibles)]
        if notifies:
            config['notifications'] = notifies
        return config


class AlertRanges:
    def __init__(self, warn, crit):
        self.warn = warn
        self.crit = crit

    def to_jinja_dict(self):
        return self.__dict__


class AlertConfiguration:
    def __init__(self, controllers_4xx_alert_ranges, controllers_5xx_alert_ranges, clients_4xx_alert_ranges,
                 clients_5xx_alert_ranges, juggler_config):
        self.controllers_4xx_alert_ranges = controllers_4xx_alert_ranges
        self.controllers_5xx_alert_ranges = controllers_5xx_alert_ranges
        self.clients_4xx_alert_ranges = clients_4xx_alert_ranges
        self.clients_5xx_alert_ranges = clients_5xx_alert_ranges
        self.juggler_config = juggler_config

    def to_jinja_dict(self, responsibles):
        return {
            'controllers_4xx_alert_ranges': self.controllers_4xx_alert_ranges.to_jinja_dict(),
            'controllers_5xx_alert_ranges': self.controllers_5xx_alert_ranges.to_jinja_dict(),
            'clients_4xx_alert_ranges': self.clients_4xx_alert_ranges.to_jinja_dict(),
            'clients_5xx_alert_ranges': self.clients_5xx_alert_ranges.to_jinja_dict(),
            'juggler_config': self.juggler_config.to_jinja_dict(responsibles)
        }


class ExtraAlert:
    def __init__(self, name, signal, alert_ranges):
        self.name = name
        self.signal = signal
        self.ranges = alert_ranges

    def to_jinja(self, component):
        return dedent('''
            <% set service = '{component}_{name}' %>
            << Alert(
                name=q_prj ~ '_' ~ service,
                signal='{signal}',
                tags={{
                    "itype": "qloud",
                    "prj": q_prj,
                    "tier": "{component}*"
                }},
                mgroups=["QLOUD"],
                warn={warn},
                crit={crit},
                juggler_check=juggler_config|merge_with_dict({{
                    "host": q_prj,
                    "service": service
                }})
            ) >>
        ''').format(component=component,
                    name=self.name,
                    signal=self.signal,
                    warn=self.ranges.warn,
                    crit=self.ranges.crit)


def generate_alerts(responsibles, qloud_project_id, env, configuration, alert_configuration, extra_alerts=None):
    if extra_alerts is None:
        extra_alerts = []

    extra_alert_templates = [alert.to_jinja(configuration.component) for alert in extra_alerts]

    return '''[
<% set q_prj = '{qloud_project_id}.{env}' %>

<% set controller_endpoint_sets = [{controller_endpoint_sets}] %>
<% set client_endpoint_sets = [{client_endpoint_sets}] %>
<% set alert_config = {alert_config} %>
<% set juggler_config = alert_config['juggler_config'] %>

<# Controllers #>
<% for ctrl_endpoint_set in controller_endpoint_sets %>
    <% for endpoint in ctrl_endpoint_set['endpoints'] %>
        <% for code in ['4xx', '5xx'] %>
            <% set service = '{component}_api_' ~ endpoint['name'] ~ '_' ~ code %>
            << Alert(
                    name=q_prj ~ '_' ~ service,
                    signal=endpoint['signal_' ~ code],
                    tags={{
                        "itype": "qloud",
                        "prj": q_prj,
                        "tier": "{component}*"
                    }},
                    mgroups=["QLOUD"],
                    warn=alert_config['controllers_' ~ code ~ '_alert_ranges']['warn'],
                    crit=alert_config['controllers_' ~ code ~ '_alert_ranges']['crit'],
                    juggler_check=juggler_config|merge_with_dict({{
                        "host": q_prj,
                        "service": service
                    }})
                ) >> ,
        <%- endfor %>
    <%- endfor %>
<%- endfor %>

<# Clients #>
<% for client_endpoint_set in client_endpoint_sets %>
    <% for endpoint in client_endpoint_set['endpoints'] %>
        <% for code in ['4xx', '5xx'] %>
            <% set service = '{component}_client_' ~ endpoint['name'] ~ '_' ~ code %>
            << Alert(
                    name=q_prj ~ '_' ~ service,
                    signal=endpoint['signal_' ~ code],
                    tags={{
                        "itype": "qloud",
                        "prj": q_prj,
                        "tier": "{component}*"
                    }},
                    mgroups=["QLOUD"],
                    warn=alert_config['clients_' ~ code ~ '_alert_ranges']['warn'],
                    crit=alert_config['clients_' ~ code ~ '_alert_ranges']['crit'],
                    juggler_check=juggler_config|merge_with_dict({{
                        "host": q_prj,
                        "service": service
                    }})
                ) >> ,
        <%- endfor %>
    <%- endfor %>
<%- endfor %>

<# Code cache #>
<% for region in ['non_nmethods', 'non_profiled_nmethods', 'profiled_nmethods'] %>
    <% set service = '{component}_code_cache_' ~ region %>
    << Alert(
            name=q_prj ~ '_' ~ service,
            signal='perc(unistat-jvm_memory_used_nonheap_codeheap_' ~ region ~ '_axxx, unistat-jvm_memory_max_nonheap_codeheap_' ~ region ~ '_axxx)',
            tags={{
                "itype": "qloud",
                "prj": q_prj,
                "tier": "{component}*"
            }},
            mgroups=["QLOUD"],
            warn=[90, None],
            crit=[None, None],
            juggler_check=juggler_config|merge_with_dict({{
                "host": q_prj,
                "service": service
            }})
        ) >>
,
<%- endfor %>

<# Extra alerts #>
{extra_alerts}
]'''.format(qloud_project_id=qloud_project_id,
            env=env,
            component=configuration.component,
            controller_endpoint_sets=','.join([str(s.to_jinja_dict()) for s in configuration.controller_endpoint_sets]),
            client_endpoint_sets=','.join([str(s.to_jinja_dict()) for s in configuration.client_endpoint_sets]),
            alert_config=alert_configuration.to_jinja_dict(responsibles),
            extra_alerts=','.join(extra_alert_templates))
