from enum import Enum


class EndpointType(Enum):
    SERVER = 1
    CLIENT = 2


class Endpoint:
    def __init__(self, endpoint_type, method, uri):
        self.endpoint_type = endpoint_type
        self.suffix = 'http_server_requests' if endpoint_type == EndpointType.SERVER else 'http_client_requests'
        self.method = method
        self.uri = uri
        self.normalized = None

    @staticmethod
    def server(method, uri):
        return Endpoint(endpoint_type=EndpointType.SERVER, method=method, uri=uri)

    @staticmethod
    def client(method, uri):
        return Endpoint(endpoint_type=EndpointType.CLIENT, method=method, uri=uri)

    def normalize(self):
        if self.normalized is not None:
            return self.normalized

        uri = self.uri
        if uri.startswith('/'):
            uri = uri[1:]
        uri = uri.replace('/', '_').lower()

        self.normalized = Endpoint(endpoint_type=self.endpoint_type, method=self.method.lower(), uri=uri)
        return self.normalized

    def __status_signal(self, status):
        normalized = self.normalize()
        return 'unistat-{suffix}_{method}_{status}_{uri}_count_summ'.format(suffix=normalized.suffix,
                                                                            method=normalized.method,
                                                                            uri=normalized.uri,
                                                                            status=status)

    def signal_timings(self):
        normalized = self.normalize()
        return 'hmerge(unistat-{suffix}_{method}_<2xx|4xx|5xx>_{uri}_time_hgram)'.format(suffix=normalized.suffix,
                                                                                         method=normalized.method,
                                                                                         uri=normalized.uri)

    def signal_2xx(self):
        return self.__status_signal(status='2xx')

    def signal_4xx(self):
        return self.__status_signal(status='4xx')

    def signal_5xx(self):
        return self.__status_signal(status='5xx')

    def signal_total(self):
        normalized = self.normalize()
        return 'sum(unistat-{suffix}_{method}_<2xx|4xx|5xx>_{uri}_count_summ)'.format(suffix=normalized.suffix,
                                                                                      method=normalized.method,
                                                                                      uri=normalized.uri)

    def to_jinja_dict(self):
        return {'name': self.uri,
                'signal_2xx': self.signal_2xx(),
                'signal_4xx': self.signal_4xx(),
                'signal_5xx': self.signal_5xx(),
                'signal_total': self.signal_total(),
                'signal_timings': self.signal_timings()}


class EndpointSet:
    def __init__(self, name, endpoints):
        self.name = name
        self.endpoints = endpoints

    def signal_2xx(self):
        return 'sum(' + ','.join([endpoint.signal_2xx() for endpoint in self.endpoints]) + ')'

    def signal_4xx(self):
        return 'sum(' + ','.join([endpoint.signal_4xx() for endpoint in self.endpoints]) + ')'

    def signal_5xx(self):
        return 'sum(' + ','.join([endpoint.signal_5xx() for endpoint in self.endpoints]) + ')'

    def signal_total(self):
        return 'sum(' + ','.join([endpoint.signal_total() for endpoint in self.endpoints]) + ')'

    def to_jinja_dict(self):
        return {'name': self.name,
                'signal_total': self.signal_total(),
                'signal_4xx': self.signal_4xx(),
                'signal_5xx': self.signal_5xx(),
                'endpoints': [e.to_jinja_dict() for e in self.endpoints]}


def generate_endpoints_panel(name, editors_list, endpoint_set_list, qloud_project_id, env_list, default_env,
                             qloud_component=None, layout_height=1, layout_width=3, layout_columns_count=12,
                             header_height=0.2):
    return '''<< suggest.clear() >>
<< suggest.add_var("env") >>
<< suggest.set_choice_list("env", [{envs}]) >>
<% set env = env | default("{default_env}") %>

<% set q_prj = "{qloud_project_id}" ~ "." ~ env %>

<% set q_component = "{qloud_component}" %>
<% if q_component != '' %>
    <% set q_component = 'tier=' ~ q_component ~ '*' %>
<% endif %>
<% set app_tag = "itype=qloud;prj=" ~ q_prj ~ ";" ~ q_component %>

<% set main_layout = create_main_layout("vertical", default_height={layout_height}, default_width={layout_width}) %>
<% set endpoint_sets = [{endpoint_sets}] %>

{{
    "title": "{panel_name} (<< env >>)",
    "type": "panel",
    "editors": [{editors}],
    "charts": [
        <% for endpoint_set in endpoint_sets %>
            {{
                "type": "text",
                "text": "<< endpoint_set["name"] >>",
                << main_layout.coords(width={layout_columns_count}, height={header_height}) >>
            }},
            <% set primary_layout = main_layout.sub('flow', columns={layout_columns_count}) %>
            {{
                "title": "total requests",
                "type": "graphic",
                "stacked": true,
                "minValue": 0,
                "signals": [
                    <% for endpoint in endpoint_set["endpoints"] %>
                    {{
                        "title": "<< endpoint["name"] >>",
                        "tag": "<< app_tag >>",
                        "host": "QLOUD",
                        "name": "<< endpoint["signal_total"] >>",
                        "color": "<< make_color(endpoint["name"], "sparse") >>"
                    }}
                    << "," if not loop.last >>
                    <%- endfor %>
                ],
                << primary_layout.coords() >>
            }},
            {{
                "title": "2xx requests",
                "type": "graphic",
                "stacked": true,
                "minValue": 0,
                "signals": [
                    <% for endpoint in endpoint_set["endpoints"] %>
                    {{
                        "title": "<< endpoint["name"] >>",
                        "tag": "<< app_tag >>",
                        "host": "QLOUD",
                        "name": "<< endpoint["signal_2xx"] >>",
                        "color": "<< make_color(endpoint["name"], "sparse") >>"
                    }}
                    << "," if not loop.last >>
                    <%- endfor %>
                ],
                << primary_layout.coords() >>
            }},
            {{
                "title": "4xx requests",
                "type": "graphic",
                "stacked": true,
                "minValue": 0,
                "signals": [
                    <% for endpoint in endpoint_set["endpoints"] %>
                    <% set signal = endpoint["signal_4xx"] %>
                    {{
                        "title": "<< endpoint["name"] >>",
                        "tag": "<< app_tag >>",
                        "host": "QLOUD",
                        "name": "<< signal >>",
                        "color": "<< make_color(endpoint["name"], "sparse") >>"
                        <% set alert = list_alerts(signal_pattern=signal, hosts="QLOUD", itype="qloud", prj=q_prj, tier="{qloud_component}*").get() %>
                        <% if alert %>
                            ,"alertName": "<< alert >>"
                        <% endif %>
                    }}
                    << "," if not loop.last >>
                    <%- endfor %>
                ],
                << primary_layout.coords() >>
            }},
            {{
                "title": "5xx requests",
                "type": "graphic",
                "stacked": true,
                "minValue": 0,
                "signals": [
                    <% for endpoint in endpoint_set["endpoints"] %>
                    <% set signal = endpoint["signal_5xx"] %>
                    {{
                        "title": "<< endpoint["name"] >>",
                        "tag": "<< app_tag >>",
                        "host": "QLOUD",
                        "name": "<< signal >>",
                        "color": "<< make_color(endpoint["name"], "sparse") >>"
                        <% set alert = list_alerts(signal_pattern=signal, hosts="QLOUD", itype="qloud", prj=q_prj, tier="{qloud_component}*").get() %>
                        <% if alert %>
                            ,"alertName": "<< alert >>"
                        <% endif %>
                    }}
                    << "," if not loop.last >>
                    <%- endfor %>
                ],
                << primary_layout.coords() >>
            }},
            <% for endpoint in endpoint_set["endpoints"] %>
            {{
                "title": "<< endpoint["name"] >> timings, ms",
                "type": "graphic",
                "minValue": 0,
                "signals": [
                    <% for q in [50, 80, 90, 95, 98, 99] %>
                    {{
                        "title": "<< q >>",
                        "tag": "<< app_tag >>",
                        "host": "QLOUD",
                        "name": "or(quant(<< endpoint["signal_timings"] >>, << q >>), 0)",
                        "color": "<< make_color(q, start=50, stop=99, scheme="gradient") >>"
                   }}
                   << "," if not loop.last >>
                   <%- endfor %>
                ],
                << primary_layout.coords() >>
            }}
            << "," if not loop.last >>
            <%- endfor %>
        << "," if not loop.last >>
        <%- endfor %>
    ]
}}'''.format(qloud_project_id=qloud_project_id,
             qloud_component=qloud_component or '',
             panel_name=name,
             editors=','.join(['"' + env + '"' for env in editors_list]),
             envs=','.join(['"' + env + '"' for env in env_list]),
             default_env=default_env,
             layout_columns_count=layout_columns_count,
             header_height=header_height,
             layout_height=layout_height,
             layout_width=layout_width,
             endpoint_sets=', '.join([str(e.to_jinja_dict()) for e in endpoint_set_list]))
