from enum import Enum


class GC(Enum):
    G1 = 'g1'
    Parallel = 'parallel'
    Shenandoah = 'shenandoah'


class ExecutorType(Enum):
    FIXED = 'fixed'
    FORK_JOIN = 'fork-join'
    SCHEDULED = 'scheduled'


class Executor:
    def __init__(self, type, name):
        self.type = type
        self.name = name


def generate_jvm_panel(name, editors_list, qloud_project_id, env_list, default_env, qloud_component=None, gc=GC.G1,
                       executors=None, layout_height=1, layout_width=3, layout_columns_count=12):
    if executors is None:
        executors = []

    return '''<< suggest.clear() >>
<< suggest.add_var("env") >>
<< suggest.set_choice_list("env", [{envs}]) >>
<% set env = env | default("{default_env}") %>

<% set q_prj = "{qloud_project_id}" ~ "." ~ env %>

<% set q_component = "{qloud_component}" %>
<% if q_component != '' %>
    <% set q_component = 'tier=' ~ q_component ~ '*' %>
<% endif %>

<% set app_tag = "itype=qloud;prj=" ~ q_prj ~ ";" ~ q_component %>
<% set gc = "{gc}" %>
<% set executors = {executors} %>
<% set main_layout = create_main_layout("vertical", default_height={layout_height}, default_width={layout_width}) %>
{{
    "title": "{panel_name} (<< env >>)",
    "type": "panel",
    "editors": [{editors}],
    "charts": [
        <% set primary_layout = main_layout.sub('flow', columns={layout_columns_count}) %>
        {{
            "title": "Threads",
            "type": "graphic",
            "stacked": true,
            "minValue": 0,
            "signals": [
                <% for state in ["new", "runnable", "blocked", "waiting", "timed_waiting", "terminated"] %>
                {{
                    "title": "<< state >>",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "unistat-jvm_threads_states_<< state >>_axxx",
                    "color": "<< make_color(state, "sparse") >>"
                }}
                << "," if not loop.last >>
                <%- endfor %>
            ],
            << primary_layout.coords() >>
        }},
        {{
            "title": "Classes",
            "type": "graphic",
            "minValue": 0,
            "signals": [
                {{
                    "title": "Classes that are currently loaded in the JVM",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "unistat-jvm_classes_loaded_axxx",
                    "color": "#2bc687"
                }},
                {{
                    "title": "The total number of classes unloaded since the JVM has started",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "unistat-jvm_classes_unloaded_summ",
                    "color": "#37bff2"
                }}
            ],
            << primary_layout.coords() >>
        }},
        <% for buffer_type in ["direct", "mapped"] %>
        {{
            "title": "<< buffer_type >> memory",
            "type": "graphic",
            "minValue": 0,
            "links": [
                {{
                    "title": "Moar about buffers",
                    "url": "https://dzone.com/articles/understanding-java-buffer-pool"
                }}
            ],
            "signals": [
                {{
                    "title": "An estimate of the number of buffers in the pool",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "unistat-jvm_buffer_count_<< buffer_type >>_axxx",
                    "color": "<< make_color("count", "sparse") >>"
                }},
                {{
                    "title": "An estimate of the memory that the JVM is using for this buffer pool, Mb",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "conv(unistat-jvm_buffer_memory_used_<< buffer_type >>_axxx, Mi)",
                    "color": "<< make_color("memory_used", "sparse") >>"
                }},
                {{
                    "title": "An estimate of the total capacity of the buffers in this pool",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "conv(unistat-jvm_buffer_total_capacity_<< buffer_type >>_axxx, Mi)",
                    "color": "<< make_color("total_capacity", "sparse") >>"
                }}
            ],
            << primary_layout.coords() >>
        }},
        <%- endfor %>
        {{
            "title": "Non-heap memory usage, Mb",
            "type": "graphic",
            "minValue": 0,
            "signals": [
                <% for region in ["metaspace", "compressed_class_space"] %>
                {{
                    "title": "<< region >> used",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "conv(unistat-jvm_memory_used_nonheap_<< region >>_axxx, Mi)",
                    "color": "<< make_color(region ~ "used", "sparse") >>"
                }},
                {{
                    "title": "<< region >> committed",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "conv(unistat-jvm_memory_committed_nonheap_<< region >>_axxx, Mi)",
                    "color": "<< make_color(region ~ "committed", "sparse") >>"
                }},
                {{
                    "title": "<< region >> limit (0 - unlimited)",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "conv(max(unistat-jvm_memory_max_nonheap_<< region >>_axxx, 0), Mi)",
                    "color": "<< make_color(region ~ "limit", "sparse") >>"
                }}
                << "," if not loop.last >>
                <%- endfor %>
            ],
            << primary_layout.coords() >>
        }},
        {{
            "title": "Code cache memory usage, Mb",
            "type": "graphic",
            "minValue": 0,
            "signals": [
                <% for region in ["non_nmethods", "non_profiled_nmethods", "profiled_nmethods"] %>
                {{
                    "title": "<< region >> used",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "conv(unistat-jvm_memory_used_nonheap_codeheap_<< region >>_axxx, Mi)",
                    "color": "<< make_color(region ~ "used", "sparse") >>"
                    <% set signal_pattern = 'unistat-jvm_memory_used_nonheap_codeheap_' ~ region ~ '_axxx' %>
                    <% set alert = list_alerts(signal_pattern=signal_pattern, hosts="QLOUD", itype="qloud", prj=q_prj, tier="{qloud_component}*").get() %>
                    <% if alert %>
                        ,"alertName": "<< alert >>"
                    <% endif %>
                }},
                {{
                    "title": "<< region >> committed",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "conv(unistat-jvm_memory_committed_nonheap_codeheap_<< region >>_axxx, Mi)",
                    "color": "<< make_color(region ~ "committed", "sparse") >>"
                }},
                {{
                    "title": "<< region >> limit (0 - unlimited)",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "conv(max(unistat-jvm_memory_max_nonheap_codeheap_<< region >>_axxx, 0), Mi)",
                    "color": "<< make_color(region ~ "limit", "sparse") >>"
                }}
                << "," if not loop.last >>
                <%- endfor %>
            ],
            << primary_layout.coords() >>
        }}
        <% if gc == "g1" or gc == "parallel" %>
            <% if gc == "g1" %>
                <% set gc_name = "G1" %>
                <% set gc_alias = "g1" %>
            <% endif %>
            <% if gc == "parallel" %>
                <% set gc_name = "Parallel" %>
                <% set gc_alias = "ps" %>
            <% endif %>
        ,{{
            "title": "<< gc_name >> GC memory usage, Mb",
            "type": "graphic",
            "minValue": 0,
            "signals": [
                <% for region in ["eden_space", "survivor_space", "old_gen"] %>
                {{
                    "title": "<< region >> used",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "conv(unistat-jvm_memory_used_heap_<< gc_alias >>_<< region >>_axxx, Mi)",
                    "color": "<< make_color(region ~ "used", "sparse") >>"
                }},
                {{
                    "title": "<< region >> committed",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "conv(unistat-jvm_memory_committed_heap_<< gc_alias >>_<< region >>_axxx, Mi)",
                    "color": "<< make_color(region ~ "committed", "sparse") >>"
                }},
                {{
                    "title": "<< region >> limit (0 - unlimited)",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "conv(max(unistat-jvm_memory_max_heap_<< gc_alias >>_<< region >>_axxx, 0), Mi)",
                    "color": "<< make_color(region ~ "limit", "sparse") >>"
                }}
                << "," if not loop.last >>
                <%- endfor %>
            ],
            << primary_layout.coords() >>
        }},
        <% endif %>
        <% if gc == "shenandoah" %>
        ,{{
            "title": "Shenandoah memory usage, Mb",
            "type": "graphic",
            "minValue": 0,
            "signals": [
                {{
                    "title": "Used",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "conv(unistat-jvm_memory_used_heap_shenandoah_axxx, Mi)",
                    "color": "<< make_color("used", "sparse") >>"
                }},
                {{
                    "title": "Committed",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "conv(unistat-jvm_memory_committed_heap_shenandoah_axxx, Mi)",
                    "color": "<< make_color("committed", "sparse") >>"
                }},
                {{
                    "title": "Limit (0 - unlimited)",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "conv(max(unistat-jvm_memory_max_heap_shenandoah_axxx, 0), Mi)",
                    "color": "<< make_color("limit", "sparse") >>"
                }}
            ],
            << primary_layout.coords() >>
        }},
        <% endif %>
        {{
            "title": "GC stats, Mb",
            "type": "graphic",
            "minValue": 0,
            "signals": [
                {{
                    "title": "Max size of old gen",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "conv(unistat-jvm_gc_max_data_size_axxx, Mi)",
                    "color": "<< make_color("max_data_size", "sparse") >>"
                }},
                {{
                    "title": "Size of old gen after a full GC",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "conv(unistat-jvm_gc_live_data_size_axxx, Mi)",
                    "color": "<< make_color("live_data_size", "sparse") >>"
                }},
                {{
                    "title": "Count of positive increases in the size of the old gen before GC to after GC",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "conv(unistat-jvm_gc_memory_promoted_summ, Mi)",
                    "color": "<< make_color("memory_promoted", "sparse") >>"
                }},
                {{
                    "title": "Incremented for an increase in the size of the young gen after one GC to before the next",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "conv(unistat-jvm_gc_memory_allocated_summ, Mi)",
                    "color": "<< make_color("memory_allocated", "sparse") >>"
                }},
                {{
                    "title": "GC pause count",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "unistat-jvm_gc_pause_count_summ",
                    "color": "<< make_color("pause_count", "sparse") >>"
                }}
            ],
            << primary_layout.coords() >>
        }},
        {{
            "title": "GC pause time, ms",
            "type": "graphic",
            "minValue": 0,
            "signals": [
                <% for q in [50, 80, 90, 95, 98, 99] %>
                {{
                    "title": "<< q >>",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "or(quant(unistat-jvm_gc_pause_time_hgram, << q >>), 0)",
                    "color": "<< make_color(q, start=50, stop=99, scheme="gradient") >>"
                }}
                << "," if not loop.last >>
                <%- endfor %>
            ],
            << primary_layout.coords() >>
        }}
        <% for executor in executors %>
        <% set executor_name = executor["name"] %>
        <% set executor_type = executor["type"] %>
        ,{{
            "title": "<< executor["name"] >> executor stats",
            "type": "graphic",
            "minValue": 0,
            "signals": [
                {{
                    "title": "Submitted tasks count",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "unistat-executor_<< executor_name >>_count_summ",
                    "color": "<< make_color("count", "sparse") >>"
                }}

                <% if executor_type == "fixed" %>
                ,{{
                    "title": "The approximate total number of tasks that have completed execution",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "unistat-executor_completed_<< executor_name >>_summ",
                    "color": "<< make_color("completed", "sparse") >>"
                }},
                {{
                    "title": "The approximate number of threads that are actively executing tasks",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "unistat-executor_active_<< executor_name >>_axxx",
                    "color": "<< make_color("active", "sparse") >>"
                }},
                {{
                    "title": "The approximate number of tasks that are queued for execution",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "unistat-executor_queued_<< executor_name >>_axxx",
                    "color": "<< make_color("queued", "sparse") >>"
                }},
                {{
                    "title": "The number of additional elements that this queue can ideally accept without blocking",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "unistat-executor_queue_remaining_<< executor_name >>_axxx",
                    "color": "<< make_color("queue_remaining", "sparse") >>"
                }},
                {{
                    "title": "The current number of threads in the pool",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "unistat-executor_pool_size_<< executor_name >>_axxx",
                    "color": "<< make_color("pool_size", "sparse") >>"
                }}
                <% endif %>

                <% if executor_type == "fork-join" %>
                ,{{
                    "title": "Estimate of the total number of tasks stolen from one thread's work queue by another",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "unistat-executor_steals_<< executor_name >>_summ",
                    "color": "<< make_color("steals", "sparse") >>"
                }},
                {{
                    "title": "An estimate of the total number of tasks currently held in queues by worker threads",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "unistat-executor_queued_<< executor_name >>_axxx",
                    "color": "<< make_color("queued", "sparse") >>"
                }},
                {{
                    "title": "An estimate of the number of threads that are currently stealing or executing tasks",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "unistat-executor_active_<< executor_name >>_axxx",
                    "color": "<< make_color("active", "sparse") >>"
                }},
                {{
                    "title": "An estimate of the number of worker threads that are not blocked waiting to join tasks or for other managed synchronization threads",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "unistat-executor_running_<< executor_name >>_axxx",
                    "color": "<< make_color("running", "sparse") >>"
                }}
                <% endif %>
            ],
            << primary_layout.coords() >>
        }},
        {{
            "title": "<< executor_name >> executor tasks execution time, ms",
            "type": "graphic",
            "minValue": 0,
            "signals": [
                <% for q in [50, 80, 90, 95, 98, 99] %>
                {{
                    "title": "<< q >>",
                    "tag": "<< app_tag >>",
                    "host": "QLOUD",
                    "name": "or(quant(unistat-executor_<< executor_name >>_time_hgram, << q >>), 0)",
                    "color": "<< make_color(q, start=50, stop=99, scheme="gradient") >>"
                }}
                << "," if not loop.last >>
                <%- endfor %>
            ],
            << primary_layout.coords() >>
        }}
        <%- endfor %>
    ]
}}'''.format(qloud_project_id=qloud_project_id,
             qloud_component=qloud_component or '',
             panel_name=name,
             editors=','.join(['"' + env + '"' for env in editors_list]),
             envs=','.join(['"' + env + '"' for env in env_list]),
             default_env=default_env,
             gc=gc.value,
             executors=','.join([str({'type': e.type.value, 'name': e.name}) for e in executors]),
             layout_columns_count=layout_columns_count,
             layout_height=layout_height,
             layout_width=layout_width)
