#ifndef _YMODBLACKBOX_ERROR_H_
#define _YMODBLACKBOX_ERROR_H_

#include <ymod_blackbox/config.h>
#include <yplatform/exception.h>
#include <boost/assign.hpp>
#include <vector>

namespace ymod_blackbox {

enum err_code
{
    err_code_no_error = 0,
    // libblackbox2 throw bb::TempError - see ext description for detailed info.
    err_code_bb_temp,
    // libblackbox2 throw bb::FatalError - see ext description for detailed info.
    err_code_bb_fatal,
    err_code_network,
    // Response object is empty - see error for details.
    err_code_resp_fail,
    err_code_task_queue_overflow,
    err_code_proto_error,
    // Server return non-200 response code.
    err_code_server_code_error,
};

class error_category : public boost::system::error_category
{
public:
    const char* name() const noexcept
    {
        return "ymod_blackbox";
    }

    string message(int ev) const
    {
        const std::vector<string> error_names =
            boost::assign::list_of("no error")("blackbox temporary error")("blackbox fatal error")(
                "network error")("response is empty")("task queue overflow")(
                "http response format invalid")("server return non-200 response code");
        return error_names.at(ev);
    }
};

inline const error_category& category()
{
    static const error_category category;
    return category;
}

}

namespace boost { namespace system {

template <>
struct is_error_code_enum<ymod_blackbox::err_code>
{
    static const bool value = true;
};

}}

namespace ymod_blackbox {

struct error
{
    error() : code(err_code_no_error, category())
    {
    }

    error(err_code err, const string& ext_reason) : code(err, category()), ext_reason(ext_reason)
    {
    }

    operator bool() const
    {
        return code.value();
    }

    error_code code;
    string ext_reason;
};

inline error make_error(err_code err = err_code_no_error, const string& ext_reason = "")
{
    return error(err, ext_reason);
}

class auth_error : public yplatform::exception
{
public:
    explicit inline auth_error(const string& msg) : yplatform::exception("auth_error", msg)
    {
    }

    virtual inline ~auth_error() throw()
    {
    }
};

class auth_init_error : public auth_error
{
public:
    explicit inline auth_init_error(const string& msg) : auth_error(msg)
    {
    }

    virtual inline ~auth_init_error() throw()
    {
    }
};

class auth_timeout_error : public auth_error
{
public:
    explicit inline auth_timeout_error(const string& msg) : auth_error(msg)
    {
    }

    virtual inline ~auth_timeout_error() throw()
    {
    }
};

class auth_connect_error : public auth_error
{
public:
    explicit inline auth_connect_error(const string& msg) : auth_error(msg)
    {
    }

    virtual inline ~auth_connect_error() throw()
    {
    }
};

class auth_no_mail_error : public auth_error
{
public:
    explicit inline auth_no_mail_error(const string& msg) : auth_error(msg)
    {
    }

    virtual inline ~auth_no_mail_error() throw()
    {
    }
};

}

#endif // _YMODBLACKBOX_ERROR_H_
