#pragma once

#include <ymod_blackbox/config.h>
#include <boost/shared_ptr.hpp>
#include <set>
#include <list>
#include <deque>
#include <vector>

namespace ymod_blackbox {

// Blackbox db_field option.
struct db_field
{
    explicit db_field(const string& f) : field(f)
    {
    }
    string field;
};

// Blackbox request extended option.
struct option
{
    enum opt_t
    {
        regname,            // show name as used for registration
        get_social_aliases, // get social aliases info
        get_all_emails,     // get all known user emails
        get_yandex_emails,  // get all user Yandex emails
        get_default_email,  // get default email
    };
    option(opt_t o_) : o(o_)
    {
    }
    opt_t o;
};

using attribute = string;

typedef std::vector<option> options_list;
typedef std::vector<db_field> db_fields_list;
typedef std::vector<string> attribute_list;

class options_stream
{
public:
    options_stream() : impl_(new impl())
    {
    }
    options_stream& operator<<(const db_field& f)
    {
        impl_->fields_.push_back(f);
        return *this;
    }
    options_stream& operator<<(const option& o)
    {
        impl_->opts_.push_back(o);
        return *this;
    }
    options_stream& operator<<(const attribute& a)
    {
        impl_->attributes_.push_back(a);
        return *this;
    }

    const options_list& opts() const
    {
        return impl_->opts_;
    }

    const db_fields_list& fields() const
    {
        return impl_->fields_;
    }

    const attribute_list& attributes() const
    {
        return impl_->attributes_;
    }

private:
    struct impl
    {
        options_list opts_;
        db_fields_list fields_;
        attribute_list attributes_;
    };
    boost::shared_ptr<impl> impl_;
};

// Registration name.
struct regname_option
{
    string value;
};

struct uid_option
{
    string uid;
    bool hosted;
};

struct lite_uid_option
{
    string lite_uid;
    bool lite_account;
};

struct pdd_user_info_option
{
    string dom_id; // domain id
    string domain; // domain name
    string mx;     // mx field
};

struct karma_info_option
{
    string karma;
    string ban_time;
};

struct display_name_option
{
    string name;
    bool social;
    string soc_profile;  // social profile
    string soc_provider; // social provider
};

struct login_info_option
{
    string login;
    bool have_password;
};

struct oauth_info_option
{
    std::vector<string> scopes;
    string client_id;
    string client_name;
    bool client_is_yandex;
};

struct email_list_option
{
    // Email item.
    struct item
    {
        item(bool n, bool v, bool d, bool r, const string& addr, const string& dt)
            : native(n), validated(v), def(d), rpop(r), address(addr), date(dt)
        {
        }
        bool native;
        bool validated;
        bool def;
        bool rpop; // is added by rpopper
        string address;
        string date;
    };
    typedef std::vector<item> item_list;
    item_list items;
    unsigned def_i; // default email index
};

struct new_session_id_option
{
    string value;
};

struct auth_info_option
{
    bool social;
    // string auth age
    string age;
    // string profile identifier
    string profile_id;
};

struct mtoken_option
{
    string value;
};

struct connection_id_option
{
    string value;
};

// List of user aliases.
struct alias_list_option
{
    struct item
    {
        enum types
        {
            login = 1,
            mail = 2,
            narod = 3,
            lite = 4,
            pdd = 5,
            social = 6
        };
        item(types t, const string& a) : type(t), alias(a)
        {
        }
        types type;
        string alias; // alias value
    };

    typedef std::vector<item> item_list;
    item_list items;
};

}
