#include "bbclient.h"
#include <boost/assign/std.hpp>
#include <boost/assign/list_of.hpp>
#include <boost/lexical_cast.hpp>

#include <pa/async.h>

namespace ymod_blackbox {

namespace {
template <typename BBRequest>
yhttp::request to_http_request(BBRequest&& bbreq)
{
    return yhttp::request::POST(std::move(bbreq.uri_), boost::move(bbreq.postData_));
}
template <>
yhttp::request to_http_request<string>(string&& bbreq)
{
    return yhttp::request::POST(std::move(bbreq), string());
}
}

bbclient::bbclient(
    boost::asio::io_service& io,
    std::shared_ptr<ymod_httpclient::cluster_call> http_client,
    const settings& st)
    : st_(st), http_client_(http_client)
{
    rc = ymod_ratecontroller::create_controller(io, st.pool_size, st.max_queue_size);
}

bb::Options bbclient::fill_options(
    const options_list& opts,
    const db_fields_list& fields,
    const attribute_list& attributes,
    const address& addr)
{
    bb::Options opt;
    for (options_list::const_iterator i = opts.begin(), i_end = opts.end(); i != i_end; ++i)
    {
        switch (i->o)
        {
        case option::regname:
            opt << bb::optRegname;
            break;
        case option::get_social_aliases:
            opt << bb::optGetSocialAliases;
            break;
        case option::get_all_emails:
            opt << bb::optGetAllEmails;
            break;
        case option::get_yandex_emails:
            opt << bb::optGetYandexEmails;
            break;
        case option::get_default_email:
            opt << bb::optGetDefaultEmail;
            break;
        default:
            break;
        }
    }
    bb::DBFields db_fields;
    for (db_fields_list::const_iterator i = fields.begin(), i_end = fields.end(); i != i_end; ++i)
    {
        db_fields << i->field;
    }
    opt << db_fields;

    bb::Attributes db_attributes;
    for (auto&& attr : attributes)
    {
        db_attributes << attr;
    }
    opt << db_attributes;
    if (addr.port)
    {
        opt << bb::Option("user_port", std::to_string(addr.port));
    }
    return opt;
}

void bbclient::async_mhost_find(
    yplatform::task_context_ptr ctx,
    const string& scope,
    const string& prio,
    const mhost_find_callback& cb)
{
    profile_data pdata("", "", "mhost_find");
    auto req = to_http_request(bb::MailHostFindRequest(scope, prio));
    auto self = shared_from_this();
    async_schedule_http(
        ctx,
        std::move(req),
        std::bind(
            &bbclient::parse_bb_response<mhost_find_response, mhost_find_callback>,
            self,
            p::_1,
            p::_2,
            pdata,
            std::move(cb)));
}

void bbclient::async_info(
    yplatform::task_context_ptr ctx,
    const string& uid,
    const address& addr,
    const info_callback& cb,
    const options_list& opts,
    const db_fields_list& fields,
    const attribute_list& attributes)
{
    profile_data pdata(addr.ip, uid, "info");
    auto req = to_http_request(
        bb::InfoRequest(uid, addr.ip, fill_options(opts, fields, attributes, addr)));
    auto self = shared_from_this();
    async_schedule_http(
        ctx,
        std::move(req),
        std::bind(
            &bbclient::parse_bb_response<info_response, info_callback>,
            self,
            p::_1,
            p::_2,
            pdata,
            std::move(cb)));
}

void bbclient::async_info_suid(
    yplatform::task_context_ptr ctx,
    const string& suid,
    const string& sid,
    const address& addr,
    const info_callback& cb,
    const options_list& opts,
    const db_fields_list& fields,
    const attribute_list& attributes)
{
    profile_data pdata(addr.ip, suid, "info_suid");
    auto req = to_http_request(bb::InfoRequest(
        bb::SuidSid(suid, sid), addr.ip, fill_options(opts, fields, attributes, addr)));
    auto self = shared_from_this();
    async_schedule_http(
        ctx,
        std::move(req),
        std::bind(
            &bbclient::parse_bb_response<info_response, info_callback>,
            self,
            p::_1,
            p::_2,
            pdata,
            std::move(cb)));
}

void bbclient::async_info_login(
    yplatform::task_context_ptr ctx,
    const string& login,
    const string& sid,
    const address& addr,
    const info_callback& cb,
    const options_list& opts,
    const db_fields_list& fields,
    const attribute_list& attributes)
{
    profile_data pdata(addr.ip, "", "login");
    auto req = to_http_request(bb::InfoRequest(
        bb::LoginSid(login, sid), addr.ip, fill_options(opts, fields, attributes, addr)));
    auto self = shared_from_this();
    async_schedule_http(
        ctx,
        std::move(req),
        std::bind(
            &bbclient::parse_bb_response<info_response, info_callback>,
            self,
            p::_1,
            p::_2,
            pdata,
            std::move(cb)));
}

void bbclient::async_login(
    yplatform::task_context_ptr ctx,
    const string& login,
    const string& sid,
    const string& password,
    const address& addr,
    const login_callback& cb,
    const options_list& opts,
    const db_fields_list& fields,
    const attribute_list& attributes)
{
    profile_data pdata(addr.ip, "", "login2");
    auto req = to_http_request(bb::LoginRequest(
        bb::LoginSid(login, sid),
        password,
        st_.authtype,
        addr.ip,
        fill_options(opts, fields, attributes, addr)));
    auto self = shared_from_this();
    async_schedule_http(
        ctx,
        std::move(req),
        std::bind(
            &bbclient::parse_bb_response<login_response, login_callback>,
            self,
            p::_1,
            p::_2,
            pdata,
            std::move(cb)));
}

void bbclient::async_session_id(
    yplatform::task_context_ptr ctx,
    const string& session_id,
    const string& hostname,
    const address& addr,
    const session_id_callback& cb,
    const options_list& opts,
    const db_fields_list& fields,
    const attribute_list& attributes)
{
    profile_data pdata(addr.ip, "", "session_id");
    auto req = to_http_request(bb::SessionIDRequest(
        session_id, hostname, addr.ip, fill_options(opts, fields, attributes, addr)));
    auto self = shared_from_this();
    async_schedule_http(
        ctx,
        std::move(req),
        std::bind(
            &bbclient::parse_bb_response<session_id_response, session_id_callback>,
            self,
            p::_1,
            p::_2,
            pdata,
            std::move(cb)));
}

void bbclient::async_oauth(
    yplatform::task_context_ptr ctx,
    const string& oauthToken,
    const address& addr,
    const session_id_callback& cb,
    const options_list& opts,
    const db_fields_list& fields,
    const attribute_list& attributes)
{
    profile_data pdata(addr.ip, "", "oauth");
    auto req = to_http_request(
        bb::OAuthRequestSecure(oauthToken, addr.ip, fill_options(opts, fields, attributes, addr)));
    auto self = shared_from_this();
    async_schedule_http(
        ctx,
        std::move(req),
        std::bind(
            &bbclient::parse_bb_response<session_id_response, session_id_callback>,
            self,
            p::_1,
            p::_2,
            pdata,
            std::move(cb)));
}

bbclient::task_queue_stats bbclient::get_stats()
{
    return { rc->running_tasks_count(), rc->queue_size() };
}

void bbclient::log_profile(const profile_data& pdata)
{
    if (st_.log_pa)
    {
        using yplatform::time_traits::duration_cast;
        using yplatform::time_traits::milliseconds;
        auto req_time = duration_cast<milliseconds>(profile_data::now() - pdata.start).count();
        pa::async_profiler::add(
            pa::passport, pdata.host, pdata.req, pdata.context, static_cast<uint32_t>(req_time));
    }
}

}
