#pragma once

#include "settings.h"
#include "task_queue.h"
#include <ymod_blackbox/client.h>
#include <ymod_httpclient/client.h>
#include <yandex/blackbox/blackbox2.h>
#include <ymod_ratecontroller/rate_controller.h>
#include <yplatform/util/execution_holder.h>

namespace ymod_blackbox {

namespace p = std::placeholders;

class bbclient : public std::enable_shared_from_this<bbclient>
{
    struct profile_data
    {
        using ptime_t = yplatform::time_traits::time_point;

        static ptime_t now()
        {
            return yplatform::time_traits::clock::now();
        }

        string host;
        string req;
        string context;

        ptime_t start;

        profile_data(string const& host, string const& req, string const& suid)
            : host(host), req(req), context(suid), start(now())
        {
        }
    };

    struct task_queue_stats
    {
        std::size_t running;
        std::size_t in_queue;
    };

    template <typename Callback>
    struct request_data
    {
        request_data() : retry(0)
        {
        }

        unsigned retry;
        string request;
        string post;
        Callback cb;
    };
    typedef request_data<info_callback> info_data;
    typedef request_data<login_callback> login_data;
    typedef request_data<session_id_callback> session_id_data;
    typedef request_data<mhost_find_callback> mhost_find_data;

public:
    bbclient(
        boost::asio::io_service& io,
        std::shared_ptr<ymod_httpclient::cluster_call> http_client,
        const settings& st);

    void async_mhost(const string& scope, const string& prio, const mhost_find_callback& cb);

    void async_info(
        yplatform::task_context_ptr ctx,
        const string& uid,
        const address& addr,
        const info_callback& cb,
        const options_list& opts,
        const db_fields_list& fields,
        const attribute_list& attributes);

    void async_info_suid(
        yplatform::task_context_ptr ctx,
        const string& suid,
        const string& sid,
        const address& addr,
        const info_callback& cb,
        const options_list& opts,
        const db_fields_list& fields,
        const attribute_list& attributes);

    void async_info_login(
        yplatform::task_context_ptr ctx,
        const string& login,
        const string& sid,
        const address& addr,
        const info_callback& cb,
        const options_list& opts,
        const db_fields_list& fields,
        const attribute_list& attributes);

    void async_login(
        yplatform::task_context_ptr ctx,
        const string& login,
        const string& sid,
        const string& password,
        const address& addr,
        const login_callback& cb,
        const options_list& opts,
        const db_fields_list& fields,
        const attribute_list& attributes);

    void async_oauth(
        yplatform::task_context_ptr ctx,
        const string& oauthToken,
        const address& addr,
        const session_id_callback& cb,
        const options_list& opts,
        const db_fields_list& fields,
        const attribute_list& attributes);

    void async_session_id(
        yplatform::task_context_ptr ctx,
        const string& session_id,
        const string& hostname,
        const address& addr,
        const session_id_callback& cb,
        const options_list& opts,
        const db_fields_list& fields,
        const attribute_list& attributes);

    void async_mhost_find(
        yplatform::task_context_ptr ctx,
        const string& scope,
        const string& prio,
        const mhost_find_callback& cb);

    task_queue_stats get_stats();

private:
    bb::Options fill_options(
        const options_list& opts,
        const db_fields_list& fields,
        const attribute_list& attributes,
        const address& addr);

    void log_profile(const profile_data& pdata);

    template <typename Handler>
    void async_schedule_http(
        yplatform::task_context_ptr ctx,
        yhttp::request req,
        Handler&& handler);

    template <typename Handler>
    void async_run_http(
        yplatform::task_context_ptr ctx,
        yhttp::request req,
        yplatform::execution_holder_ptr rc_holder,
        Handler&& handler);

    template <typename Handler>
    void handle_http_request(const error_code& ec, yhttp::response resp, Handler&& handler);

    template <typename Info, typename Callback>
    void parse_bb_response(
        const error_code& ec,
        yhttp::response resp,
        const profile_data& pdata,
        const Callback& cb);

    settings st_;
    std::shared_ptr<ymod_httpclient::cluster_call> http_client_;
    ymod_ratecontroller::rate_controller_ptr rc;
};

template <typename Handler>
void bbclient::async_schedule_http(
    yplatform::task_context_ptr ctx,
    yhttp::request req,
    Handler&& handler)
{
    rc->post(
        ctx,
        [this, self = shared_from_this(), ctx, req = std::move(req), h = handler](
            error_code err, ymod_ratecontroller::completion_handler compl_handler) {
            auto rc_holder = std::make_shared<yplatform::execution_holder>(compl_handler);
            if (err)
            {
                yplatform::safe_call(h, err, yhttp::response());
            }
            else
            {
                async_run_http(ctx, std::move(req), rc_holder, std::move(h));
            }
        });
}

template <typename Handler>
void bbclient::async_run_http(
    yplatform::task_context_ptr ctx,
    yhttp::request req,
    yplatform::execution_holder_ptr rc_holder,
    Handler&& handler)
{
    // Copy the original request.
    auto self = shared_from_this();
    auto cb = [this, self, rc_holder, h = std::forward<Handler>(handler)](
                  const error_code& ec, yhttp::response resp) {
        handle_http_request(ec, resp, std::move(h));
    };
    http_client_->async_run(ctx, std::move(req), std::move(cb));
}

template <typename Handler>
void bbclient::handle_http_request(const error_code& ec, yhttp::response resp, Handler&& handler)
{
    yplatform::safe_call(handler, ec, std::move(resp));
}

template <typename Info, typename Callback>
void bbclient::parse_bb_response(
    const error_code& ec,
    yhttp::response resp,
    const profile_data& pdata,
    const Callback& cb)
{
    log_profile(pdata);

    if (ec)
    {
        yplatform::safe_call(cb, make_error(err_code_network, ec.message()), Info());
        return;
    }

    try
    {
        auto response = Info(std::move(resp.body));
        yplatform::safe_call(cb, make_error(), response);
    }
    catch (const bb::FatalError& err)
    {
        return yplatform::safe_call(cb, make_error(err_code_bb_fatal, err.what()), Info());
    }
    catch (const bb::TempError& err)
    {
        return yplatform::safe_call(cb, make_error(err_code_bb_temp, err.what()), Info());
    }
}

}
