#include "module.h"

#include "settings.h"

#include <ymod_httpclient/client.h>
#include <yplatform/time_traits.h>

namespace ymod_blackbox {

namespace detail {

static bool has_dot_in_login(string const& email)
{
    string::size_type p = email.find('.');
    if (p == string::npos) return false;

    string::size_type d = email.find('@');
    return (d == string::npos || p < d);
}

}

void auth_impl::init(const yplatform::ptree& xml)
{
    settings st;
    st.parse_ptree(xml);
    default_domain_ = xml.get("default_domain", "");

    auto reactor = yplatform::find<yplatform::reactor, std::shared_ptr>(xml.get<string>("reactor"));
    auto http_client = std::make_shared<yhttp::cluster_client>(*reactor, st.http);
    impl_ = std::make_shared<bbclient>(*(reactor->io()), http_client, st);
    fake_context_ = boost::make_shared<yplatform::task_context>("bb-module-ctx");
}

yplatform::future::future<response> auth_impl::authenticate(const request& req)
{
    req_data_ptr data(new req_data);
    data->req = req;

    if (!req.service.empty() && req.suid.empty() && req.uid.empty())
    {
        data->f_login.append("-");
        data->f_suid.append("-");
        data->f_db_id.append("-");
    }
    else
    {
        string sid = boost::lexical_cast<string>(req.sid);
        data->f_login.append(sid);
        data->f_suid.append(sid);
        data->f_db_id.append(sid);
    }
    std::set<std::string> fields;
    fields.insert(data->f_login);
    if (req.suid.empty()) fields.insert(data->f_suid);
    fields.insert(data->f_db_id);

    // special actions to know the email
    fields.insert("subscription.login.2");
    if (req.suid.empty()) fields.insert("subscription.login.8");

    if (req.load_timezone) fields.insert("account_info.tz.uid");
    if (req.load_language) fields.insert("userinfo.lang.uid");
    if (req.suid.empty() && !req.no_password) fields.insert("subscription.login_rule.8");
    for (std::vector<unsigned>::const_iterator i = req.sids_to_check.begin(),
                                               i_end = req.sids_to_check.end();
         i != i_end;
         ++i)
    {
        data->sid_fields.push_back("subscription.suid." + boost::lexical_cast<string>(*i));
        fields.insert(data->sid_fields.back());
    }
    options_stream opt_stream;

    for (std::set<std::string>::const_iterator i = fields.begin(), i_end = fields.end(); i != i_end;
         ++i)
    {
        opt_stream << db_field(*i);
    }

    for (options_list::const_iterator i = req.opt_list.begin(), i_end = req.opt_list.end();
         i != i_end;
         ++i)
    {
        opt_stream << *i;
    }

    if (!req.uid.empty())
    {
        impl_->async_info(
            fake_context_,
            req.uid,
            req.address,
            boost::bind(&auth_impl::info_cb, shared_from(this), _1, _2, data),
            opt_stream.opts(),
            opt_stream.fields(),
            opt_stream.attributes());
    }
    else if (!req.suid.empty())
    {
        impl_->async_info_suid(
            fake_context_,
            req.suid,
            req.service,
            req.address,
            boost::bind(&auth_impl::info_cb, shared_from(this), _1, _2, data),
            opt_stream.opts(),
            opt_stream.fields(),
            opt_stream.attributes());
    }
    else if (req.no_password)
    {
        impl_->async_info_login(
            fake_context_,
            req.username,
            req.service,
            req.address,
            boost::bind(&auth_impl::info_cb, shared_from(this), _1, _2, data),
            opt_stream.opts(),
            opt_stream.fields(),
            opt_stream.attributes());
    }
    else
    {
        impl_->async_login(
            fake_context_,
            req.username,
            req.service,
            req.password,
            req.address,
            boost::bind(&auth_impl::login_cb, shared_from(this), _1, _2, data),
            opt_stream.opts(),
            opt_stream.fields(),
            opt_stream.attributes());
    }

    return data->prom;
}

void auth_impl::info_cb(const error& e, const info_response& info, const req_data_ptr& data)
{
    auth_cb(e, info, data);
}

yplatform::future::future<db_host_map> auth_impl::mhost_find(
    yplatform::task_context_ptr ctx,
    const string& scope,
    const string& prio)
{
    yplatform::future::promise<db_host_map> prom;
    impl_->async_mhost_find(
        ctx,
        scope,
        prio,
        boost::bind(&auth_impl::mhost_find_cb, shared_from(this), ctx, prom, _1, _2));
    return prom;
}

void auth_impl::login_cb(const error& e, const login_response& info, const req_data_ptr& data)
{
    if (!e && info.status != login_response::valid)
    {
        YLOG_CTX_LOCAL(data->req.context, info)
            << "auth_impl::login_cb failure, reason: '" << info.message() << "'";
        response resp;
        resp.success = false;
        resp.error = e.code.value();
        resp.err_str = info.message();
        resp.err_verbose = info.message();
        karma_info_option karma_opt;
        try
        {
            const bbresponse<bb::LoginResp>& karma_info = info;
            karma_info >> karma_opt;
            resp.karma = karma_opt.karma;
            resp.ban_time = karma_opt.ban_time;
        }
        catch (...)
        {
        }
        data->prom.set(resp);
    }
    else
    {
        auth_cb(e, info, data);
    }
}

template <typename T>
void auth_impl::auth_cb(const error& e, const bbresponse<T>& info, const req_data_ptr& data)
{
    response resp;
    if (e)
    {
        YLOG_CTX_LOCAL(data->req.context, info)
            << "auth_impl::auth_cb failure: sys='" << e.code.message() << "', ext='" << e.ext_reason
            << "'";
        resp.success = false;
        resp.error = e.code.value();
        resp.err_str = e.code.message();
        resp.err_verbose = e.ext_reason;
    }
    else
    {
        resp.success = true;
        resp.error = 0;
        resp.err_verbose = resp.err_str = std::string();
        resp.email = info["subscription.login.8"];
        if (!detail::has_dot_in_login(resp.email)) resp.email = info["subscription.login.2"];
        if (resp.email.find('@') == string::npos)
        {
            std::size_t pos;
            if ((pos = data->req.username.find('@')) != string::npos)
            {
                resp.email.append(data->req.username, pos, data->req.username.length());
            }
            else
            {
                if (default_domain_.size() > 0 && default_domain_[0] != '@') resp.email.append("@");
                resp.email.append(default_domain_);
            }
        }
        uid_option uid_opt;
        karma_info_option karma_opt;
        info >> uid_opt;
        info >> karma_opt;
        resp.uid = uid_opt.uid;
        resp.login = info[data->f_login];
        resp.suid = data->req.suid.empty() ? info[data->f_suid] : data->req.suid;
        resp.storage = info[data->f_db_id];
        resp.karma = karma_opt.karma;
        resp.ban_time = karma_opt.ban_time;
        resp.raw = info.raw_response();

        if (data->req.opt_list.size() != 0)
        {
            email_list_option email_opt;
            info >> email_opt;
            resp.email_list = email_opt;
        }
        if (data->req.load_timezone) resp.timezone = info["account_info.tz.uid"];
        if (data->req.load_language) resp.language = info["userinfo.lang.uid"];
        if (data->req.suid.empty() && !data->req.no_password)
        {
            try
            {
                int login_rule_val = boost::lexical_cast<int>(info["subscription.login_rule.8"]);
                resp.user_blocked = (login_rule_val & 0x04);
            }
            catch (...)
            {
            }
        }
        for (std::size_t i = 0; i < data->sid_fields.size(); ++i)
        {
            if (info[data->sid_fields[i]].empty()) continue;
            resp.found_sids.insert(data->req.sids_to_check[i]);
        }
    }
#if 0
  YLOG_CTX_LOCAL(data->req.context, debug) << "auth_impl::authenticate_i: "
      << "login '" << data->req.username
      << "': result: " << (resp.success ? "OK" : "FAIL")
      << ", uid=" << resp.uid
      << " , suid=" << resp.suid
      << ", storage=" << resp.storage;
#endif
    data->prom.set(resp);
}

void auth_impl::mhost_find_cb(
    yplatform::task_context_ptr ctx,
    yplatform::future::promise<db_host_map> prom,
    const error& e,
    const mhost_find_response& info)
{
    if (e)
    {
        YLOG_CTX_LOCAL(ctx, error) << "auth_impl::mhost_find_cb failure: sys='" << e.code.message()
                                   << "', ext='" << e.ext_reason << "'";
        prom.set_exception(ymod_blackbox::auth_error(e.code.message()));
        return;
    }
    typedef mhost_find_response::item_list::const_iterator host_iter;
    db_host_map result;
    for (host_iter i = info.items.begin(), i_end = info.items.end(); i != i_end; ++i)
    {
        result[i->db_id] = ::atoi(i->priority.c_str());
    }
    prom.set(result);
}

}
