#include <ymod_blackbox/response.h>
#include <yplatform/util/split.h>
#include <boost/algorithm/string.hpp>

namespace ymod_blackbox {

namespace {
template <typename Dest>
std::unique_ptr<Dest> parse(const string& source);

template <>
std::unique_ptr<bb::Response> parse<bb::Response>(const string& source)
{
    return bb::InfoResponse(source);
}

template <>
std::unique_ptr<bb::LoginResp> parse<bb::LoginResp>(const string& source)
{
    return bb::LoginResponse(source);
}

template <>
std::unique_ptr<bb::SessionResp> parse<bb::SessionResp>(const string& source)
{
    return bb::SessionIDResponse(source);
}

template <>
std::unique_ptr<bb::HostResp> parse<bb::HostResp>(const string& source)
{
    return bb::MailHostResponse(source);
}
}

template <typename BBType>
bbresponse<BBType>::bbresponse()
{
}

template <typename BBType>
bbresponse<BBType>::bbresponse(string&& raw_response)
    : raw_response_(std::move(raw_response)), native_response_(parse<BBType>(raw_response_))
{
}

template <typename BBType>
const bbresponse<BBType>& bbresponse<BBType>::operator>>(regname_option& o) const
{
    o.value = bb::Regname(&native_response()).value_;
    return *this;
}

template <typename BBType>
const bbresponse<BBType>& bbresponse<BBType>::operator>>(uid_option& o) const
{
    bb::Uid u(&native_response());
    o.hosted = u.hosted_;
    o.uid = u.uid_;
    return *this;
}

template <typename BBType>
const bbresponse<BBType>& bbresponse<BBType>::operator>>(lite_uid_option& o) const
{
    bb::LiteUid u(&native_response());
    o.lite_account = u.liteAccount_;
    o.lite_uid = u.liteUid_;
    return *this;
}

template <typename BBType>
const bbresponse<BBType>& bbresponse<BBType>::operator>>(pdd_user_info_option& o) const
{
    bb::PDDUserInfo u(&native_response());
    o.dom_id = u.domId_;
    o.domain = u.domain_;
    o.mx = u.mx_;
    return *this;
}

template <typename BBType>
const bbresponse<BBType>& bbresponse<BBType>::operator>>(karma_info_option& o) const
{
    bb::KarmaInfo u(&native_response());
    o.ban_time = u.banTime_;
    o.karma = u.karma_;
    return *this;
}

template <typename BBType>
const bbresponse<BBType>& bbresponse<BBType>::operator>>(display_name_option& o) const
{
    bb::DisplayNameInfo u(&native_response());
    o.name = u.name_;
    o.soc_profile = u.socProfile_;
    o.soc_provider = u.socProvider_;
    o.social = u.social_;
    return *this;
}

template <typename BBType>
const bbresponse<BBType>& bbresponse<BBType>::operator>>(login_info_option& o) const
{
    bb::LoginInfo u(&native_response());
    o.have_password = u.havePassword_;
    o.login = u.login_;
    return *this;
}

template <typename BBType>
const bbresponse<BBType>& bbresponse<BBType>::operator>>(email_list_option& o) const
{
    bb::EmailList u(&native_response());
    unsigned def_i = 0;
    o.items.reserve(u.size());
    for (auto& host : u.getEmailItems())
    {
        if (&host == u.getDefaultItem()) o.def_i = def_i;
        o.items.push_back(email_list_option::item(
            host.native_, host.validated_, host.default_, host.rpop_, host.address_, host.date_));
    }
    return *this;
}

template <typename BBType>
const bbresponse<BBType>& bbresponse<BBType>::operator>>(alias_list_option& o) const
{
    bb::AliasList u(&native_response());
    o.items.reserve(u.size());
    for (bb::AliasList::ListType::const_iterator i = u.getAliases().begin(),
                                                 i_end = u.getAliases().end();
         i != i_end;
         ++i)
    {
        auto item_type = static_cast<alias_list_option::item::types>(i->type_);
        o.items.push_back(alias_list_option::item(item_type, i->alias_));
    }
    return *this;
}

template <typename BBType>
const string& bbresponse<BBType>::operator[](const string& field_name) const
{
    if (!db_fields_) db_fields_ = bb::DBFields(&native_response());
    return db_fields_->get(field_name);
}

template <typename BBType>
const string& bbresponse<BBType>::message() const
{
    return native_response_->message();
}

template <typename BBType>
const BBType& bbresponse<BBType>::native_response() const
{
    return *native_response_;
}

template <typename BBType>
const string& bbresponse<BBType>::raw_response() const
{
    return raw_response_;
}

login_response::login_response()
{
}

login_response::login_response(string&& source) : bbresponse(std::move(source))
{
    status = static_cast<login_status>(native_response().status());
}

const login_response& login_response::operator>>(connection_id_option& o) const
{
    o.value = bb::ConnectionId(&native_response());
    return *this;
}

session_id_response::session_id_response()
{
}

session_id_response::session_id_response(string&& source) : bbresponse(std::move(source))
{
    age = native_response().age();
    is_lite = native_response().isLite();
    status = static_cast<session_id_status>(native_response().status());
}

const session_id_response& session_id_response::operator>>(new_session_id_option& o) const
{
    o.value = bb::NewSessionId(&native_response()).value_;
    return *this;
}

const session_id_response& session_id_response::operator>>(auth_info_option& o) const
{
    bb::AuthInfo u(&native_response());
    o.age = u.age_;
    o.profile_id = u.profileId_;
    o.social = u.social_;
    return *this;
}

const session_id_response& session_id_response::operator>>(oauth_info_option& o) const
{
    bb::OAuthInfo u(&native_response());
    const auto scopes = u.getInfo().find("scope");
    if (scopes != u.getInfo().end())
    {
        o.scopes = yplatform::util::split(scopes->second, " ");
    }

    const auto is_yandex = u.getInfo().find("client_is_yandex");
    o.client_is_yandex = (is_yandex != u.getInfo().end()) && (is_yandex->second == "1");

    const auto iclient_id = u.getInfo().find("client_id");
    if (iclient_id != u.getInfo().end()) o.client_id = iclient_id->second;

    const auto iclient_name = u.getInfo().find("client_name");
    if (iclient_name != u.getInfo().end()) o.client_name = iclient_name->second;

    return *this;
}

const session_id_response& session_id_response::operator>>(connection_id_option& o) const
{
    o.value = bb::ConnectionId(&native_response());
    return *this;
}

mhost_find_response::mhost_find_response()
{
}

mhost_find_response::mhost_find_response(string&& raw_response)
    : raw_response_(std::move(raw_response)), native_response_(parse<bb::HostResp>(raw_response_))
{
    message = native_response().message();
    status = static_cast<mhost_status>(native_response().status());
    bb::MailHostsList list(&native_response());
    items.reserve(list.size());
    for (auto& host : list.getHosts())
    {
        items.push_back(mhost_find_response::item(
            host.hostId_,
            host.hostIp_,
            host.hostName_,
            host.dbId_,
            host.sid_,
            host.priority_,
            host.mx_));
    }
}

const bb::HostResp& mhost_find_response::native_response() const
{
    return *native_response_;
}

const string& mhost_find_response::raw_response() const
{
    return raw_response_;
}

template class bbresponse<bb::Response>;
template class bbresponse<bb::LoginResp>;
template class bbresponse<bb::SessionResp>;

}
