#ifndef _YMOD_CACHE_CACHE_H_
#define _YMOD_CACHE_CACHE_H_

#include <ymod_cache/config.h>
#include <ymod_cache/error.h>

#include <yplatform/module.h>
#include <yplatform/task_context.h>
#include <yplatform/future/future.hpp>
#include <yplatform/zerocopy/streambuf.h>

#include <boost/optional.hpp>
#include <ostream>
#include <boost/make_shared.hpp>

namespace ymod_cache {

typedef boost::optional<segment> segment_opt;

struct base_result {
    error_code error = error_code::none;

    base_result() = default;
    base_result(error_code error) : error(error) {}
    virtual ~base_result() {}

    virtual operator bool() const { return error == error_code::none; }
};

template <class T>
struct result : base_result {
    T value;

    result() = default;
    result(const T& value)
        : base_result(error_code::none), value(value) {}
    result(error_code error)
        : base_result(error) {}
    result(error_code::code error)
        : base_result(error) {}
};

template <>
struct result<void> : base_result {
    result(error_code error = error_code::none) : base_result(error) {}
};

template <>
struct result<segment_opt> : base_result {
    segment_opt value;

    result() = default;
    result(const segment& value)
        : base_result(error_code::none), value(value) {}
    result(const boost::none_t& value)
        : base_result(error_code::none), value(value) {}
    result(const segment_opt& value)
        : base_result(error_code::none), value(value) {}
    result(error_code error)
        : base_result(error) {}

    virtual operator bool() const {
        return error == error_code::none && value;
    }
};

typedef result<segment_opt> segment_result;
typedef result<void> void_result;
typedef result<bool> bool_result;

typedef yplatform::future::future<segment_result> future_segment;
typedef yplatform::future::future<void_result> future_result;
typedef yplatform::future::future<bool_result> future_bool;

namespace detail {

template <class T>
segment make_segment(const T& x)
{
    yplatform::zerocopy::streambuf buffer;
    std::ostream stream(&buffer);
    stream << x << std::flush;
    return buffer.detach(buffer.end());
}

} // namespace detail

using boost::make_shared;

class cache : public yplatform::module
{
public:
    cache()
    {}
    virtual ~cache()
    {}

    virtual future_result set(
        yplatform::task_context_ptr ctx,
        const segment& key,
        const segment& value) = 0;

    template <class K, class V>
    future_result set(
        yplatform::task_context_ptr ctx,
        const K& key,
        const V& value)
    {
        return set(ctx, detail::make_segment(key), detail::make_segment(value));
    }

    virtual future_segment get(
        yplatform::task_context_ptr ctx,
        const segment& key) = 0;

    template <class K>
    future_segment get(
        yplatform::task_context_ptr ctx,
        const K& key)
    {
        return get(ctx, detail::make_segment(key));
    }

    virtual future_bool has(
        yplatform::task_context_ptr ctx,
        const segment& key) = 0;

    template <class K>
    future_bool has(
        yplatform::task_context_ptr ctx,
        const K& key)
    {
        return has(ctx, detail::make_segment(key));
    }

    virtual future_result remove(
        yplatform::task_context_ptr ctx,
        const segment& key) = 0;

    template <class K>
    future_result remove(
        yplatform::task_context_ptr ctx,
        const K& key)
    {
        return remove(ctx, detail::make_segment(key));
    }
};

}

#endif // _YMOD_CACHE_CACHE_H_
