#include "local_cache_impl.h"
#include <ymod_cache/error.h>

#include <boost/functional/hash.hpp>

namespace yplatform { namespace zerocopy {

std::size_t hash_value(const segment& seg)
{
    return boost::hash_range(seg.begin(), seg.end());
}

bool operator==(const segment& lhs, const segment& rhs)
{
    segment::iterator li = lhs.begin();
    segment::iterator ri = rhs.begin();
    for (; li != lhs.end() && ri != rhs.end(); ++li, ++ri)
        if (*li != *ri)
            return false;
    return li == lhs.end() && ri == rhs.end();
}

}} // namespace yplatform::zerocopy

namespace ymod_cache {
namespace local_cache {

void impl::init(const yplatform::ptree& xml)
{
    impl_base::init(xml);

    // Configure local cache
    const yplatform::ptree& local_cache_xml = xml.get_child("local_cache");
    max_size_ = local_cache_xml.get("max_size", 1000*1000);
    lazy_clean_ = local_cache_xml.find("lazy_clean") != local_cache_xml.not_found();
}

void impl::fini()
{
    impl_base::fini();
}

future_result impl::set(
    yplatform::task_context_ptr /*ctx*/,
    const segment& key,
    const segment& value)
{
    promise_result out;
    try
    {
        boost::unique_lock<mutex_t> lock(mux_);

        entry new_entry(key, value, boost::chrono::steady_clock::now() + boost::chrono::seconds(value_ttl_));

        typedef container_t::index<key_tag>::type key_view_t;
        key_view_t& key_view = cont_.get<key_tag>();
        key_view_t::iterator it = key_view.find(key);
        if (it != key_view.end())
        {
            cont_.relocate(cont_.begin(), cont_.iterator_to(*it));
            key_view.replace(it, new_entry);
        }
        else
        {
            if (cont_.size() == max_size_)
            {
                if (cont_.back().expiry < boost::chrono::steady_clock::now())
                    cont_.pop_back();
                else
                {
                    out.set_exception(backend_error() << BOOST_ERROR_INFO << yplatform::error_private_info("cache capacity exhausted"));
                    return out;
                }
            }
            cont_.push_front(new_entry);
        }
        out.set(void_result());
    }
    catch(const std::exception& e)
    {
        out.set_exception(internal_error() << BOOST_ERROR_INFO << yplatform::error_private_info(e.what()));
    }
    catch(...)
    {
        out.set_current_exception();
    }
    return out;
}

future_segment impl::get(
    yplatform::task_context_ptr ctx,
    const segment& key)
{
    promise_segment out;
    try
    {
        boost::shared_lock<mutex_t> lock(mux_);

        typedef container_t::index<key_tag>::type key_view_t;
        key_view_t& key_view = cont_.get<key_tag>();
        key_view_t::iterator it = key_view.find(key);
        if (it != key_view.end())
        {
            if (it->expiry >= boost::chrono::steady_clock::now())
                out.set(it->value);
            else
            {
                out.set(boost::none);
                if (!lazy_clean_)
                {
                    lock.unlock();
                    remove(ctx, key);
                }
            }
        }
        else
            out.set(boost::none);
    }
    catch(const std::exception& e)
    {
        out.set_exception(internal_error() << BOOST_ERROR_INFO << yplatform::error_private_info(e.what()));
    }
    catch(...)
    {
        out.set_current_exception();
    }
    return out;
}

future_bool impl::has(
    yplatform::task_context_ptr ctx,
    const segment& key)
{
    promise_bool out;
    try
    {
        boost::shared_lock<mutex_t> lock(mux_);

        typedef container_t::index<key_tag>::type key_view_t;
        key_view_t& key_view = cont_.get<key_tag>();
        key_view_t::iterator it = key_view.find(key);
        if (it != key_view.end())
        {
            if (it->expiry >= boost::chrono::steady_clock::now())
                out.set(true);
            else
            {
                out.set(false);
                if (!lazy_clean_)
                {
                    lock.unlock();
                    remove(ctx, key);
                }
            }
        }
        else
            out.set(false);
    }
    catch(const std::exception& e)
    {
        out.set_exception(internal_error() << BOOST_ERROR_INFO << yplatform::error_private_info(e.what()));
    }
    catch(...)
    {
        out.set_current_exception();
    }
    return out;
}

future_result impl::remove(
    yplatform::task_context_ptr /*ctx*/,
    const segment& key)
{
    promise_result out;
    try
    {
        {
            boost::unique_lock<mutex_t> lock(mux_);
            cont_.get<key_tag>().erase(key);
        }
        out.set(void_result());
    }
    catch(const std::exception& e)
    {
        out.set_exception(internal_error() << BOOST_ERROR_INFO << yplatform::error_private_info(e.what()));
    }
    catch(...)
    {
        out.set_current_exception();
    }
    return out;
}

const impl::module_stats_ptr impl::get_module_stats() const
{
    boost::unique_lock<mutex_t> lock(mux_);
    const size_t size = cont_.size();
    lock.unlock();

    return make_shared<impl_stats>(size);
}

impl_stats::ptree_ptr impl_stats::core_get_stat() const
{
    ptree_ptr stats = base::core_get_stat();

    stats->put("size", size_);

    return stats;
}

}}
