#ifndef _CONSISTENT_MAPPER_H_
#define _CONSISTENT_MAPPER_H_

#include "mapper.h"

#include <boost/random/mersenne_twister.hpp>
#include <boost/random/uniform_int.hpp>
#include <boost/multi_index_container.hpp>
#include <boost/multi_index/ordered_index.hpp>
#include <boost/multi_index/member.hpp>
#include <map>
#include <vector>
#include <limits>
#include <cassert>

template <class In, class Out>
class consistent_mapper : public mapper<In, Out>
{
    typedef mapper<In, Out> base_t;

    struct point
    {
        In in;
        Out out;
        point(In in_, Out out_)
            : in(in_)
            , out(out_)
        {}
    };
    struct in_tag {};
    struct out_tag {};
    typedef boost::multi_index_container<
        point,
        boost::multi_index::indexed_by<
            boost::multi_index::ordered_unique<
                boost::multi_index::tag<in_tag>,
                boost::multi_index::member<point, In, &point::in>
            >,
            boost::multi_index::ordered_non_unique<
                boost::multi_index::tag<out_tag>,
                boost::multi_index::member<point, Out, &point::out>
            >
        >
    > circle_t;
    typedef typename circle_t::template index<in_tag>::type in_view_t;
    typedef typename in_view_t::iterator in_view_iterator_t;
    typedef typename circle_t::template index<out_tag>::type out_view_t;
    typedef typename out_view_t::iterator out_view_iterator_t;

    typedef std::vector<In> points_t;
    typedef typename points_t::iterator points_iterator_t;
    typedef std::map<Out, points_t> masked_t;
    typedef typename masked_t::iterator masked_iterator_t;

public:
    using typename base_t::input_type;
    using typename base_t::output_type;

    consistent_mapper(
        Out min, Out max,
        uint32_t seed,
        size_t dots_per_output)
        : base_t(min, max)
        , c_dots_per_output(dots_per_output)
        , gen_(seed)
        , rand_(std::numeric_limits<In>::min(), std::numeric_limits<In>::max())
    {
        fill_range(min, max);
    }

    void set_output_range(Out min, Out max)
    {
        fill_range(min, min_-1);
        erase_range(min_, min-1);
        fill_range(max_+1, max);
        erase_range(max+1, max_);

        base_t::set_output_range(min, max);
    }


#ifdef DEBUG
    void debug_print(std::ostream& s) const
    {
        typedef typename in_view_t::const_iterator in_view_const_iterator_t;
        typedef typename points_t::const_iterator points_const_iterator_t;
        typedef typename masked_t::const_iterator masked_const_iterator_t;
        const in_view_t& in_view = circle_.template get<in_tag>();
        s << "Circle points:\n";
        for (in_view_const_iterator_t pit = in_view.begin(); pit != in_view.end(); ++pit)
            s << "\t(" << pit->in << ", " << pit->out << ")\n";
        s << "Masked outs:\n";
        for (masked_const_iterator_t mit = masked_.begin(); mit != masked_.end(); ++mit)
        {
            s << "\t" << mit->first << ": ";
            for (points_const_iterator_t pit = mit->second.begin(); pit != mit->second.end(); ++pit)
                s << *pit << ", ";
            s << "\n";
        }
    }
#endif

private:

    void core_mask(Out out)
    {
        points_t& points = masked_[out];
        points.reserve(c_dots_per_output);

        out_view_t& out_view = circle_.template get<out_tag>();
        const out_view_iterator_t points_begin = out_view.lower_bound(out);
        const out_view_iterator_t points_end = out_view.upper_bound(out);
        for (out_view_iterator_t it = points_begin; it != points_end; ++it)
            points.push_back(it->in);
        out_view.erase(points_begin, points_end);
    }

    void core_unmask(Out out)
    {
        masked_iterator_t out_it = masked_.find(out);
        if (out_it != masked_.end())
        {
            points_t& points = out_it->second;
            out_view_t& out_view = circle_.template get<out_tag>();
            out_view_iterator_t hint_it = out_view.find(out);
            for (points_iterator_t it = points.begin(); it != points.end(); ++it)
            {
                out_view_iterator_t new_point_it = out_view.insert(hint_it, point(*it, out));
                if (new_point_it->out != out) // if insertion failed
                    out_view.replace(new_point_it, point(*it, out));
            }
            masked_.erase(out_it);
        }
    }

    bool core_masked(Out out) const
    {
        return masked_.find(out) != masked_.end();
    }

    bool core_all_masked() const
    {
        return masked_.size() == max_-min_+1;
    }

    Out core_map(In in) const
    {
        const in_view_t& in_view = circle_.template get<in_tag>();
        in_view_iterator_t point_it = in_view.lower_bound(in);
        if (point_it != in_view.end())
            return point_it->out;
        else
            return in_view.lower_bound(min_)->out;
    }

    void fill_range(Out a, Out b)
    {
        for (Out i = a; i <= b; ++i)
        {
            for (size_t j = 0; j < c_dots_per_output; ++j)
                circle_.insert(point(rand_(gen_), i));
        }
    }

    void erase_range(Out a, Out b)
    {
        for (Out i = a; i <= b; ++i)
        {
            circle_.template get<out_tag>().erase(i);
            masked_.erase(i);
        }
    }

    using base_t::min_;
    using base_t::max_;
    const size_t c_dots_per_output;
    boost::mt19937 gen_;
    boost::uniform_int<In> rand_;
    circle_t circle_;
    masked_t masked_;
};

#endif // _CONSISTENT_MAPPER_H_
