#ifndef _YMOD_CACHE_MEMCACHED_IMPL_H_
#define _YMOD_CACHE_MEMCACHED_IMPL_H_

#include "common.h"
#include "session.h"
#include "mapper.h"
#include "session_pool.h"

#include <yplatform/net/module.h>

#include <boost/shared_ptr.hpp>
#include <boost/thread/mutex.hpp>
#include <vector>

namespace ymod_cache {
namespace memcached {

typedef yamail::resource_pool::time_traits::duration time_duration;

struct server_info
{
    using session_pools_t = std::unordered_map<boost::asio::io_service*, std::unique_ptr<session_pool>>;

    server_info(
        const string& address,
        unsigned short port,
        session_pools_t pools)
        : address(address)
        , port(port)
        , pools(std::move(pools))
    {}

    string address;
    unsigned short port;
    boost::shared_ptr<yplatform::net::timer_t> recovery_timer;
    session_pools_t pools;

    template <class Callback>
    void get_session(
            boost::asio::io_service* io,
            const Callback& callback,
            const time_duration &wait_timeout) const {
        pools.at(io)->get_auto_waste(*io, callback, wait_timeout);
    }
};

typedef std::vector<server_info> server_vector;

class impl
    : public impl_base
    , public yplatform::net::client_module<session>
{
public:
    typedef yplatform::task_context_ptr task_context_ptr;
    typedef yplatform::module_stats_ptr module_stats_ptr;

    impl(yplatform::reactor& reactor);

    void init(const yplatform::ptree& xml);
    void start();
    void stop();
    void fini();

    future_result set(
        task_context_ptr ctx,
        const segment& key,
        const segment& value);

    future_segment get(
        task_context_ptr ctx,
        const segment& key);

    future_bool has(
        task_context_ptr ctx,
        const segment& key);

    future_result remove(
        task_context_ptr ctx,
        const segment& key);

    virtual const module_stats_ptr get_module_stats() const;

private:
    void server_recovery_hook(
        const boost::system::error_code& err,
        std::size_t server_id);

    future_response execute(
        task_context_ptr ctx,
        const request_ptr& req,
        const time_duration& queue_timeout,
        unsigned hops_count = 0,
        promise_response resp_p = promise_response());

    void get_session_cb(task_context_ptr ctx,
        const request_ptr& req,
        const time_duration& queue_timeout,
        std::size_t server_id,
        promise_response resp_p,
        unsigned hops_count,
        const boost::system::error_code& error,
        session_handle sess_handle);

    void execute_cb(task_context_ptr ctx,
        const request_ptr& req,
        const time_duration& queue_timeout,
        std::size_t server_id,
        session_handle sess_handle,
        future_response sess_resp_f,
        promise_response resp_p,
        unsigned hops_count);

    void set_cb(
        future_response resp_f,
        promise_result out);

    void get_cb(
        future_response resp_f,
        promise_segment out);

    void has_cb(
        future_response resp_f,
        promise_bool out);

    void remove_cb(
        future_response resp_f,
        promise_result out);

    void mask_server(task_context_ptr ctx, std::size_t server_id);

    yplatform::reactor_ptr reactor_;

    server_vector servers_;
#define MAPPER_TEMPLATE_ARGS std::size_t, std::size_t
    boost::shared_ptr<mapper<MAPPER_TEMPLATE_ARGS> > hash_mapper_;
    unsigned int recovery_timeout_;
    unsigned max_hops_count_;
    time_duration get_queue_timeout_;
    time_duration has_queue_timeout_;
    time_duration remove_queue_timeout_;
    time_duration set_queue_timeout_;
};

using boost::property_tree::ptree;

ptree make_session_pool_stats(const session_pool& pool);
ptree make_server_stats(const server_info& server);
string make_server_key(const server_info& server);
std::pair<string, ptree> make_server_pair(const server_info& server);

class impl_stats : public yplatform::module_stats
{
public:
    typedef yplatform::module_stats base;

    impl_stats(const server_vector& servers) : servers_(std::move(servers)) {}

private:
    const server_vector& servers_;

    virtual ptree_ptr core_get_stat() const;
};

}}

#endif // _YMOD_CACHE_MEMCACHED_IMPL_H_
