#ifndef _MAPPER_H_
#define _MAPPER_H_

#include <ymod_cache/error.h>

template <class In, class Out>
class mapper
{
public:
    typedef In input_type;
    typedef Out output_type;

    mapper(Out min, Out max)
        : min_(min)
        , max_(max)
    {}
    virtual ~mapper()
    {}

    virtual void set_output_range(Out min, Out max)
    {
        min_ = min;
        max_ = max;
    }

    Out min_output() const
    {
        return min_;
    }

    Out max_output() const
    {
        return max_;
    }

    void mask(Out out)
    {
        unique_lock_t lock(mux_);
        if( !core_masked(out) ) {
            core_mask(out);
        }
    }

    void unmask(Out out)
    {
        unique_lock_t lock(mux_);
        if( core_masked(out) ) {
            core_unmask(out);
        }
    }

    bool masked(Out out) const
    {
        shared_lock_t lock(mux_);
        return core_masked(out);
    }

    bool all_masked() const
    {
        shared_lock_t lock(mux_);
        return core_all_masked();
    }

    Out map(In in) const
    {
        shared_lock_t lock(mux_);
        if( core_all_masked() ) {
            throw ymod_cache::backend_error() << BOOST_ERROR_INFO << yplatform::error_private_info("all servers masked");
        }
        return core_map(in);
    }

private:
    virtual void core_mask(Out out) = 0;
    virtual void core_unmask(Out out) = 0;
    virtual bool core_masked(Out out) const = 0;
    virtual bool core_all_masked() const = 0;
    virtual Out core_map(In in) const = 0;

protected:
    Out min_;
    Out max_;
private:
    typedef boost::shared_mutex shared_mutex_t;
    typedef boost::unique_lock<shared_mutex_t> unique_lock_t;
    typedef boost::shared_lock<shared_mutex_t> shared_lock_t;

    mutable shared_mutex_t mux_;
};

#endif // _MAPPER_H_
