#include "protocol.h"

#include <yplatform/net/byte_order.h>

#include <boost/cast.hpp>

#define HTON(x) \
    x = yplatform::net::host_2_network(x);
#define NTOH(x) \
    x = yplatform::net::network_2_host(x);

namespace ymod_cache {
namespace memcached {

size_t request::header_length()
{
    return sizeof(protocol_binary_request_header);
}

uint8_t* request::header_raw()
{
    return reinterpret_cast<uint8_t*>(&variant.generic.message.header.request);
}

size_t request::extras_length() const
{
    return variant.generic.message.header.request.extlen;
}

uint8_t* request::extras_raw()
{
    switch (variant.generic.message.header.request.opcode)
    {
    case PROTOCOL_BINARY_CMD_SET:
        return reinterpret_cast<uint8_t*>(&variant.set.message.body);
    case PROTOCOL_BINARY_CMD_GET:
        return 0; //&variant.get.message.body;
    case PROTOCOL_BINARY_CMD_DELETE:
        return 0; //&variant.delete_.message.body;
    default:
        return 0;
    }
}

void request::host_2_network_header()
{
    HTON(variant.generic.message.header.request.magic);
    HTON(variant.generic.message.header.request.opcode);
    HTON(variant.generic.message.header.request.keylen);
    HTON(variant.generic.message.header.request.extlen);
    HTON(variant.generic.message.header.request.datatype);
    HTON(variant.generic.message.header.request.vbucket);
    HTON(variant.generic.message.header.request.bodylen);
    HTON(variant.generic.message.header.request.opaque);
    HTON(variant.generic.message.header.request.cas);
}

void request::host_2_network_extras()
{
    switch (variant.generic.message.header.request.opcode)
    {
    case PROTOCOL_BINARY_CMD_SET:
        HTON(variant.set.message.body.flags);
        HTON(variant.set.message.body.expiration);
        break;
    case PROTOCOL_BINARY_CMD_GET:
    case PROTOCOL_BINARY_CMD_DELETE:
    default:
        break;
    }
}

size_t response::header_length()
{
    return sizeof(protocol_binary_response_header);
}

uint8_t* response::header_raw()
{
    return reinterpret_cast<uint8_t*>(&variant.generic.message.header.response);
}

size_t response::extras_length() const
{
    return variant.generic.message.header.response.extlen;
}

uint8_t* response::extras_raw()
{
    switch (variant.generic.message.header.response.opcode)
    {
    case PROTOCOL_BINARY_CMD_SET:
        return 0; //&variant.set.message.body;
    case PROTOCOL_BINARY_CMD_GET:
        return reinterpret_cast<uint8_t*>(&variant.get.message.body);
    case PROTOCOL_BINARY_CMD_DELETE:
        return 0; //&variant.delete_.message.body;
    default:
        return 0;
    }
}

void response::network_2_host_header()
{
    NTOH(variant.generic.message.header.response.magic);
    NTOH(variant.generic.message.header.response.opcode);
    NTOH(variant.generic.message.header.response.keylen);
    NTOH(variant.generic.message.header.response.extlen);
    NTOH(variant.generic.message.header.response.datatype);
    NTOH(variant.generic.message.header.response.status);
    NTOH(variant.generic.message.header.response.bodylen);
    NTOH(variant.generic.message.header.response.opaque);
    NTOH(variant.generic.message.header.response.cas);
}

void response::network_2_host_extras()
{
    switch (variant.generic.message.header.response.opcode)
    {
    case PROTOCOL_BINARY_CMD_GET:
        NTOH(variant.get.message.body.flags);
        break;
    case PROTOCOL_BINARY_CMD_SET:
    case PROTOCOL_BINARY_CMD_DELETE:
    default:
        break;
    }
}

u_int16_t make_keylen(const boost::optional<const segment&>& key) {
    try {
        return key ? boost::numeric_cast<u_int16_t>(key.get().size()) : u_int16_t(0);
    } catch (const boost::bad_numeric_cast&) {
        throw key_overflow();
    }
}

u_int32_t make_value_len(const boost::optional<const segment&>& value) {
    try {
        return value ? boost::numeric_cast<u_int32_t>(value.get().size()) : 0u;
    } catch (const boost::bad_numeric_cast&) {
        throw value_overflow();
    }
}

request_ptr new_request(
    protocol_binary_command cmd,
    const boost::optional<const segment&> key,
    const boost::optional<const segment&> value)
{
    request_ptr req(new request);
    protocol_binary_request_header& header = req->variant.generic.message.header;

    header.request.magic = PROTOCOL_BINARY_REQ;
    header.request.opcode = cmd;
    header.request.keylen = make_keylen(key);
    switch (cmd)
    {
    case PROTOCOL_BINARY_CMD_SET:
        header.request.extlen = sizeof(req->variant.set.message.body);
        break;
    case PROTOCOL_BINARY_CMD_GET:
        header.request.extlen = 0; //sizeof(req->variant.get.message.body);
        break;
    case PROTOCOL_BINARY_CMD_DELETE:
        header.request.extlen = 0; //sizeof(req->variant.delete_.message.body);
        break;
    default:
        header.request.extlen = 0;
    }
    header.request.datatype = PROTOCOL_BINARY_RAW_BYTES;
    header.request.vbucket = 0;
    header.request.bodylen =
        header.request.extlen +
        header.request.keylen +
        make_value_len(value);
    header.request.opaque = 0;
    header.request.cas = 0;

    if (key)
        req->key = key.get();
    if (value)
        req->value = value.get();
    return req;
}

}}
