#include "session.h"

#include <yplatform/net/byte_order.h>

#include <boost/bind.hpp>
#include <algorithm>
#include <iterator>

namespace ymod_cache {
namespace memcached {

session::session(
    yplatform::net::base_service* service,
    const yplatform::net::client_settings& settings)
    : base_t (service, settings), is_new_(true)
{
    std::stringstream stream;
    stream << id();
    set_log_prefix(stream.str());
}

session::~session()
{
}

void session::set_server(const string& address, unsigned short port)
{
    async_close([self=shared_from_this()](boost::system::error_code){}); // force reconnect
    address_ = address;
    port_ = port;
}

yplatform::net::zcopy_list<yplatform::net::buffers::const_chunk_buffer>
prepare_out_sequence(const request_ptr& req)
{
    size_t extras_length = req->extras_length();
    req->host_2_network_extras();
    req->host_2_network_header();

    yplatform::net::zcopy_list<yplatform::net::buffers::const_chunk_buffer> seq;
    seq.push_back(yplatform::net::buffers::make_chunk_buffer(new yplatform::net::buffers::const_raw_ptr_chunk(
        req->header_raw(),
        response::header_length()
    )));
    seq.push_back(yplatform::net::buffers::make_chunk_buffer(new yplatform::net::buffers::const_raw_ptr_chunk(
        req->extras_raw(),
        extras_length
    )));
    yplatform::net::buffers::insert_chunks_2_queue(req->key.begin_fragment(), req->key.end_fragment(), req->key.head(), req->key.tail(), seq);
    yplatform::net::buffers::insert_chunks_2_queue(req->value.begin_fragment(), req->value.end_fragment(), req->value.head(), req->value.tail(), seq);
    return seq;
}

future_response session::run(const request_ptr& req)
{
    promise_response out;
    if (stream().is_open())
        handle_connect(boost::system::error_code(), req, out);
    else
        connect(
            address_, port_,
            boost::bind(&session::handle_connect, shared_from_this(), _1, req, out),
            boost::bind(&session::handle_timeout, shared_from_this(), "connect")
        );
    return out;
}

void session::handle_timeout(const char* operation)
{
    NET_SESSION_LOG(error) << "request cancelled by timer (in " << operation << ")";
    try
    {
        cancel_operations();
    }
    catch (::boost::system::system_error const& err)
    {
        NET_SESSION_LOG(error) << "handle_timeout: " << err.what();
    }
    catch (...)
    {
        NET_SESSION_LOG(error) << "handle_timeout: unknown exception";
    }
}

void session::handle_connect(const boost::system::error_code& err, const request_ptr& req, promise_response out)
{
    if (!err)
        async_write(
            prepare_out_sequence(req),
            boost::bind(&session::handle_write_request, shared_from_this(), _1, req, out),
            boost::bind(&session::handle_timeout, shared_from_this(), "send")
        );
    else
    {
        out.set(error_code::connect_error);
    }
}

void session::handle_write_request(const boost::system::error_code& err, const request_ptr&, promise_response out)
{
    // Passing request to this function to ensure that all write buffers remain valid until they writen
    if (!err)
    {
        read_buff_ = create_buffer();
        size_t to_read = response::header_length();
        async_read(
            boost::bind(&session::handle_read_header, shared_from_this(), _1, _2, out),
            boost::bind(&session::handle_timeout, shared_from_this(), "receive"),
            read_buff_->prepare(to_read),
            to_read
        );
    }
    else
    {
        NET_SESSION_LOG(error) << "handle_write_request: error: " << err.message();
        out.set(error_code::write_request_error);
    }
}

void session::handle_read_header(const boost::system::error_code& err, std::size_t bytes, promise_response out)
{
    if (!err)
    {
        read_buff_->commit(bytes);
        response_ptr resp(new response);

        yplatform::zerocopy::streambuf::iterator part_end = read_buff_->begin();
        std::advance(part_end, response::header_length());
        std::copy(read_buff_->begin(), part_end, resp->header_raw());
        resp->network_2_host_header();
        read_buff_->detach(part_end);

        if (read_buff_->size() < resp->extras_length())
        {
            size_t to_read = resp->extras_length() - read_buff_->size();
            async_read(
                boost::bind(&session::handle_read_extras, shared_from_this(), _1, _2, resp, out),
                boost::bind(&session::handle_timeout, shared_from_this(), "receive"),
                read_buff_->prepare(to_read),
                to_read
            );
        }
        else
            handle_read_extras(err, 0, resp, out);
    }
    else
    {
        NET_SESSION_LOG(error) << "handle_read_header: error: " << err.message();
        out.set(error_code::read_header_error);
    }
}

void session::handle_read_extras(const boost::system::error_code& err, std::size_t bytes, response_ptr resp, promise_response out)
{
    if (!err)
    {
        read_buff_->commit(bytes);
        yplatform::zerocopy::streambuf::iterator part_end = read_buff_->begin();
        std::advance(part_end, resp->extras_length());
        std::copy(read_buff_->begin(), part_end, resp->extras_raw());
        resp->network_2_host_extras();
        read_buff_->detach(part_end);

        const size_t key_length = resp->variant.generic.message.header.response.keylen;
        if (read_buff_->size() < key_length)
        {
            size_t to_read = key_length - read_buff_->size();
            async_read(
                boost::bind(&session::handle_read_key, shared_from_this(), _1, _2, resp, out),
                boost::bind(&session::handle_timeout, shared_from_this(), "receive"),
                read_buff_->prepare(to_read),
                to_read
            );
        }
        else
            handle_read_key(err, 0, resp, out);
    }
    else
    {
        NET_SESSION_LOG(error) << "handle_read_extras: error: " << err.message();
        out.set(error_code::read_extras_error);
    }
}

void session::handle_read_key(const boost::system::error_code& err, std::size_t bytes, response_ptr resp, promise_response out)
{
    if (!err)
    {
        read_buff_->commit(bytes);
        const size_t key_length = resp->variant.generic.message.header.response.keylen;
        yplatform::zerocopy::streambuf::iterator part_end = read_buff_->begin();
        std::advance(part_end, key_length);
        resp->key = read_buff_->detach(part_end);

        const size_t value_length =
            resp->variant.generic.message.header.response.bodylen -
            resp->extras_length() -
            key_length;
        if (read_buff_->size() < value_length)
        {
            size_t to_read = value_length - read_buff_->size();
            async_read(
                boost::bind(&session::handle_read_value, shared_from_this(), _1, _2, resp, out),
                boost::bind(&session::handle_timeout, shared_from_this(), "receive"),
                read_buff_->prepare(to_read),
                to_read
            );
        }
        else
            handle_read_value(err, 0, resp, out);
    }
    else
    {
        NET_SESSION_LOG(error) << "handle_read_key: error: " << err.message();
        out.set(error_code::read_key_error);
    }
}

void session::handle_read_value(const boost::system::error_code& err, std::size_t bytes, response_ptr resp, promise_response out)
{
    if (!err)
    {
        read_buff_->commit(bytes);
        const size_t key_length = resp->variant.generic.message.header.response.keylen;
        const size_t value_length =
            resp->variant.generic.message.header.response.bodylen -
            resp->extras_length() -
            key_length;
        yplatform::zerocopy::streambuf::iterator part_end = read_buff_->begin();
        std::advance(part_end, value_length);
        resp->value = read_buff_->detach(part_end);

        out.set(resp);
    }
    else
    {
        NET_SESSION_LOG(error) << "handle_read_value: error: " << err.message();
        out.set(error_code::read_value_error);
    }
}

}}
