#include <mail/ymod_cachedb/include/cache.h>
#include <mail/ymod_cachedb/include/error.h>
#include <mail/ymod_cachedb/include/internal/query.h>
#include <mail/ymod_cachedb/include/internal/logger.h>
#include <mail/ymod_cachedb/include/internal/callbacks.h>
#include <mail/webmail/commondb/include/with_executers.h>
#include <mail/webmail/commondb/include/settings_reflection.h>
#include <yamail/data/deserialization/ptree.h>
#include <yamail/data/serialization/json_writer.h>
#include <yplatform/module_registration.h>
#include <pgg/database/pool_factory.h>
#include <pgg/factory.h>
#include <fstream>


BOOST_FUSION_DEFINE_STRUCT((ymod_cachedb), Settings,
    (std::string, reactor)
    (std::string, default_datatype)
    (std::string, logger)
)

BOOST_FUSION_DEFINE_STRUCT((ymod_cachedb), DummySettings,
    (bool, put_status)
)

namespace ymod_cachedb {

using namespace commondb;
using namespace std::string_literals;

void onGet(pgg::error_code ec, std::vector<result::Get> r, CallbackGet cb) {
    if (ec) {
        cb(std::move(ec), std::nullopt);
    } else {
        if (r.empty()) {
            cb(mail_errors::error_code(), std::nullopt);
        } else if (r.size() == 1) {
            cb(mail_errors::error_code(), Value(std::move(r.front().value)));
        } else {
            cb(make_error(CacheResult::internalError,
                "strange number of results: "s + std::to_string(r.size())
            ), std::nullopt);
        }
    }
}

void onPut(pgg::error_code ec, result::Put r, CallbackPut cb) {
    if (ec) {
        cb(std::move(ec), false);
    } else {
        cb(mail_errors::error_code(), r.value);
    }
}

auto wrap(CallbackGet cb) {
    return [cb](pgg::error_code ec, auto result) {
        onGet(ec, result, cb);
    };
}

auto wrap(CallbackPut cb) {
    return [cb](pgg::error_code ec, auto result) {
        onPut(ec, result, cb);
    };
}

struct Dummy: public Cache, public yplatform::module {
    virtual ~Dummy() { }

    void init(const yplatform::ptree& cfg) {
        common = yamail::data::deserialization::fromPtree<Settings>(cfg);
        dummy = yamail::data::deserialization::fromPtree<DummySettings>(cfg.get_child("dummy"));

        LOGDOG_(getModuleLogger(common.logger), notice,
            log::message="ymod_cachedb::Dummy module initialized",
            log::settings=yamail::data::serialization::JsonWriter(common).result(),
            log::dummy_settings=yamail::data::serialization::JsonWriter(dummy).result(),
            log::password_length=0
        );
    }

protected:
    void getAsync(Uid, Key, DataType, RequestId, CallbackGet cb) const override {
        cb(mail_errors::error_code(), std::nullopt);
    }

    void putAsync(Uid, Key, DataType, Value, RequestId, CallbackPut cb) const override {
        cb(mail_errors::error_code(), dummy.put_status);
    }

    DataType defaultDataType() const override {
        static DataType dataType;
        return dataType;
    }

    Settings common;
    DummySettings dummy;
};

struct Postgres: public WithExecuters, public Cache, public yplatform::module {

    virtual ~Postgres() { }

    using WithExecuters::anyExecutor;
    using WithExecuters::readWriteExecutor;

    auto anyExecutor(Uid uid, RequestId reqId) const {
        const std::string requestId = static_cast<std::string>(reqId);

        auto ctxLogger = getContextLogger(common.logger, uid, reqId);
        auto logger = makePggLogger(ctxLogger);
        auto profiler = commondb::pggProfiler(std::to_string(uid.t), requestId, "cachedb");

        return anyExecutor(requestId, logger, profiler);
    }

    auto readWriteExecutor(Uid uid, RequestId reqId) const {
        const std::string requestId = static_cast<std::string>(reqId);

        auto ctxLogger = getContextLogger(common.logger, uid, reqId);
        auto logger = makePggLogger(ctxLogger);
        auto profiler = commondb::pggProfiler(std::to_string(uid.t), requestId, "cachedb");

        return readWriteExecutor(requestId, logger, profiler);
    }

    void init(const yplatform::ptree& cfg) {
        common = yamail::data::deserialization::fromPtree<Settings>(cfg);
        pg = readPgSettings(cfg.get_child("pg"));

        initExecuters(common.reactor, pg, query::QueriesRegister(), query::ParametersRegister());

        LOGDOG_(getModuleLogger(common.logger), notice,
            log::message="ymod_cachedb::Postgres module initialized",
            log::settings=yamail::data::serialization::JsonWriter(common).result(),
            log::pg_settings=yamail::data::serialization::JsonWriter(pg).result(),
            log::password_length=passwordSize()
        );
    }

    void getAsync(Uid uid, Key key, DataType dataType, RequestId reqId, CallbackGet cb) const override {
        anyExecutor(uid, reqId)->request<std::vector<result::Get>>(
            pgg::withQuery<query::Get>(
                query::Uid(uid),
                query::Key(std::string(key.t.data())),
                query::DataType(std::string(dataType.t.data()))
            ),
            wrap(std::move(cb))
        );
    }

    void putAsync(Uid uid, Key key, DataType dataType, Value value, RequestId reqId, CallbackPut cb) const override {
        readWriteExecutor(uid, reqId)->request<result::Put>(
            pgg::withQuery<query::Put>(
                query::Uid(uid),
                query::Key(std::string(key.t.data())),
                query::Value(std::move(value)),
                query::DataType(std::string(dataType.t.data()))
            ),
            wrap(std::move(cb))
        );
    }

    DataType defaultDataType() const override {
        return DataType(common.default_datatype);
    }

    Settings common;
    PgSettings pg;
};

}

DEFINE_SERVICE_OBJECT(ymod_cachedb::Postgres)
DEFINE_SERVICE_OBJECT(ymod_cachedb::Dummy)
