#include "hostlist_builder.h"
#include "errors.h"

#include <yplatform/yield.h>

#include <algorithm>
#include <cstdlib>

namespace NYmodHttpWatcher::NQloudInstances {

THostListBuilder::THostListBuilder(
    const yhttp::response& response,
    const THosts& prevHosts,
    THostCheckerPtr hostChecker,
    TCallback callback
)
    : HttpResponse(response)
    , PrevHosts(prevHosts)
    , HostChecker(std::move(hostChecker))
    , Callback(std::move(callback))
{}

void THostListBuilder::operator()(TYieldCtx yieldCtx, bool alive) {
    reenter (yieldCtx) {
        HandleResponse();
        if (ErrorCode) {
            yield break;
        }

        CheckResidingInstances();

        while (!ResidingInstances.empty()) {
            yield {
                const auto& [state, nowInstanceToPing] = *ResidingInstances.back();
                HostChecker->AsyncCheck(nowInstanceToPing, yieldCtx);
            }
            if (!alive) {
                Instances.erase(ResidingInstances.back());
            }
            ResidingInstances.pop_back();
        }
    }

    if (yieldCtx.is_complete()) {
        if (ErrorCode) {
            YLOG_L(error) << "Fail to get Qloud instances: " << ErrorCode.message();
        } else {
            CallbackIfNeed();
        }
    }
}

void THostListBuilder::HandleResponse() {
    auto [ec, rawInstances] = ParseResponse(HttpResponse);
    ErrorCode = ec;
    if (!ErrorCode) {
        Instances = ReduceInstances(std::move(rawInstances));
    }
}

void THostListBuilder::CheckResidingInstances() {
    for (auto it = Instances.begin(); it != Instances.end();) {
        const auto& [state, fqdn] = *it;

        if (state == EState::Residing) {
            if (!HostChecker) {
                it = Instances.erase(it);
                continue;
            }
            ResidingInstances.push_back(it);
        }

        ++it;
    }
}

void THostListBuilder::CallbackIfNeed() {
    if (PrevHosts.size() == Instances.size()) {
        auto comparator = [](const auto& lhs, const auto& rhs) {
            return lhs == rhs.second;
        };
        if (std::equal(PrevHosts.begin(), PrevHosts.end(), Instances.begin(), std::move(comparator))) {
            return;
        }
    }

    const auto rawLocalHost = std::getenv("QLOUD_HOSTNAME");
    std::string localHost(rawLocalHost ? rawLocalHost : "");

    THosts hosts;
    for (auto& [state, fqdn] : Instances) {
        hosts.push_back(std::move(fqdn));
    }

    Callback(std::move(localHost), std::move(hosts));
}

} // namespace NYmodHttpWatcher::NQloudInstances
