#include "parser.h"

#include <library/cpp/json/json_reader.h>

namespace NYmodHttpWatcher::NQloudInstances {

namespace {

EErrorCode ParseInstance(const NJson::TJsonValue& jsonValue, uint64_t& id, TInstance& instance) {
    const auto& data = jsonValue.GetMap();
    const auto& stateStr = data.at("state").GetString();
    auto& [state, fqdn] = instance;

    if (stateStr == "NEW") {
        state = EState::New;
    } else if (stateStr == "EXISTING") {
        state = EState::Existing;
    } else if (stateStr == "RESIDING") {
        state = EState::Residing;
    } else {
        return EC_UNKNOWN_STATE;
    }

    fqdn = static_cast<std::string>(data.at("fqdn").GetString());
    if (fqdn.empty()) {
        return EC_EMPTY_FQDN;
    }

    id = data.at("id").GetUInteger();
    if (id == 0) {
        return EC_INVALID_ID;
    }

    return EC_OK;
}

} // namespace

TParseRes ParseResponse(const yhttp::response& response) {
    if (response.status == 200) {
        return ParseSuccessResponse(response.body);
    }
    return {FromHttpStatus(response.status), {}};
}

TParseRes ParseSuccessResponse(const std::string& body) {
    NJson::TJsonValue value;
    if (!NJson::ReadJsonTree(body, &value, false)) {
        return {EC_PARSE_ERROR, {}};
    }

    TInstances instances;
    for (const auto& instValue : value["instances"].GetArray()) {
        uint64_t id;
        TInstance instance;
        auto ec = ParseInstance(instValue, id, instance);
        if (ec) {
            return {ec, {}};
        }
        instances.emplace(id, std::move(instance));
    }

    return {EC_OK, std::move(instances)};
}

TReducedInstances ReduceInstances(TInstances instances) {
    TReducedInstances reducedInstances;

    for (auto it = instances.begin(); it != instances.end();) {
        auto nextIt = instances.upper_bound(it->first);
        TInstance instance;
        bool found = false;

        for (auto cit = it; cit != nextIt && !found; ++cit) {
            auto& currInstance = cit->second;
            switch (currInstance.first) {
                case EState::Existing:
                    instance = std::move(currInstance);
                    found = true;
                    break;
                case EState::New:
                    instance = std::move(currInstance);
                    break;
                case EState::Residing:
                    // Prefer new
                    if (instance.second.empty()) {
                        instance = std::move(currInstance);
                    }
                    break;
            }
        }

        reducedInstances.push_back(std::move(instance));
        it = nextIt;
    }

    return reducedInstances;
}

} // namespace NYmodHttpWatcher::NQloudInstances
