#include <mail/ymod_http_watcher/handlers/qloud_instances/hostlist_builder.h>
#include <mail/ymod_http_watcher/handlers/qloud_instances/host_checker.h>

#include <library/cpp/testing/unittest/gtest.h>

using namespace NYmodHttpWatcher::NQloudInstances;

using TCheckCallback = std::function<void(std::string host, IHostChecker::TCallback callback)>;

class TTestHostChecker : public IHostChecker {
public:
    explicit TTestHostChecker(TCheckCallback callback)
        : Callback(std::move(callback))
    {}

    void AsyncCheck(std::string host, TCallback callback) override {
        Callback(std::move(host), std::move(callback));
    }

private:
    TCheckCallback Callback;
};

namespace {

std::pair<std::string, THostListBuilder::THosts> Run(
    const yhttp::response& response,
    const THostListBuilder::THosts& prevHosts,
    TCheckCallback checkCallback)
{
    std::shared_ptr<TTestHostChecker> checker;
    std::string localHostRes;
    THostListBuilder::THosts hostsRes;
    int CallCount = 0;

    if (checkCallback) {
        checker = std::make_shared<TTestHostChecker>(std::move(checkCallback));
    }

    auto callback = [&](std::string localHost, THostListBuilder::THosts hosts) {
        ASSERT_EQ(CallCount++, 0);
        localHostRes = std::move(localHost);
        hostsRes = std::move(hosts);
    };

    auto builder = std::make_shared<THostListBuilder>(response, prevHosts, std::move(checker), std::move(callback));
    yplatform::spawn(std::move(builder));

    return {std::move(localHostRes), std::move(hostsRes)};
}

yhttp::response MakeResponse(int code, std::string body) {
    return yhttp::response{code, {}, std::move(body), "Reason"};
}

} // namespace

TEST(THostListBuilder, FirstRun) {
    const auto jsonResponse = R"({
    "instances": [
    {
      "id": 1,
      "name": "name-1",
      "state": "NEW",
      "fqdn": "fqdn-1",
      "ip": "ip-1"
    }
    ]})";

    THostListBuilder::THosts prevHosts;
    auto [localHost, hosts] = Run(MakeResponse(200, jsonResponse), prevHosts, [](auto, auto) {
        ASSERT_TRUE(false);
    });

    EXPECT_TRUE(hosts == (THostListBuilder::THosts{"fqdn-1"}));
}

TEST(THostListBuilder, SecondRun) {
    const auto jsonResponse = R"({
    "instances": [
    {
      "id": 1,
      "name": "name-1",
      "state": "NEW",
      "fqdn": "fqdn-1",
      "ip": "ip-1"
    }
    ]})";

    THostListBuilder::THosts prevHosts{"fqdn-1"};
    auto [localHost, hosts] = Run(MakeResponse(200, jsonResponse), prevHosts, [](auto, auto) {
        ASSERT_TRUE(false);
    });

    EXPECT_EQ(hosts.size(), 0);
}

TEST(THostListBuilder, Expansion) {
    const auto jsonResponse = R"({
    "instances": [
    {
      "id": 1,
      "name": "name-1",
      "state": "EXISTING",
      "fqdn": "fqdn-1",
      "ip": "ip-1"
    },
    {
      "id": 2,
      "name": "name-2",
      "state": "NEW",
      "fqdn": "fqdn-2",
      "ip": "ip-2"
    }
    ]})";

    THostListBuilder::THosts prevHosts{"fqdn-1"};
    auto [localHost, hosts] = Run(MakeResponse(200, jsonResponse), prevHosts, [](auto, auto) {
        ASSERT_TRUE(false);
    });

    EXPECT_TRUE(hosts == (THostListBuilder::THosts{"fqdn-1", "fqdn-2"}));
}

TEST(THostListBuilder, Reduction) {
    const auto jsonResponse = R"({
    "instances": [
    {
      "id": 1,
      "name": "name-1",
      "state": "EXISTING",
      "fqdn": "fqdn-1",
      "ip": "ip-1"
    },
    {
      "id": 2,
      "name": "name-2",
      "state": "NEW",
      "fqdn": "fqdn-2",
      "ip": "ip-2"
    }
    ]})";

    THostListBuilder::THosts prevHosts{"fqdn-1", "fqdn-2", "fqdn-3"};
    auto [localHost, hosts] = Run(MakeResponse(200, jsonResponse), prevHosts, [](auto, auto) {
        ASSERT_TRUE(false);
    });

    EXPECT_TRUE(hosts == (THostListBuilder::THosts{"fqdn-1", "fqdn-2"}));
}

TEST(THostListBuilder, ResidingWithoutChecker) {
    const auto jsonResponse = R"({
    "instances": [
    {
      "id": 1,
      "name": "name-1",
      "state": "EXISTING",
      "fqdn": "fqdn-1",
      "ip": "ip-1"
    },
    {
      "id": 2,
      "name": "name-2",
      "state": "NEW",
      "fqdn": "fqdn-2",
      "ip": "ip-2"
    },
    {
      "id": 3,
      "name": "name-3",
      "state": "RESIDING",
      "fqdn": "fqdn-3",
      "ip": "ip-3"
    }
    ]})";

    THostListBuilder::THosts prevHosts{"fqdn-1", "fqdn-2", "fqdn-3"};
    auto [localHost, hosts] = Run(MakeResponse(200, jsonResponse), prevHosts, {});

    EXPECT_TRUE(hosts == (THostListBuilder::THosts{"fqdn-1", "fqdn-2"}));
}

TEST(THostListBuilder, ResidingWithCheckerAndDead) {
    const auto jsonResponse = R"({
    "instances": [
    {
      "id": 1,
      "name": "name-1",
      "state": "EXISTING",
      "fqdn": "fqdn-1",
      "ip": "ip-1"
    },
    {
      "id": 2,
      "name": "name-2",
      "state": "NEW",
      "fqdn": "fqdn-2",
      "ip": "ip-2"
    },
    {
      "id": 3,
      "name": "name-3",
      "state": "RESIDING",
      "fqdn": "fqdn-3",
      "ip": "ip-3"
    }
    ]})";

    THostListBuilder::THosts prevHosts{"fqdn-1", "fqdn-2", "fqdn-3"};
    int callCount = 0;
    auto [localHost, hosts] = Run(MakeResponse(200, jsonResponse), prevHosts, [&callCount](auto host, auto callback) {
        ++callCount;
        EXPECT_STREQ(host, "fqdn-3");
        callback(false);
    });

    EXPECT_EQ(callCount, 1);
    EXPECT_TRUE(hosts == (THostListBuilder::THosts{"fqdn-1", "fqdn-2"}));
}

TEST(THostListBuilder, ResidingWithCheckerAndAlive) {
    const auto jsonResponse = R"({
    "instances": [
    {
      "id": 1,
      "name": "name-1",
      "state": "EXISTING",
      "fqdn": "fqdn-1",
      "ip": "ip-1"
    },
    {
      "id": 2,
      "name": "name-2",
      "state": "NEW",
      "fqdn": "fqdn-2",
      "ip": "ip-2"
    },
    {
      "id": 3,
      "name": "name-3",
      "state": "RESIDING",
      "fqdn": "fqdn-3",
      "ip": "ip-3"
    }
    ]})";

    THostListBuilder::THosts prevHosts{"fqdn-1", "fqdn-2", "fqdn-3"};
    int callCount = 0;
    auto [localHost, hosts] = Run(MakeResponse(200, jsonResponse), prevHosts, [&callCount](auto host, auto callback) {
        ++callCount;
        EXPECT_STREQ(host, "fqdn-3");
        callback(true);
    });

    EXPECT_EQ(callCount, 1);
    EXPECT_EQ(hosts.size(), 0);
}
