#include <mail/ymod_http_watcher/handlers/qloud_instances/parser.h>

#include <library/cpp/testing/unittest/gtest.h>

namespace {

yhttp::response MakeResponse(int code, std::string body) {
    return yhttp::response{code, {}, std::move(body), "Reason"};
}

} // namespace

using namespace NYmodHttpWatcher::NQloudInstances;

TEST(TParser, Ok) {
    const auto jsonResponse = R"({
    "instances": [
    {
      "id": 1,
      "name": "name-1",
      "state": "RESIDING",
      "fqdn": "fqdn-1",
      "ip": "ip-1"
    },
    {
      "id": 1,
      "name": "name-1-1",
      "state": "NEW",
      "fqdn": "fqdn-1-1",
      "ip": "ip-1-1"
    },
    {
      "id": 2,
      "name": "name-2",
      "state": "EXISTING",
      "fqdn": "fqdn-2",
      "ip": "ip-2"
    }
    ]})";

    auto [ec, instances] = ParseResponse(MakeResponse(200, jsonResponse));
    EXPECT_TRUE(ec == EC_OK);
    EXPECT_TRUE(instances == (TInstances{
        {1, {EState::Residing, "fqdn-1"}},
        {1, {EState::New, "fqdn-1-1"}},
        {2, {EState::Existing, "fqdn-2"}}}));
}

TEST(TParser, Http500) {
    auto [ec, instances] = ParseResponse(MakeResponse(500, "Internal error"));
    EXPECT_TRUE(ec == EC_SERVER_ERROR);
}

TEST(TParser, Http404) {
    auto [ec, instances] = ParseResponse(MakeResponse(404, "Not found"));
    EXPECT_TRUE(ec == EC_NOT_FOUND);
}

TEST(TParser, InvalidJson) {
    const auto jsonResponse = R"({
    "instances": [
    {
      "id": 1,
      "name": "name-1",
      "state": "EXISTING",
      "fqdn": "fqdn-1",
      "ip": "ip-1"
    },
    {
      "id": 2,
      "name": "name-2",
      "state": "EXISTING",
      "fqdn": "fqdn-2",
      "ip": "ip-2"
    }
    ]}})";

    auto [ec, instances] = ParseResponse(MakeResponse(200, jsonResponse));
    EXPECT_TRUE(ec == EC_PARSE_ERROR);
}

TEST(TParser, UnknownState) {
    const auto jsonResponse = R"({
    "instances": [
    {
      "id": 1,
      "name": "name-1",
      "state": "EXISTING",
      "fqdn": "fqdn-1",
      "ip": "ip-1"
    },
    {
      "id": 2,
      "name": "name-2",
      "state": "TEST",
      "fqdn": "fqdn-2",
      "ip": "ip-2"
    }
    ]})";

    auto [ec, instances] = ParseResponse(MakeResponse(200, jsonResponse));
    EXPECT_TRUE(ec == EC_UNKNOWN_STATE);
}

TEST(TParser, EmptyFqdn) {
    const auto jsonResponse = R"({
    "instances": [
    {
      "id": 1,
      "name": "name-1",
      "state": "EXISTING",
      "fqdn": "fqdn-1",
      "ip": "ip-1"
    },
    {
      "id": 2,
      "name": "name-2",
      "state": "EXISTING",
      "fqdn": "",
      "ip": "ip-2"
    }
    ]})";

    auto [ec, instances] = ParseResponse(MakeResponse(200, jsonResponse));
    EXPECT_TRUE(ec == EC_EMPTY_FQDN);
}

TEST(TParser, InvalidId) {
    const auto jsonResponse = R"({
    "instances": [
    {
      "id": "test",
      "name": "name-1",
      "state": "EXISTING",
      "fqdn": "fqdn-1",
      "ip": "ip-1"
    },
    {
      "id": 2,
      "name": "name-2",
      "state": "EXISTING",
      "fqdn": "fqdn-2",
      "ip": "ip-2"
    }
    ]})";

    auto [ec, instances] = ParseResponse(MakeResponse(200, jsonResponse));
    EXPECT_TRUE(ec == EC_INVALID_ID);
}

TEST(TParser, Reduce) {
    auto reducedInstances = ReduceInstances(TInstances{
        {1, {EState::Residing, "fqdn-1"}},
        {1, {EState::New, "fqdn-1-1"}},
        {2, {EState::Existing, "fqdn-2"}},
        {2, {EState::Residing, "fqdn-2-1"}},
        {3, {EState::Existing, "fqdn-3"}},
        {4, {EState::New, "fqdn-4"}},
        {5, {EState::Residing, "fqdn-5"}}});

    EXPECT_TRUE(reducedInstances == (TReducedInstances{
        {EState::New, "fqdn-1-1"},
        {EState::Existing, "fqdn-2"},
        {EState::Existing, "fqdn-3"},
        {EState::New, "fqdn-4"},
        {EState::Residing, "fqdn-5"}}));
}
